﻿/**
 * Ads Insight Pro - Background Service Worker
 * Handles download management and data processing
 */

import { getApiUrl } from '../lib/config.js';
import { fetchWithTimeout, TIMEOUTS } from '../lib/fetch-with-timeout.js';
import { apiClient } from '../lib/api-client.js';
import { getCountryName } from '../lib/country-codes.js';
import { extractYouTubeFromAdPage } from '../lib/url-extractor.js';
import { filterFieldsByPlan } from '../lib/field-permissions.js';
import { usageMonitor } from '../lib/usage-monitor.js';
import { createLogger, logEnvironmentInfo, DEBUG_CONFIG } from '../lib/debug-config.js';
import { extractTextFromAdData, processAdsWithOCR, processAdsWithOCRAndQuota, terminateOCR } from '../lib/ocr-service.js';
import { permissionsManager } from '../lib/permissions.js';
import { RemoteConfig, initRemoteConfig } from '../lib/remote-config.js';

// ============== LOGGER ==============
const log = createLogger('[AdsInsightPro][SW]', DEBUG_CONFIG.serviceWorker);

// ============== API CONFIG ==============
const API_URL = 'https://adstransparency.google.com/anji/_/rpc/SearchService/SearchCreatives';
const API_HEADERS = {
  'Content-Type': 'application/x-www-form-urlencoded',
  'Origin': 'https://adstransparency.google.com',
  'Referer': 'https://adstransparency.google.com/',
  'X-Same-Domain': '1',
  'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36'
};

// ============== STATE ==============
let downloadState = {
  isActive: false,
  isPaused: false,
  currentPage: 0,
  totalPages: 0,
  cursor: null,
  ads: [],
  config: null,
  tabId: null,
  videoProcessing: {
    isActive: false,
    processed: 0,
    total: 0,
    found: 0
  },
  // v1.1.0 Quota reservation
  quotaReservation: {
    id: null,
    count: 0,
    expiresAt: null
  }
};

const PROGRESS_STORAGE_KEY = 'download_progress';
// v1.2.0: Rate limits now come from remote config (with local defaults)
// const RATE_LIMIT = { normal: 2000, cautious: 5000, recovery: 30000 }; // Legacy - now from RemoteConfig

// ============== MESSAGE HANDLER ==============
chrome.runtime.onMessage.addListener(async (message, _sender, sendResponse) => {
  log.debug('Received message:', message.type);
  
  switch (message.type) {
    case 'START_DOWNLOAD':
      await startDownload(message.config);
      break;
    case 'GET_PROGRESS':
      sendResponse({
        isActive: downloadState.isActive,
        currentPage: downloadState.currentPage,
        totalPages: downloadState.totalPages,
        adsCount: downloadState.ads.length,
        videoProcessing: downloadState.videoProcessing,
        config: downloadState.config,
        timestamp: Date.now()
      });
      break;
    case 'PAUSE_DOWNLOAD':
      downloadState.isPaused = true;
      log.info('Download paused');
      break;
    case 'RESUME_DOWNLOAD':
      downloadState.isPaused = false;
      log.info('Download resumed');
      break;
    case 'CANCEL_DOWNLOAD':
      cancelDownload();
      break;
    case 'LINK_INSTALL_TO_USER':
      if (message.userId) linkInstallToUser(message.userId);
      break;
    case 'RESTORE_USAGE_FROM_SERVER':
      try {
        const restored = await usageMonitor.restoreFromServer();
        if (restored) {
          log.success('✅ Usage stats restored from server after login');
        }
      } catch (err) {
        log.error('Failed to restore usage from server:', err);
      }
      break;
    case 'CLEAR_PERMISSIONS_CACHE':
      // 清除 PermissionsManager 的内存缓存（登录/登出时调用）
      (async () => {
        try {
          await permissionsManager.clearCache();
          log.info('✅ Permissions cache cleared (memory + storage)');
          sendResponse({ success: true });
        } catch (err) {
          log.error('Failed to clear permissions cache:', err);
          sendResponse({ success: false, error: err.message });
        }
      })();
      return true;
    case 'GET_DOWNLOAD_STATUS':
      sendResponse({
        isActive: downloadState.isActive,
        isPaused: downloadState.isPaused,
        currentPage: downloadState.currentPage,
        totalPages: downloadState.totalPages,
        adsCount: downloadState.ads.length
      });
      break;
    case 'GET_USAGE_STATS':
      (async () => {
        try {
          const stats = usageMonitor.getStats();
          sendResponse({ success: true, stats });
        } catch (e) {
          sendResponse({ success: false, error: e.message });
        }
      })();
      return true;
    case 'GET_USAGE_REPORT':
      (async () => {
        try {
          const report = usageMonitor.getDetailedReport();
          sendResponse({ success: true, report });
        } catch (e) {
          sendResponse({ success: false, error: e.message });
        }
      })();
      return true;
    case 'RESET_USAGE_STATS':
      (async () => {
        try {
          await usageMonitor.reset();
          sendResponse({ success: true });
        } catch (e) {
          sendResponse({ success: false, error: e.message });
        }
      })();
      return true;
    
    // OCR Offscreen 日志转发
    case 'OCR_LOG': {
      const logLevel = message.level || 'INFO';
      console.log(`[OCR Offscreen][${logLevel}] ${message.message}`);
      break;
    }
    
    // OCR 消息 - 不在这里处理，让 offscreen document 直接响应
    case 'OCR_SINGLE':
    case 'OCR_PROCESS_ADS':
    case 'OCR_SET_CONFIG':
    case 'OCR_TERMINATE':
      // 返回 false 让消息传递给 offscreen document
      return false;
    
    default:
      // 未知消息类型，不阻塞
      return false;
  }
  return true;
});

// ============== DOWNLOAD PROCESS ==============
async function startDownload(config) {
  log.info('Starting download', config);
  
  // v1.1.0: 使用配额预扣模式
  let reservationId = null;
  
  // 🔧 FIX: Use actual ads count when available for accurate quota calculation
  let estimatedAds;
  if (config.adsCount > 0) {
    // Use actual ads count (more accurate)
    const totalPagesForQuota = config.searchMode === 'brand' 
      ? config.pages * (config.advertiserIds?.length || 1)
      : config.pages;
    estimatedAds = Math.min(config.adsCount, totalPagesForQuota * 100);
  } else {
    // Fallback to estimation when actual count unknown
    const totalPagesForQuota = config.searchMode === 'brand' 
      ? config.pages * (config.advertiserIds?.length || 1)
      : config.pages;
    estimatedAds = totalPagesForQuota * 100;
  }
  
  try {
    
    log.info(`Quota check: mode=${config.searchMode}, pages=${config.pages}, accounts=${config.advertiserIds?.length || 1}, estimatedAds=${estimatedAds}`);
    
    // v1.1.0: 使用新的权限 API 和配额预扣
    const permsResult = await apiClient.getPermissions();
    if (permsResult.success) {
      const perms = permsResult.data;
      log.info(`Permissions: plan=${perms.plan}, remaining=${perms.quota.remaining}, perRequestLimit=${perms.quota.per_request_limit}`);
      
      // 检查每日配额
      if (estimatedAds > perms.quota.remaining) {
        notifyPopup('DOWNLOAD_ERROR', { 
          title: 'Insufficient Quota', 
          message: `You need ~${estimatedAds} ads but only have ${perms.quota.remaining} remaining today.\nUpgrade your plan or wait until tomorrow.` 
        });
        return;
      }
      
      // 检查单次限制（只对非品牌模式检查，品牌模式分批处理）
      if (config.searchMode !== 'brand' && estimatedAds > perms.quota.per_request_limit) {
        notifyPopup('DOWNLOAD_ERROR', { 
          title: 'Export Limit Exceeded', 
          message: `Maximum ${perms.quota.per_request_limit} ads per export for ${perms.plan} plan.` 
        });
        return;
      }
      
      // 检查视频提取权限
      if (config.fetchVideoLinks && !perms.features.video_extraction) {
        config.fetchVideoLinks = false;
        log.info('Video extraction disabled - not available for plan:', perms.plan);
      }
      
      // 预扣配额
      const reserveResult = await apiClient.reserveQuota(estimatedAds);
      if (reserveResult.success && reserveResult.data.success) {
        reservationId = reserveResult.data.reservation_id;
        log.info(`✅ Quota reserved: ${reservationId}, remaining after: ${reserveResult.data.remaining}`);
      } else {
        const reason = reserveResult.data?.reason || 'unknown';
        const message = reserveResult.data?.message || 'Failed to reserve quota';
        log.error(`❌ Quota reservation failed: ${reason} - ${message}`);
        notifyPopup('DOWNLOAD_ERROR', { title: 'Quota Error', message });
        return;
      }
    } else {
      // 回退到旧的配额检查
      log.warn('New permissions API failed, falling back to legacy validation');
      const check = await apiClient.validateQuota('export', estimatedAds, totalPagesForQuota);
      if (!check.success || !check.data.allowed) {
        notifyPopup('DOWNLOAD_ERROR', { title: 'Quota Exceeded', message: check.data.upgrade_hint || 'Daily limit reached' });
        return;
      }
      
      // 兼容性处理
      if (!check.data.features) {
        check.data.features = { video_extraction: check.data.plan === 'enterprise' };
      }
      if (config.fetchVideoLinks && !check.data.features.video_extraction) {
        config.fetchVideoLinks = false;
      }
    }
  } catch (e) {
    log.warn('Quota check/reserve failed:', e);
    // 网络错误时继续执行，依赖服务端记录
  }
  
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  if (!tab) {
    notifyPopup('DOWNLOAD_ERROR', { title: 'Error', message: 'No active tab found' });
    // v1.1.0: Cancel reservation if tab not found
    if (reservationId) {
      apiClient.cancelQuota(reservationId).catch(() => {});
    }
    return;
  }
  
  downloadState = {
    isActive: true,
    isPaused: false,
    currentPage: 0,
    totalPages: config.pages,  // 注意：这里是单个账号的页数
    cursor: null,
    ads: [],
    config: config,
    tabId: tab.id,
    videoProcessing: { isActive: false, processed: 0, total: 0, found: 0 },
    // v1.1.0 Quota reservation
    quotaReservation: {
      id: reservationId,
      count: estimatedAds,
      expiresAt: Date.now() + 5 * 60 * 1000
    }
  };
  
  try {
    // 🆕 新增：单个广告模式分支
    if (config.searchMode === 'single_ad') {
      await handleSingleAdDownload(config);
      return;
    }
    
    // 🆕 新增：品牌模式分支
    if (config.searchMode === 'brand') {
      await handleBrandDownload(config);
      return;
    }
    
    // 现有代码：domain/advertiser 模式
    await fetchAllPages();
  } catch (error) {
    log.error('Download failed:', error);
    notifyPopup('DOWNLOAD_ERROR', { title: 'Download Failed', message: error.message });
  }
}

/**
 * 🆕 处理单个广告详情页下载
 * 直接使用 SearchCreatives API 获取单个广告的详细信息
 * 
 * @param {Object} config - 下载配置
 */
async function handleSingleAdDownload(config) {
  log.info(`🎯 Starting single ad export: advertiser=${config.advertiserId}, creative=${config.creativeId}`);
  
  try {
    // 使用 SearchCreatives API 获取单个广告数据
    // 构建只获取一个广告的请求
    const requestBody = {
      "2": 1,  // 只请求1个
      "3": {
        "1": config.advertiserId  // 指定广告商
      },
      "7": { "1": 1, "2": 0, "3": 2344 }
    };
    
    const formData = `f.req=${encodeURIComponent(JSON.stringify(requestBody))}`;
    const referer = `https://adstransparency.google.com/advertiser/${config.advertiserId}/creative/${config.creativeId}?region=${config.region || 'anywhere'}`;
    
    const response = await fetchWithTimeout(API_URL, {
      method: 'POST',
      headers: { ...API_HEADERS, 'Referer': referer },
      body: formData,
      credentials: 'omit'
    }, TIMEOUTS.LONG);
    
    if (!response.ok) {
      throw new Error(`HTTP ${response.status}`);
    }
    
    const text = await response.text();
    const result = parseResponse(text);
    
    if (result.error) {
      throw new Error(result.error);
    }
    
    // 找到目标广告（匹配 creative_id）
    let targetAd = null;
    if (result.ads && result.ads.length > 0) {
      // 尝试找到精确匹配的广告
      targetAd = result.ads.find(ad => ad.creative_id === config.creativeId);
      
      // 如果没找到精确匹配，可能需要再请求一页
      if (!targetAd && result.cursor) {
        log.debug('Target ad not in first batch, fetching more...');
        // 最多再请求5页来找目标广告
        let cursor = result.cursor;
        for (let i = 0; i < 5 && !targetAd && cursor; i++) {
          const moreBody = { ...requestBody, "2": 100, "4": cursor };
          const moreFormData = `f.req=${encodeURIComponent(JSON.stringify(moreBody))}`;
          
          await sleep(1000);
          const moreResponse = await fetchWithTimeout(API_URL, {
            method: 'POST',
            headers: { ...API_HEADERS, 'Referer': referer },
            body: moreFormData,
            credentials: 'omit'
          }, TIMEOUTS.LONG);
          
          if (moreResponse.ok) {
            const moreText = await moreResponse.text();
            const moreResult = parseResponse(moreText);
            if (moreResult.ads) {
              targetAd = moreResult.ads.find(ad => ad.creative_id === config.creativeId);
              cursor = moreResult.cursor;
            }
          }
        }
      }
      
      // 如果还是没找到，使用第一个广告作为代替
      if (!targetAd) {
        log.warn(`Target creative ${config.creativeId} not found in results, using first ad`);
        targetAd = result.ads[0];
      }
    }
    
    if (!targetAd) {
      throw new Error('Could not find the ad data');
    }
    
    log.success(`✅ Single ad found: ${targetAd.creative_id}`);
    
    // 设置下载状态
    downloadState.ads = [targetAd];
    downloadState.currentPage = 1;
    downloadState.totalPages = 1;
    
    // 完成导出
    await completeDownload();
    
  } catch (error) {
    log.error('Single ad download failed:', error);
    notifyPopup('DOWNLOAD_ERROR', { title: 'Download Failed', message: error.message });
    
    // 取消配额预扣
    const reservationId = downloadState.quotaReservation?.id;
    if (reservationId) {
      apiClient.cancelQuota(reservationId).catch(() => {});
    }
    downloadState.isActive = false;
  }
}

/**
 * 🆕 处理品牌聚合导出
 * 策略：循环处理每个 advertiser，最后统一合并和导出
 * 
 * @param {Object} config - 下载配置
 */
async function handleBrandDownload(config) {
  const allAds = [];
  const failedIds = [];
  
  // 🔧 关键修正：临时禁用视频提取，避免重复和进度混乱
  const originalFetchVideoLinks = config.fetchVideoLinks;
  const tempConfig = { ...config, fetchVideoLinks: false };
  
  log.info(`🎯 Starting brand export: ${config.advertiserIds.length} accounts, ${config.pages} pages each`);
  
  for (let i = 0; i < config.advertiserIds.length; i++) {
    const advertiserId = config.advertiserIds[i];
    
    // 通知进度到 popup
    notifyPopup('BRAND_PROGRESS', {
      current: i + 1,
      total: config.advertiserIds.length,
      advertiserId: advertiserId,
      collectedAds: allAds.length
    });
    
    try {
      log.info(`📥 Fetching advertiser ${i+1}/${config.advertiserIds.length}: ${advertiserId}`);
      
      // 重置状态，准备抓取单个广告商
      downloadState.currentPage = 0;
      downloadState.cursor = null;
      downloadState.ads = [];
      downloadState.totalPages = config.pages;  // 每个广告商的页数
      downloadState.config = {
        ...tempConfig,  // 使用禁用视频的配置
        searchMode: 'advertiser',  // 临时切换为 advertiser 模式
        advertiserId: advertiserId,
        _inBrandLoop: true  // 🔧 标记：正在品牌导出循环中，不要自动完成
      };
      
      // 复用现有的 fetchAllPages() 逻辑
      // 注意：这里不会触发 completeDownload()，因为设置了 _inBrandLoop 标志
      await fetchAllPages();
      
      // 收集结果
      const adsFromThisAccount = downloadState.ads.length;
      allAds.push(...downloadState.ads);
      
      log.success(`✅ Advertiser ${advertiserId}: ${adsFromThisAccount} ads collected (total: ${allAds.length})`);
      
    } catch (error) {
      log.error(`❌ Advertiser ${advertiserId} failed:`, error.message);
      failedIds.push({ 
        advertiserId, 
        error: error.message 
      });
      // 继续处理下一个（不中断整个流程）
    }
    
    // 延迟避免速率限制（固定 2 秒，MVP 版本）
    if (i < config.advertiserIds.length - 1) {
      log.debug(`⏳ Waiting 2s before next advertiser...`);
      await sleep(2000);
    }
  }
  
  // 合并导出
  log.info(`🎉 Brand export complete: ${allAds.length} total ads from ${config.advertiserIds.length - failedIds.length}/${config.advertiserIds.length} accounts`);
  
  // 🔧 关键修正：恢复原始配置，准备统一导出
  downloadState.config = {
    ...config,
    searchMode: 'brand',  // 确保模式正确
    fetchVideoLinks: originalFetchVideoLinks  // 恢复视频提取设置
  };
  downloadState.ads = allAds;  // 设置合并后的数据
  downloadState.isActive = true;  // 确保状态正确
  
  // 调用现有的导出逻辑（会处理视频提取、字段过滤、文件生成等）
  await completeDownload();
  
  // 显示警告（如果有失败）
  if (failedIds.length > 0) {
    notifyPopup('BRAND_EXPORT_WARNING', {
      failedCount: failedIds.length,
      failedIds: failedIds.map(f => f.advertiserId),
      successCount: config.advertiserIds.length - failedIds.length,
      totalAds: allAds.length,
      message: `Successfully exported ${allAds.length} ads from ${config.advertiserIds.length - failedIds.length} accounts. ${failedIds.length} accounts failed.`
    });
  }
}

async function fetchAllPages() {
  while (downloadState.currentPage < downloadState.totalPages && downloadState.isActive) {
    while (downloadState.isPaused && downloadState.isActive) await sleep(500);
    if (!downloadState.isActive) break;
    
    const pageNum = downloadState.currentPage + 1;
    log.info(`Fetching page ${pageNum}`);
    
    try {
      const result = await fetchPage();
      if (result.error) throw new Error(result.error);
      
      if (result.ads && result.ads.length > 0) {
        downloadState.ads.push(...result.ads);
        downloadState.cursor = result.cursor;
      }
      
      downloadState.currentPage++;
      notifyPopup('DOWNLOAD_PROGRESS', {
        currentPage: downloadState.currentPage,
        totalPages: downloadState.totalPages,
        adsCount: downloadState.ads.length,
        cursor: downloadState.cursor
      });
      await saveProgress();
      
      if (!result.cursor || result.ads.length === 0) break;
      await sleep(getDelay());
      
    } catch (error) {
      if (error.message.includes('429')) {
        await sleep(getRecoveryDelay());
        continue;
      }
      throw error;
    }
  }
  
  // 🔧 修正：品牌模式循环中不自动调用 completeDownload（由 handleBrandDownload 统一处理）
  if (downloadState.isActive && !downloadState.config._inBrandLoop) {
    await completeDownload();
  }
}

async function fetchPage() {
  const config = downloadState.config;
  const requestBody = buildRequestBody(config);
  const formData = `f.req=${encodeURIComponent(JSON.stringify(requestBody))}`;
  let referer = `https://adstransparency.google.com/?region=${config.region || 'anywhere'}`;
  
  try {
    const response = await fetchWithTimeout(API_URL, {
      method: 'POST',
      headers: { ...API_HEADERS, 'Referer': referer },
      body: formData,
      credentials: 'omit'
    }, TIMEOUTS.LONG);
    
    if (!response.ok) return { error: `HTTP ${response.status}` };
    return parseResponse(await response.text());
  } catch (error) {
    return { error: error.message };
  }
}

function buildRequestBody(config) {
  const body = { "2": 100, "3": {}, "7": { "1": 1, "2": 0, "3": 2344 } };
  if (config.searchMode === 'domain' && config.domain) {
    body["3"]["12"] = { "1": config.domain, "2": true };
  } else if ((config.searchMode === 'advertiser' || config.searchMode === 'brand') && config.advertiserId) {
    // 🔧 CRITICAL FIX: Brand mode must also filter by advertiserId
    // Otherwise it returns ALL ads from ALL advertisers
    body["3"]["1"] = config.advertiserId;
  }
  if (downloadState.cursor) body["4"] = downloadState.cursor;
  return body;
}

function parseResponse(text) {
  try {
    let cleanText = text;
    if (text.startsWith(")]}'")) cleanText = text.substring(text.indexOf('\n') + 1);
    
    const outerData = JSON.parse(cleanText);
    let innerData = (Array.isArray(outerData) && outerData[0] && outerData[0][2] && typeof outerData[0][2] === 'string') 
      ? JSON.parse(outerData[0][2]) : outerData;
      
    const rawAds = innerData?.["1"] || innerData?.[1] || [];
    const ads = [];
    
    if (Array.isArray(rawAds)) {
      for (const item of rawAds) {
        const ad = parseAdItem(item);
        if (ad) ads.push(ad);
      }
    }
    
    return { ads, cursor: innerData?.["2"] || innerData?.[2] || null };
  } catch (e) {
    return { ads: [], cursor: null, error: e.message };
  }
}

function parseAdItem(item) {
  try {
    const formatCode = item["4"] || item[4];
    
    // Debug: Log raw advertiser_name field
    const rawAdvertiserName = item["12"] || item[12];
    if (rawAdvertiserName && typeof rawAdvertiserName !== 'string') {
      log.warn('advertiser_name is not a string:', typeof rawAdvertiserName, JSON.stringify(rawAdvertiserName).substring(0, 100));
    }
    
    const ad = {
      advertiser_id: item["1"] || item[1] || '',
      creative_id: item["2"] || item[2] || '',
      format: formatCode,
      format_name: getFormatName(formatCode),
      advertiser_name: typeof rawAdvertiserName === 'string' ? rawAdvertiserName : '',
      target_domain: item["14"] || item[14] || '',
      ad_type: formatCode === 3 ? 'Video' : 'Image',
      days_shown: item["13"] || item[13] || 0,
      transparency_url: `https://adstransparency.google.com/advertiser/${item["1"] || item[1]}/creative/${item["2"] || item[2]}`,
      start_date: '',
      end_date: '',
      first_shown_date: '',
      last_shown_date: '',
      image_url: '',
      preview_url: '',
      youtube_video_id: '',
      video_thumbnail_url: '',
      // v1.1.0 OCR fields
      ad_domain: '',
      ad_title: '',
      ad_description: '',
      confidence: 0
    };
    
    // Extract start date (field "6")
    const startDate = item["6"] || item[6];
    if (startDate && (startDate["1"] || startDate[1])) {
      const dateStr = formatDate(parseInt(startDate["1"] || startDate[1]));
      ad.start_date = dateStr;
      ad.first_shown_date = dateStr; // Same as start_date
    }
    
    // Extract end date (field "7")
    const endDate = item["7"] || item[7];
    if (endDate && (endDate["1"] || endDate[1])) {
      const dateStr = formatDate(parseInt(endDate["1"] || endDate[1]));
      ad.end_date = dateStr;
      ad.last_shown_date = dateStr; // Same as end_date
    }
    
    // Extract creative
    const creative = item["3"] || item[3];
    if (creative) {
      if (creative["1"]?.["4"] || creative[1]?.[4]) ad.preview_url = creative["1"]?.["4"] || creative[1]?.[4];
      const imgMatch = (creative["3"]?.["2"] || creative[3]?.[2])?.match(/src="([^"]+)"/);
      if (imgMatch) ad.image_url = imgMatch[1];
    }
    
    // v1.1.0: Extract text content for OCR fields
    try {
      const textData = extractTextFromAdData(ad, item);
      if (textData) {
        ad.ad_domain = textData.ad_domain || '';
        ad.ad_title = textData.ad_title || '';
        ad.ad_description = textData.ad_description || '';
        ad.confidence = textData.confidence || 0;
      }
    } catch (ocrErr) {
      // OCR extraction failed, continue with empty fields
    }
    
    return ad;
  } catch (e) { return null; }
}

function getFormatName(code) { return code === 1 ? 'TEXT' : code === 2 ? 'IMAGE' : code === 3 ? 'VIDEO' : 'UNKNOWN'; }
function formatDate(ts) { return new Date(ts * 1000).toISOString().split('T')[0]; }

// ============== USAGE RECORDING ==============

/**
 * 记录使用量到服务器（服务器是唯一真实数据源）
 * @param {number} adsCount - 下载的广告数量
 * @param {string} domain - 域名或广告商ID
 * @param {number} pages - 下载的页数
 * @returns {Promise<{success: boolean, warnings: Array}>}
 */
async function recordUsageToServer(adsCount, domain, pages) {
  try {
    const authData = await chrome.storage.local.get('authToken');
    if (!authData.authToken) {
      log.warn('No auth token, skipping server usage recording');
      return { success: false, warnings: [] };
    }
    
    const BACKEND_API = getApiUrl();
    const response = await fetch(`${BACKEND_API}/api/usage/record`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${authData.authToken}`
      },
      body: JSON.stringify({
        action: 'download',
        domain: domain,
        pages: pages,
        adsCount: adsCount
      })
    });
    
    if (response.ok) {
      const result = await response.json();
      log.success(`✅ Usage recorded to server: ${adsCount} ads, ${result.data?.todayUsed || '?'}/${result.data?.todayRemaining || '?'} used`);
      return { success: true, warnings: [] };
    } else if (response.status === 400) {
      const error = await response.json();
      if (error.error === 'QUOTA_EXCEEDED') {
        log.error('❌ Quota exceeded:', error.message);
        return {
          success: false,
          warnings: [{
            level: 'danger',
            type: 'quota',
            title: '🔴 Daily Limit Reached',
            message: error.message || 'You have reached your daily download limit.',
            suggestions: [
              'Your download limit resets at midnight UTC',
              'Upgrade your plan for higher limits',
              'Contact support if you need assistance'
            ]
          }]
        };
      }
    }
    
    log.warn('Server usage recording failed:', response.status);
    return { success: false, warnings: [] };
    
  } catch (error) {
    log.error('Server usage recording error:', error);
    return { success: false, warnings: [] };
  }
}

// ============== EXPORT ==============
async function completeDownload() {
  log.info('Processing and exporting data');
  const config = downloadState.config;
  let ads = [...downloadState.ads];
  
  try {
    const regionName = getCountryName(config.region);
    ads = ads.map(ad => ({ ...ad, query_region: regionName }));
    
    // v1.1.0: 记录是否为单个广告模式（稍后过滤字段）
    const isSingleAdMode = config.searchMode === 'single_ad';
    
    notifyPopup('PROCESSING_START', { total: ads.length });
    
    // 🔧 FIX: 在过滤字段之前保存 image_url 以便 OCR 使用
    // Free 用户需要 image_url 来执行 OCR，但导出时不包含此字段
    const imageUrlsForOCR = new Map();
    if (config.includeAdContent) {
      ads.forEach(ad => {
        if (ad.image_url) {
          imageUrlsForOCR.set(ad.creative_id, ad.image_url);
        }
      });
      log.info(`💾 Saved ${imageUrlsForOCR.size} image URLs for OCR processing (before field filtering)`);
    }
    
    // Step 1: 调用服务器处理数据（字段过滤）
    let processedAds = ads;
    let videoExtractionAllowed = false;
    
    try {
      const processResult = await apiClient.processAds(ads, {
        extractVideo: false,  // 先不在这里提取，后面单独处理
        format: config.exportFormat
      });
      
      if (processResult.success) {
        processedAds = processResult.data.ads;
        videoExtractionAllowed = processResult.data.meta?.video_extraction_allowed || false;
        
        // 记录后端返回的计划信息
        const backendMeta = processResult.data.meta || {};
        log.info(`📋 Backend response: plan="${backendMeta.plan}", video_extraction_allowed=${backendMeta.video_extraction_allowed}, filtered_fields=${backendMeta.filtered_fields}`);
      } else {
        log.warn('Backend processing failed, using local field filtering');
        // 备份方案：使用本地字段过滤
        const { userPlan } = await chrome.storage.local.get('userPlan');
        const plan = userPlan || 'free';
        processedAds = filterFieldsByPlan(ads, plan);
        videoExtractionAllowed = (plan === 'enterprise');
      }
    } catch (apiError) {
      log.error('API call failed, falling back to local filtering:', apiError);
      // 备份方案：使用本地字段过滤
      const { userPlan } = await chrome.storage.local.get('userPlan');
      const plan = userPlan || 'free';
      processedAds = filterFieldsByPlan(ads, plan);
      videoExtractionAllowed = (plan === 'enterprise');
    }
    
    // Step 2: 应用格式筛选（如果用户选择了特定格式）
    if (config.format && config.format !== '') {
      const formatMap = { '1': 'TEXT', '2': 'IMAGE', '3': 'VIDEO' };
      const targetFormat = formatMap[config.format];
      const before = processedAds.length;
      processedAds = processedAds.filter(ad => ad.format_name === targetFormat);
      log.info(`🔍 Format filter (${targetFormat}): ${before} → ${processedAds.length}`);
    }
    
    // Step 3: 如果需要提取视频且用户有权限（Enterprise）
    log.info(`🎥 Video extraction check: fetchVideoLinks=${config.fetchVideoLinks}, videoExtractionAllowed=${videoExtractionAllowed}, processedAds=${processedAds.length}`);
    
    if (config.fetchVideoLinks && videoExtractionAllowed) {
      // 筛选出需要提取视频的广告
      const videoAds = processedAds.filter(ad => 
        ad.format_name === 'VIDEO' && 
        (ad.advertiser_id || ad.creative_id) &&
        !ad.youtube_video_id
      );
      
      log.info('🎥 Video ads filtered:', {
        'total ads': processedAds.length,
        'video ads': videoAds.length,
        'sample ad': videoAds[0] ? {
          format_name: videoAds[0].format_name,
          advertiser_id: videoAds[0].advertiser_id,
          creative_id: videoAds[0].creative_id,
          youtube_video_id: videoAds[0].youtube_video_id
        } : 'none'
      });
      
      if (videoAds.length > 0) {
        log.info(`✅ Extracting videos for ${videoAds.length} ads`);
        notifyPopup('VIDEO_PROCESSING_START', { total: videoAds.length });
        
        try {
          // 提取视频信息
          const videoResults = await extractVideoInfo(videoAds);
          
          // 合并视频信息到处理后的数据
          const videoMap = new Map(videoResults.map(v => [v.creative_id, v]));
          processedAds = processedAds.map(ad => {
            const videoInfo = videoMap.get(ad.creative_id);
            if (videoInfo) {
              return { ...ad, ...videoInfo };
            }
            return ad;
          });
          
          const extracted = videoResults.filter(v => v.youtube_video_id).length;
          notifyPopup('VIDEO_PROCESSING_COMPLETE', { 
            processed: videoAds.length, 
            found: extracted 
          });
        } catch (videoError) {
          log.error('Video extraction failed:', videoError);
          // 继续导出，即使视频提取失败
        }
      } else {
        log.warn('⚠️ No video ads found to extract');
      }
    } else {
      const reason = !config.fetchVideoLinks ? 'fetchVideoLinks is false' : 
                     !videoExtractionAllowed ? 'videoExtractionAllowed is false' : 
                     'unknown';
      log.warn(`❌ Video extraction skipped: reason="${reason}", fetchVideoLinks=${config.fetchVideoLinks}, videoExtractionAllowed=${videoExtractionAllowed}`);
    }
    
    // Step 4: v1.1.0 OCR 处理 - 对有图片的广告执行文字识别（带配额管理）
    // 检查用户是否明确勾选了“包含广告内容”选项
    const includeOCR = config.includeAdContent === true; // 只有用户勾选了才处理 OCR
    
    if (includeOCR) {
      // 🔧 FIX: 使用保存的 image_url（Free 用户在字段过滤后 ad.image_url 可能为空）
      // 注意：TEXT 和 IMAGE 类型都是图片格式，都需要 OCR
      const imageAdsForOCR = processedAds.filter(ad => {
        const imageUrl = ad.image_url || imageUrlsForOCR.get(ad.creative_id);
        return imageUrl && 
               imageUrl.startsWith('http') &&
               !ad.ad_title; // 还没有提取到文本
      });
      
      // 恢复 image_url 以便 OCR 处理（临时添加，导出前会再次过滤）
      imageAdsForOCR.forEach(ad => {
        if (!ad.image_url && imageUrlsForOCR.has(ad.creative_id)) {
          ad._temp_image_url = imageUrlsForOCR.get(ad.creative_id);
        }
      });
      
      if (imageAdsForOCR.length > 0) {
        // v1.1.0: 检查 OCR 配额
        const ocrQuotaCheck = await permissionsManager.checkOCRQuota(imageAdsForOCR.length);
        log.info(`📝 OCR quota check: allowed=${ocrQuotaCheck.allowed}, effective_remaining=${ocrQuotaCheck.effective_remaining}, needed=${imageAdsForOCR.length}`);
        
        if (ocrQuotaCheck.effective_remaining > 0) {
          const maxOCR = ocrQuotaCheck.effective_remaining;
          log.info(`📝 Starting OCR for ${Math.min(imageAdsForOCR.length, maxOCR)} of ${imageAdsForOCR.length} image ads`);
          notifyPopup('OCR_PROCESSING_START', { 
            total: Math.min(imageAdsForOCR.length, maxOCR),
            quota_limited: imageAdsForOCR.length > maxOCR
          });
          
          try {
            // v1.1.0: 使用带配额管理的 OCR 处理
            const { ads: ocrProcessedAds, ocrStats } = await processAdsWithOCRAndQuota(
              processedAds, 
              { maxOCR }, 
              (progress) => {
                notifyPopup('OCR_PROCESSING_PROGRESS', {
                  processed: progress.processed,
                  total: progress.total,
                  successful: progress.successful,
                  percentage: progress.percentage
                });
              }
            );
            
            processedAds = ocrProcessedAds;
            
            log.success(`📝 OCR completed: ${ocrStats.successful}/${ocrStats.processed} successful, ${ocrStats.skipped} skipped, ${ocrStats.quota_exceeded} quota exceeded`);
            notifyPopup('OCR_PROCESSING_COMPLETE', { 
              processed: ocrStats.processed, 
              successful: ocrStats.successful,
              skipped: ocrStats.skipped,
              quota_exceeded: ocrStats.quota_exceeded
            });
            
            // 释放 OCR worker 资源
            await terminateOCR();
          } catch (ocrError) {
            log.error('OCR processing failed:', ocrError);
            // 继续导出，即使 OCR 失败
          }
        } else {
          log.warn('📝 OCR quota exhausted, skipping OCR processing');
          notifyPopup('OCR_QUOTA_EXCEEDED', {
            message: ocrQuotaCheck.message || 'OCR quota exhausted',
            blocked_by: ocrQuotaCheck.blocked_by
          });
        }
      } else {
        log.info('📝 No image ads need OCR processing');
      }
    } else {
      log.info('📝 OCR processing disabled by user config');
    }
    
    // Step 4.5: v1.1.0 单个广告模式 - 移除不需要的字段（在所有处理完成后）
    if (isSingleAdMode) {
      log.info('🧪 Single ad mode - removing unnecessary fields before export');
      const fieldsToRemove = ['target_domain', 'preview_url', 'ocr_status', 'confidence'];
      processedAds = processedAds.map(ad => {
        const cleaned = { ...ad };
        fieldsToRemove.forEach(field => delete cleaned[field]);
        return cleaned;
      });
      log.info('✅ Removed fields:', fieldsToRemove);
    }
    
    // 🔧 FIX: 清理临时 OCR 字段
    processedAds = processedAds.map(ad => {
      const cleaned = { ...ad };
      delete cleaned._temp_image_url;
      return cleaned;
    });
    
    // Step 5: 导出文件
    const filename = generateFilename(config);
    
    if (config.exportFormat === 'csv') await exportCSV(processedAds, filename);
    else if (config.exportFormat === 'json') await exportJSON(processedAds, filename);
    else await exportExcel(processedAds, filename);
    
    // v1.1.0: 确认配额消耗（使用预扣-确认模式）
    const reservationId = downloadState.quotaReservation?.id;
    if (reservationId) {
      try {
        const confirmResult = await apiClient.confirmQuota(reservationId, processedAds.length);
        if (confirmResult.success) {
          log.success(`✅ Quota confirmed: ${processedAds.length} ads, remaining: ${confirmResult.data?.remaining}`);
        } else {
          log.warn('⚠️ Quota confirmation failed:', confirmResult.error);
        }
      } catch (e) {
        log.warn('⚠️ Quota confirmation error:', e.message);
        // Non-critical - the export already succeeded
      }
      // 本地统计（配额已通过服务端记录）
      await usageMonitor.recordDownload(processedAds.length, config.domain || config.advertiserId);
    } else {
      // Fallback: 使用旧的 recordUsageToServer
      const usageRecorded = await recordUsageToServer(processedAds.length, config.domain || config.advertiserId, config.pages);
      if (usageRecorded.success) {
        await usageMonitor.recordDownload(processedAds.length, config.domain || config.advertiserId);
        usageMonitor.syncToServer().catch(() => {});
      }
    }
    
    const warnings = [];
    await clearProgress();
    
    notifyPopup('DOWNLOAD_COMPLETE', {
      stats: { totalAds: processedAds.length, fields: Object.keys(processedAds[0] || {}).length },
      warnings: warnings
    });
    
    await addToDownloadHistory({
      domain: config.domain || config.advertiserId,
      adsCount: processedAds.length,
      format: config.exportFormat,
      filename: filename
    });
    
  } catch (error) {
    log.error('Processing failed:', error);
    notifyPopup('DOWNLOAD_ERROR', { title: 'Processing Failed', message: error.message });
    
    // v1.1.0: 取消配额预扣（导出失败）
    const reservationId = downloadState.quotaReservation?.id;
    if (reservationId) {
      apiClient.cancelQuota(reservationId).catch((e) => {
        log.warn('Failed to cancel quota reservation:', e.message);
      });
    }
  } finally {
    downloadState.isActive = false;
    // Clear reservation
    downloadState.quotaReservation = { id: null, count: 0, expiresAt: null };
  }
}

/**
 * 提取视频信息（移植自本地方案）
 * @param {Array} videoAds - 需要提取视频的广告列表
 * @returns {Promise<Array>} - 提取结果
 */
async function extractVideoInfo(videoAds) {
  const items = [];
  const CONCURRENT = 3;  // 并发处理 3 个视频，提速约 3 倍
  
  // 初始化视频处理状态
  downloadState.videoProcessing = {
    isActive: true,
    processed: 0,
    total: videoAds.length,
    found: 0
  };
  
  let processedCount = 0;
  let foundCount = 0;
  let cacheHits = 0;
  
  const startTime = Date.now();
  log.info(`🎬 Starting video extraction: ${videoAds.length} videos (concurrent=${CONCURRENT}, with cache)`);
  
  // 分批并发处理
  for (let i = 0; i < videoAds.length; i += CONCURRENT) {
    const batch = videoAds.slice(i, i + CONCURRENT);
    
    const batchResults = await Promise.allSettled(
      batch.map(async (ad, batchIndex) => {
        const currentIndex = i + batchIndex + 1;
        log.info(`Processing video ${currentIndex}/${videoAds.length}: ${ad.creative_id}`);
        
        try {
          // 使用移植的本地方案逻辑：直接从 transparency 页面提取
          // 这种方法最可靠，因为它利用了 Google 自己的渲染逻辑
          const results = await extractYouTubeFromAdPage(ad.advertiser_id, ad.creative_id);
          
          if (results && results.length > 0) {
            const youtubeId = results[0].youtubeId;
            log.success(`✅ Extracted video ${currentIndex}/${videoAds.length}: ${youtubeId}`);
            return {
              creative_id: ad.creative_id,
              advertiser_id: ad.advertiser_id,
              youtube_video_id: `https://www.youtube.com/watch?v=${youtubeId}`,
              video_thumbnail_url: `https://img.youtube.com/vi/${youtubeId}/maxresdefault.jpg`
            };
          }
          
          // 如果失败，返回空结果
          log.warn(`❌ Failed to extract video ${currentIndex}/${videoAds.length}: ${ad.creative_id} (no results)`);
          return {
            creative_id: ad.creative_id,
            advertiser_id: ad.advertiser_id
          };
        } catch (e) {
          log.error(`❌ Failed to extract video ${currentIndex}/${videoAds.length}: ${ad.creative_id}`, e);
          return {
            creative_id: ad.creative_id,
            advertiser_id: ad.advertiser_id
          };
        }
      })
    );
    
    // 处理结果并更新计数
    batchResults.forEach((result) => {
      if (result.status === 'fulfilled') {
        items.push(result.value);
        processedCount++;
        if (result.value.youtube_video_id) {
          foundCount++;
        }
      }
    });
    
    // 更新状态和进度
    downloadState.videoProcessing.processed = processedCount;
    downloadState.videoProcessing.found = foundCount;
    
    notifyPopup('VIDEO_PROCESSING_PROGRESS', {
      processed: processedCount,
      total: videoAds.length,
      found: foundCount,
      percentage: Math.round((processedCount / videoAds.length) * 100)
    });
    
    // 批次间延迟
    if (i + CONCURRENT < videoAds.length) {
      await sleep(500);
    }
  }
  
  // 重置视频处理状态
  downloadState.videoProcessing.isActive = false;
  
  // 统计日志
  const elapsed = ((Date.now() - startTime) / 1000).toFixed(1);
  const avgTime = (elapsed / videoAds.length).toFixed(1);
  log.success(`🎬 Video extraction complete: ${foundCount}/${videoAds.length} found in ${elapsed}s (avg ${avgTime}s/video)`);
  
  return items;
}

async function exportCSV(ads, filename) {
  if (ads.length === 0) return;
  
  // v1.2.1: 统一导出字段（删除内部字段）
  const preferredOrder = [
    'advertiser_id', 'advertiser_name', 'creative_id', 'target_domain',
    'format', 'format_name', 'days_shown',
    'first_shown_date', 'last_shown_date',
    'ad_domain', 'ad_title', 'ad_description',
    'image_url', 'preview_url', 'youtube_video_id', 'video_thumbnail_url',
    'transparency_url', 'query_region'
  ];
  
  // 要排除的内部字段
  const excludeFields = ['ad_type', 'confidence', 'ocr_status', 'start_date', 'end_date'];
  
  // 收集所有实际存在的字段（排除内部字段）
  const allKeys = new Set();
  ads.forEach(ad => Object.keys(ad).forEach(k => {
    if (!excludeFields.includes(k)) {
      allKeys.add(k);
    }
  }));
  
  // 按照偏好顺序排列，只包含实际存在的字段
  const orderedKeys = preferredOrder.filter(k => allKeys.has(k));
  const extraKeys = Array.from(allKeys).filter(k => !preferredOrder.includes(k));
  const headers = [...orderedKeys, ...extraKeys];
  
  // v1.2.1: Add UTF-8 BOM for Excel compatibility with CJK characters
  // BOM: \uFEFF tells Excel to interpret the file as UTF-8
  let csv = '\uFEFF';
  csv += headers.join(',') + '\n';
  
  for (const ad of ads) {
    csv += headers.map(h => {
      const val = ad[h] ?? '';
      let strVal = String(val);
      
      // v1.2.1: Clean control characters that can break CSV parsing
      strVal = strVal
        .replace(/\r\n/g, ' ')  // Windows newlines -> space
        .replace(/\r/g, ' ')    // CR -> space  
        .replace(/\n/g, ' ')    // LF -> space
        .replace(/\t/g, ' ');   // TAB -> space
      
      // Escape if contains comma, quote, or newline
      if (strVal.includes(',') || strVal.includes('"')) {
        return `"${strVal.replace(/"/g, '""')}"`;
      }
      return strVal;
    }).join(',') + '\n';
  }
  await downloadFile(csv, filename + '.csv', 'text/csv;charset=utf-8');
}

async function exportJSON(ads, filename) {
  // v1.2.1: 统一导出字段（删除内部字段）
  const excludeFields = ['ad_type', 'confidence', 'ocr_status', 'start_date', 'end_date'];
  
  const filteredAds = ads.map(ad => {
    const filtered = {};
    for (const [key, value] of Object.entries(ad)) {
      if (!excludeFields.includes(key)) {
        filtered[key] = value;
      }
    }
    return filtered;
  });
  
  await downloadFile(JSON.stringify(filteredAds, null, 2), filename + '.json', 'application/json');
}

async function exportExcel(ads, filename) {
  // Generate Excel XML format (compatible with Excel without external libraries)
  if (ads.length === 0) {
    log.warn('No ads to export');
    return;
  }
  
  // v1.2.1: 统一导出字段（删除内部字段）
  const preferredOrder = [
    'advertiser_id',
    'advertiser_name',
    'creative_id',
    'target_domain',
    'format',
    'format_name',
    'days_shown',
    'first_shown_date',
    'last_shown_date',
    'ad_domain',
    'ad_title',
    'ad_description',
    'image_url',
    'preview_url',
    'youtube_video_id',
    'video_thumbnail_url',
    'transparency_url',
    'query_region'
  ];
  
  // 要排除的内部字段
  const excludeFields = ['ad_type', 'confidence', 'ocr_status', 'start_date', 'end_date'];
  
  const availableFields = Object.keys(ads[0]).filter(f => !excludeFields.includes(f));
  const headers = preferredOrder.filter(h => availableFields.includes(h));
  
  let xml = '<?xml version="1.0" encoding="UTF-8"?>\n';
  xml += '<?mso-application progid="Excel.Sheet"?>\n';
  xml += '<Workbook xmlns="urn:schemas-microsoft-com:office:spreadsheet"\n';
  xml += '  xmlns:ss="urn:schemas-microsoft-com:office:spreadsheet">\n';
  xml += '  <Styles>\n';
  xml += '    <Style ss:ID="Header"><Font ss:Bold="1"/><Interior ss:Color="#4285F4" ss:Pattern="Solid"/></Style>\n';
  xml += '  </Styles>\n';
  xml += '  <Worksheet ss:Name="Ads Data">\n';
  xml += '    <Table>\n';
  
  // Header row
  xml += '      <Row ss:StyleID="Header">\n';
  for (const h of headers) {
    xml += `        <Cell><Data ss:Type="String">${escapeXml(h)}</Data></Cell>\n`;
  }
  xml += '      </Row>\n';
  
  // Data rows
  for (const ad of ads) {
    xml += '      <Row>\n';
    for (const h of headers) {
      const val = ad[h] ?? '';
      const type = typeof val === 'number' ? 'Number' : 'String';
      xml += `        <Cell><Data ss:Type="${type}">${escapeXml(String(val))}</Data></Cell>\n`;
    }
    xml += '      </Row>\n';
  }
  
  xml += '    </Table>\n';
  xml += '  </Worksheet>\n';
  xml += '</Workbook>';
  
  const base64 = utf8ToBase64(xml);
  const dataUrl = `data:application/vnd.ms-excel;charset=utf-8;base64,${base64}`;
  
  try {
    await chrome.downloads.download({
      url: dataUrl,
      filename: filename + '.xls',
      saveAs: false
    });
    log.success('Excel download initiated');
  } catch (e) {
    log.error('Download error:', e);
  }
}

function escapeXml(str) {
  return str
    .replace(/&/g, '&amp;')
    .replace(/</g, '&lt;')
    .replace(/>/g, '&gt;')
    .replace(/"/g, '&quot;')
    .replace(/'/g, '&apos;');
}

async function downloadFile(content, filename, type) {
  const base64 = utf8ToBase64(content);
  await chrome.downloads.download({
    url: `data:${type};charset=utf-8;base64,${base64}`,
    filename: filename,
    saveAs: false
  });
}

function generateFilename(config) {
  // 🔧 修正：统一时间戳格式
  const timestamp = new Date().toISOString().slice(0,19).replace(/[-:T]/g,'_');
  
  // 🆕 新增：品牌模式文件名（使用广告商名字，不带 _brand 后缀）
  if (config.searchMode === 'brand') {
    let baseName = config.brandName || 'ads';
    
    // 如果品牌名称是 Unknown Brand，使用第一个广告商 ID
    if (baseName === 'Unknown Brand' && config.advertiserIds && config.advertiserIds.length > 0) {
      baseName = config.advertiserIds[0].substring(0, 20); // 使用前20个字符
    }
    
    // 🔧 支持 Unicode 字符（包括汉字）
    const safeName = baseName
      .replace(/\s+/g, '_')                    // 空格转下划线
      .replace(/[<>:"/\\|?*]/g, '')            // 只移除文件系统不允许的字符
      .substring(0, 50);                       // 限制长度
    
    // 如果处理后为空（全是特殊字符），使用广告商 ID
    const finalName = safeName.trim() || 
      (config.advertiserIds && config.advertiserIds.length > 0 
        ? config.advertiserIds[0].substring(0, 20) 
        : 'ads');
    
    // 可选：添加 region 后缀
    const region = config.region && config.region !== 'ANYWHERE' 
      ? `_${config.region}` 
      : '';
    
    // 格式: Nike_Hong_Kong_Limited_HK_2026_01_21_...（不带 _brand）
    return `${finalName}${region}_${timestamp}`;
  }
  
  // 现有逻辑：domain/advertiser 模式
  return `${(config.domain || 'ads').replace(/\./g, '_')}_${timestamp}`;
}

function utf8ToBase64(str) {
  const bytes = new TextEncoder().encode(str);
  const len = bytes.byteLength;
  let binary = '';
  const CHUNK_SIZE = 32768; // 32KB chunks to avoid stack overflow
  
  for (let i = 0; i < len; i += CHUNK_SIZE) {
    binary += String.fromCharCode(...bytes.subarray(i, i + CHUNK_SIZE));
  }
  
  return btoa(binary);
}

function cancelDownload() { 
  downloadState.isActive = false;
  // v1.1.0: Cancel quota reservation
  const reservationId = downloadState.quotaReservation?.id;
  if (reservationId) {
    apiClient.cancelQuota(reservationId).catch(() => {});
    downloadState.quotaReservation = { id: null, count: 0, expiresAt: null };
  }
}

// v1.2.0: Get rate limits from remote config
function getRateLimits() {
  return RemoteConfig.getRateLimits();
}

function getDelay() {
  const limits = getRateLimits();
  return downloadState.currentPage < 5 
    ? limits.normal_delay_ms 
    : limits.cautious_delay_ms;
}

function getRecoveryDelay() {
  return getRateLimits().recovery_delay_ms;
}

function sleep(ms) { return new Promise(r => setTimeout(r, ms)); }
function notifyPopup(type, data) { chrome.runtime.sendMessage({ type, ...data }).catch(() => {}); }

async function saveProgress() {
  await chrome.storage.local.set({ [PROGRESS_STORAGE_KEY]: { ...downloadState, timestamp: Date.now() } });
}
async function clearProgress() { await chrome.storage.local.remove(PROGRESS_STORAGE_KEY); }
async function addToDownloadHistory(item) {
  const { downloadHistory = [] } = await chrome.storage.local.get('downloadHistory');
  await chrome.storage.local.set({ downloadHistory: [ { ...item, timestamp: Date.now() }, ...downloadHistory ].slice(0, 50) });
}

// ============== UTILS ==============
async function generateDeviceFingerprint() {
  return { userAgent: navigator.userAgent, language: navigator.language };
}

async function linkInstallToUser(userId) {
  try {
    const { installId } = await chrome.storage.local.get('installId');
    if (!installId) return;
    await fetchWithTimeout(`${getApiUrl()}/api/stats/link`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ installId, userId })
    }, TIMEOUTS.NORMAL);
  } catch (e) {}
}

async function sendInstallPing(installId, isNew) {
  try {
    await fetchWithTimeout(`${getApiUrl()}/api/stats/ping`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ installId, isNew, version: chrome.runtime.getManifest().version })
    }, TIMEOUTS.NORMAL);
  } catch (e) {}
}

// ============== INIT ==============
chrome.runtime.onInstalled.addListener(async (details) => {
  if (details.reason === 'install') {
    const installId = 'INS_' + Math.random().toString(36).substr(2, 9);
    await chrome.storage.local.set({ installId, installDate: Date.now() });
    sendInstallPing(installId, true);
  }
});

(async () => {
  try {
    // v1.2.0: Initialize remote config first
    log.info('Initializing RemoteConfig...');
    await initRemoteConfig();
    log.success('✅ RemoteConfig initialized, version:', RemoteConfig.getConfigVersion());
    
    // Check maintenance mode
    if (RemoteConfig.isMaintenanceMode()) {
      log.warn('⚠️ Service is in maintenance mode');
    }
    
    log.info('Initializing usageMonitor...');
    await usageMonitor.init();
    log.success('✅ UsageMonitor initialized successfully');
    
    // 尝试从服务器恢复数据（如果用户已登录）
    try {
      const restored = await usageMonitor.restoreFromServer();
      if (restored) {
        log.success('✅ Usage stats restored from server');
      }
    } catch (err) {
      log.debug('Could not restore from server:', err.message);
    }
    
    // Update thresholds based on plan
    const { userPlan } = await chrome.storage.local.get('userPlan');
    if (userPlan) {
      await usageMonitor.updatePlan(userPlan);
      log.info(`UsageMonitor plan updated to: ${userPlan}`);
    }
    
    // 显示当前统计
    const stats = usageMonitor.getStats();
    log.info('Current usage stats:', stats);
  } catch (e) {
    log.error('❌ UsageMonitor initialization failed:', e);
    log.warn('Continuing without usage monitoring - this may affect rate limiting protection');
  }
})();

// 打印环境信息
logEnvironmentInfo(log);
log.info('Service worker initialized');

// v1.1.0: 定期清理 OCR 缓存（每 24 小时一次）
(async () => {
  try {
    const { cleanupExpiredCache } = await import('../lib/ocr-cache.js');
    
    // 立即执行一次
    cleanupExpiredCache().then(deleted => {
      if (deleted > 0) {
        log.info(`🧹 OCR cache cleanup: ${deleted} expired entries deleted`);
      }
    }).catch(e => {
      log.debug('OCR cache cleanup error:', e.message);
    });
    
    // 设置定时任务（每 24 小时）
    setInterval(() => {
      cleanupExpiredCache().then(deleted => {
        if (deleted > 0) {
          log.info(`🧹 OCR cache cleanup: ${deleted} expired entries deleted`);
        }
      }).catch(e => {
        log.debug('OCR cache cleanup error:', e.message);
      });
    }, 24 * 60 * 60 * 1000); // 24 hours
  } catch (e) {
    log.debug('OCR cache module not available:', e.message);
  }
})();
