/**
 * Ads Insight Pro - OCR Page Script
 * 
 * This script runs in the page context (not content script context)
 * It loads Tesseract.js and handles OCR requests via custom events
 */

(function() {
  // Prevent duplicate injection
  if (window.__AdsInsightOCR) {
    console.log('[AdsInsightOCR] Already injected');
    if (window.__AdsInsightOCR.ready) {
      window.dispatchEvent(new CustomEvent('AdsInsightOCR_Ready'));
    }
    return;
  }
  
  window.__AdsInsightOCR = { ready: false };
  console.log('[AdsInsightOCR] Initializing...');
  
  // Load Tesseract.js from CDN
  const script = document.createElement('script');
  script.src = 'https://cdn.jsdelivr.net/npm/tesseract.js@5/dist/tesseract.min.js';
  script.onload = function() {
    console.log('[AdsInsightOCR] Tesseract.js script loaded');
    window.__AdsInsightOCR.ready = true;
    window.dispatchEvent(new CustomEvent('AdsInsightOCR_Ready'));
  };
  script.onerror = function(e) {
    console.error('[AdsInsightOCR] Failed to load Tesseract.js:', e);
    window.dispatchEvent(new CustomEvent('AdsInsightOCR_LoadError'));
  };
  document.head.appendChild(script);
  
  // Listen for OCR requests
  window.addEventListener('AdsInsightOCR_Request', async function(e) {
    const { requestId, imageUrl } = e.detail;
    console.log('[AdsInsightOCR] Processing:', requestId);
    
    try {
      if (!window.Tesseract) {
        throw new Error('Tesseract not loaded');
      }
      
      const result = await Tesseract.recognize(imageUrl, 'eng', {
        logger: m => {
          if (m.status === 'recognizing text' && m.progress > 0) {
            console.log('[AdsInsightOCR] Progress:', Math.round(m.progress * 100) + '%');
          }
        }
      });
      
      console.log('[AdsInsightOCR] OCR completed, confidence:', result.data.confidence);
      
      window.dispatchEvent(new CustomEvent('AdsInsightOCR_Response', {
        detail: {
          requestId,
          success: true,
          text: result.data.text,
          confidence: result.data.confidence
        }
      }));
      
    } catch (err) {
      console.error('[AdsInsightOCR] OCR error:', err);
      window.dispatchEvent(new CustomEvent('AdsInsightOCR_Response', {
        detail: {
          requestId,
          success: false,
          error: err.message
        }
      }));
    }
  });
  
  console.log('[AdsInsightOCR] Event handlers registered');
})();
