/**
 * Video Extractor Content Script
 * 用于从 Google Ads Preview 页面提取 YouTube 视频链接
 * 
 * 工作原理：
 * 1. 被注入到 displayads-formats.googleusercontent.com 页面
 * 2. 等待页面加载完成
 * 3. 查找 YouTube iframe 或视频元素
 * 4. 将结果发送回 background script
 */

(function() {
  'use strict';
  
  const LOG_PREFIX = '[AdsInsightPro][VideoExtractor]';
  
  // 提取 YouTube 视频 ID 的函数
  function extractYouTubeId() {
    const results = {
      youtubeId: null,
      youtubeUrl: null,
      thumbnailUrl: null,
      method: null
    };
    
    // 方法 1: 查找 YouTube iframe
    const iframes = document.querySelectorAll('iframe');
    for (const iframe of iframes) {
      const src = iframe.src || iframe.getAttribute('src') || '';
      
      // YouTube embed URL
      let match = src.match(/youtube\.com\/embed\/([a-zA-Z0-9_-]{11})/);
      if (match) {
        results.youtubeId = match[1];
        results.youtubeUrl = `https://www.youtube.com/watch?v=${match[1]}`;
        results.thumbnailUrl = `https://img.youtube.com/vi/${match[1]}/maxresdefault.jpg`;
        results.method = 'iframe-embed';
        return results;
      }
      
      // YouTube watch URL
      match = src.match(/youtube\.com\/watch\?v=([a-zA-Z0-9_-]{11})/);
      if (match) {
        results.youtubeId = match[1];
        results.youtubeUrl = `https://www.youtube.com/watch?v=${match[1]}`;
        results.thumbnailUrl = `https://img.youtube.com/vi/${match[1]}/maxresdefault.jpg`;
        results.method = 'iframe-watch';
        return results;
      }
    }
    
    // 方法 2: 查找 video 元素
    const videos = document.querySelectorAll('video');
    for (const video of videos) {
      const src = video.src || video.getAttribute('src') || '';
      const poster = video.poster || video.getAttribute('poster') || '';
      
      // 检查 poster 是否是 YouTube 缩略图
      let match = poster.match(/img\.youtube\.com\/vi\/([a-zA-Z0-9_-]{11})/);
      if (match) {
        results.youtubeId = match[1];
        results.youtubeUrl = `https://www.youtube.com/watch?v=${match[1]}`;
        results.thumbnailUrl = poster;
        results.method = 'video-poster';
        return results;
      }
      
      // 检查 source 元素
      const sources = video.querySelectorAll('source');
      for (const source of sources) {
        const sourceSrc = source.src || source.getAttribute('src') || '';
        match = sourceSrc.match(/youtube\.com\/embed\/([a-zA-Z0-9_-]{11})/);
        if (match) {
          results.youtubeId = match[1];
          results.youtubeUrl = `https://www.youtube.com/watch?v=${match[1]}`;
          results.thumbnailUrl = `https://img.youtube.com/vi/${match[1]}/maxresdefault.jpg`;
          results.method = 'video-source';
          return results;
        }
      }
    }
    
    // 方法 3: 搜索整个页面的 HTML
    const html = document.documentElement.innerHTML;
    
    // YouTube embed
    let match = html.match(/youtube\.com\/embed\/([a-zA-Z0-9_-]{11})/);
    if (match) {
      results.youtubeId = match[1];
      results.youtubeUrl = `https://www.youtube.com/watch?v=${match[1]}`;
      results.thumbnailUrl = `https://img.youtube.com/vi/${match[1]}/maxresdefault.jpg`;
      results.method = 'html-embed';
      return results;
    }
    
    // YouTube watch
    match = html.match(/youtube\.com\/watch\?v=([a-zA-Z0-9_-]{11})/);
    if (match) {
      results.youtubeId = match[1];
      results.youtubeUrl = `https://www.youtube.com/watch?v=${match[1]}`;
      results.thumbnailUrl = `https://img.youtube.com/vi/${match[1]}/maxresdefault.jpg`;
      results.method = 'html-watch';
      return results;
    }
    
    // YouTube thumbnail
    match = html.match(/img\.youtube\.com\/vi\/([a-zA-Z0-9_-]{11})/);
    if (match) {
      results.youtubeId = match[1];
      results.youtubeUrl = `https://www.youtube.com/watch?v=${match[1]}`;
      results.thumbnailUrl = `https://img.youtube.com/vi/${match[1]}/maxresdefault.jpg`;
      results.method = 'html-thumbnail';
      return results;
    }
    
    return results;
  }
  
  // 等待页面加载并提取
  function waitAndExtract(maxWaitMs = 10000, checkIntervalMs = 500) {
    return new Promise((resolve) => {
      let elapsed = 0;
      
      const check = () => {
        const results = extractYouTubeId();
        
        if (results.youtubeId) {
          console.log(`${LOG_PREFIX} Found YouTube ID: ${results.youtubeId} (method: ${results.method})`);
          resolve(results);
          return;
        }
        
        elapsed += checkIntervalMs;
        if (elapsed >= maxWaitMs) {
          console.log(`${LOG_PREFIX} No YouTube ID found after ${maxWaitMs}ms`);
          resolve(results);
          return;
        }
        
        setTimeout(check, checkIntervalMs);
      };
      
      // 开始检查
      if (document.readyState === 'complete') {
        check();
      } else {
        window.addEventListener('load', check);
      }
    });
  }
  
  // 监听来自 background script 的消息
  chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.type === 'EXTRACT_YOUTUBE_ID') {
      console.log(`${LOG_PREFIX} Received extraction request`);
      
      waitAndExtract(message.maxWaitMs || 10000).then((results) => {
        sendResponse(results);
      });
      
      return true; // 保持消息通道打开
    }
  });
  
  // 自动提取并报告（用于 iframe 注入场景）
  if (window.location.hostname === 'displayads-formats.googleusercontent.com') {
    console.log(`${LOG_PREFIX} Auto-extracting from preview page...`);
    
    waitAndExtract(10000).then((results) => {
      // 发送结果到 background script
      chrome.runtime.sendMessage({
        type: 'YOUTUBE_EXTRACTED',
        results: results,
        url: window.location.href
      });
    });
  }
})();
