/**
 * Decryption utilities for sensitive data
 * Uses AES-GCM for decryption (matching backend encryption)
 * v1.2.0
 */

// Encryption key (will be set from remote config or hardcoded fallback)
let encryptionKey = null;

/**
 * Set the encryption key for decryption
 * @param {string} key - Base64 encoded 32-byte key
 */
export function setEncryptionKey(key) {
  encryptionKey = key;
}

// Default encryption key fallback (null = must be fetched from server)
// The actual key is provided via /api/quota/permissions endpoint
const DEFAULT_ENCRYPTION_KEY = null;

/**
 * Get encryption key from cached permissions
 * Note: In Service Worker context, we cannot use dynamic imports,
 * so we only use chrome.storage.local for the encryption key
 * @returns {Promise<string|null>}
 */
async function getEncryptionKey() {
  if (encryptionKey) {
    return encryptionKey;
  }
  
  // Get from cached permissions (server provides it during quota check)
  try {
    const data = await chrome.storage.local.get('cachedPermissions');
    if (data.cachedPermissions?.encryption_key) {
      encryptionKey = data.cachedPermissions.encryption_key;
      return encryptionKey;
    }
  } catch (e) {
    // Ignore - will use fallback
  }
  
  return DEFAULT_ENCRYPTION_KEY;
}

/**
 * Decrypt data using AES-GCM
 * @param {string} encryptedData - Base64 encoded encrypted data (IV + ciphertext)
 * @returns {Promise<string|null>} - Decrypted plaintext or null on failure
 */
export async function decrypt(encryptedData) {
  try {
    const key = await getEncryptionKey();
    if (!key) {
      return null;
    }
    
    // Decode the secret key from base64
    const keyData = Uint8Array.from(atob(key), c => c.charCodeAt(0));
    
    // Import the key
    const cryptoKey = await crypto.subtle.importKey(
      'raw',
      keyData,
      { name: 'AES-GCM', length: 256 },
      false,
      ['decrypt']
    );
    
    // Decode the combined data from base64
    const combined = Uint8Array.from(atob(encryptedData), c => c.charCodeAt(0));
    
    // Extract IV (first 12 bytes) and ciphertext (rest)
    const iv = combined.slice(0, 12);
    const ciphertext = combined.slice(12);
    
    // Decrypt the data
    const decrypted = await crypto.subtle.decrypt(
      { name: 'AES-GCM', iv },
      cryptoKey,
      ciphertext
    );
    
    // Convert back to string
    return new TextDecoder().decode(decrypted);
  } catch (e) {
    return null;
  }
}

/**
 * Decrypt OCR API config
 * @param {Object} ocrConfig - OCR config with api_key_encrypted field
 * @returns {Promise<Object|null>} - Config with decrypted api_key
 */
export async function decryptOCRConfig(ocrConfig) {
  if (!ocrConfig) {
    return null;
  }
  
  // Check if already decrypted (has api_key instead of api_key_encrypted)
  if (ocrConfig.api_key && !ocrConfig.api_key_encrypted) {
    return ocrConfig;
  }
  
  // Need to decrypt
  if (!ocrConfig.api_key_encrypted) {
    return null;
  }
  
  try {
    const decryptedKey = await decrypt(ocrConfig.api_key_encrypted);
    if (!decryptedKey) {
      return null;
    }
    
    // Return config with decrypted key
    return {
      api_endpoint: ocrConfig.api_endpoint,
      api_key: decryptedKey,
      engine: ocrConfig.engine,
      languages: ocrConfig.languages
    };
  } catch (e) {
    return null;
  }
}
