/**
 * Field Permissions Configuration
 * v1.1.0: Updated according to PERMISSIONS_REDESIGN_V1_1_0.md
 * Defines which fields each plan can export
 */

// v1.1.0 Field Categories
const BASIC_FIELDS = [
  'advertiser_id',
  'advertiser_name',
  'creative_id',
  'target_domain',
  'format',
  'query_region'
];

const ENHANCED_FIELDS = [
  'image_url',
  'preview_url'
];

const CAMPAIGN_INFO_FIELDS = [
  'start_date',
  'end_date',
  'days_shown',
  'regions'
];

const AD_CONTENT_FIELDS = [
  'ad_domain',
  'ad_title',
  'ad_description',
  'confidence'
];

const VIDEO_FIELDS = [
  'video_url',
  'youtube_video_id',     // Legacy compatibility
  'video_thumbnail_url'   // Legacy compatibility
];

// Legacy fields for backward compatibility
const LEGACY_FIELDS = [
  'format_name',          // TEXT/IMAGE/VIDEO
  'ad_type',              // Image/Video
  'transparency_url',
  'first_shown_date',
  'last_shown_date'
];

export const FIELD_PERMISSIONS = {
  // Free Plan - Basic fields + Ad Content Analysis (OCR) results
  // v1.2.0: OCR is available to all users with quota limits
  free: {
    allowed: [
      ...BASIC_FIELDS,
      // Ad Content Analysis fields - available to all users
      ...AD_CONTENT_FIELDS,
      // Legacy fields for UI compatibility
      'format_name',
      'transparency_url'
    ],
    excluded: [
      ...ENHANCED_FIELDS,
      ...CAMPAIGN_INFO_FIELDS,
      ...VIDEO_FIELDS
    ]
  },

  // Pro Plan - 16 fields (Basic + Enhanced + Campaign + Ad Content)
  // v1.1.0: Unlock media assets and performance data
  pro: {
    allowed: [
      ...BASIC_FIELDS,
      ...ENHANCED_FIELDS,
      ...CAMPAIGN_INFO_FIELDS,
      ...AD_CONTENT_FIELDS,
      // Legacy fields
      'format_name',
      'ad_type',
      'transparency_url',
      'first_shown_date',
      'last_shown_date'
    ],
    excluded: [
      ...VIDEO_FIELDS
    ]
  },

  // Enterprise Plan - 17+ fields (All fields including video)
  // v1.1.0: Full data access including video extraction
  enterprise: {
    allowed: 'all',
    excluded: []
  }
};

/**
 * Get allowed fields for a plan
 * @param {string} plan - User's plan (free, pro, enterprise)
 * @returns {Array|string} - Array of allowed fields or 'all'
 */
export function getAllowedFields(plan) {
  const permissions = FIELD_PERMISSIONS[plan] || FIELD_PERMISSIONS.free;
  return permissions.allowed;
}

/**
 * Get excluded fields for a plan
 * @param {string} plan - User's plan
 * @returns {Array} - Array of excluded fields
 */
export function getExcludedFields(plan) {
  const permissions = FIELD_PERMISSIONS[plan] || FIELD_PERMISSIONS.free;
  return permissions.excluded;
}

/**
 * Check if a field is allowed for a plan
 * @param {string} field - Field name
 * @param {string} plan - User's plan
 * @returns {boolean} - True if field is allowed
 */
export function isFieldAllowed(field, plan) {
  const permissions = FIELD_PERMISSIONS[plan] || FIELD_PERMISSIONS.free;
  
  if (permissions.allowed === 'all') {
    return true;
  }
  
  return permissions.allowed.includes(field);
}

/**
 * Filter data fields based on user's plan
 * @param {Array} data - Array of ad objects
 * @param {string} plan - User's plan
 * @returns {Array} - Filtered data with only allowed fields
 */
export function filterFieldsByPlan(data, plan) {
  const permissions = FIELD_PERMISSIONS[plan] || FIELD_PERMISSIONS.free;
  
  // Pro and Enterprise get all fields
  if (permissions.allowed === 'all') {
    return data;
  }
  
  // Free users get filtered fields
  return data.map(ad => {
    const filtered = {};
    permissions.allowed.forEach(field => {
      if (ad[field] !== undefined) {
        filtered[field] = ad[field];
      }
    });
    return filtered;
  });
}

/**
 * Get field count for a plan
 * @param {string} plan - User's plan
 * @returns {number} - Number of allowed fields
 */
export function getFieldCount(plan) {
  const permissions = FIELD_PERMISSIONS[plan] || FIELD_PERMISSIONS.free;
  
  if (permissions.allowed === 'all') {
    return 18; // Total number of fields (updated from 11 to 18)
  }
  
  return permissions.allowed.length;
}

/**
 * Get upgrade message for excluded fields
 * @param {string} plan - Current plan
 * @returns {string} - Upgrade message
 */
export function getUpgradeMessage(plan) {
  if (plan === 'free') {
    const excluded = FIELD_PERMISSIONS.free.excluded;
    return `Upgrade to Pro to unlock ${excluded.length} additional fields: ${excluded.join(', ')}`;
  }
  return '';
}
