/**
 * Ads Insight Pro - OCR Cache Service
 * v1.1.0: 本地 IndexedDB 缓存，减?OCR API 调用
 * 
 * 策略?
 * - 用户视角：每?OCR 都消耗配额（透明?
 * - 运营者视角：缓存命中时不调用 OCR API（节省成本）
 */

import { RemoteConfig } from './remote-config.js';
import { createLogger, DEBUG_CONFIG } from './debug-config.js';

const log = createLogger('[OCRCache]', DEBUG_CONFIG.background);

const DB_NAME = 'AdsInsightPro_OCR_Cache';
const DB_VERSION = 1;
const STORE_NAME = 'ocr_results';
const DEFAULT_CACHE_EXPIRY_DAYS = 30; // 本地默认?

/**
 * v1.2.0: 获取缓存过期天数（从远程配置获取?
 */
function getCacheExpiryDays() {
  return RemoteConfig.get('ocr_config.cache_expiry_days', DEFAULT_CACHE_EXPIRY_DAYS);
}

/**
 * 打开 IndexedDB
 */
function openDB() {
  return new Promise((resolve, reject) => {
    const request = indexedDB.open(DB_NAME, DB_VERSION);
    
    request.onerror = () => reject(request.error);
    request.onsuccess = () => resolve(request.result);
    
    request.onupgradeneeded = (event) => {
      const db = event.target.result;
      
      // 创建 object store
      if (!db.objectStoreNames.contains(STORE_NAME)) {
        const store = db.createObjectStore(STORE_NAME, { keyPath: 'creative_id' });
        store.createIndex('image_url_hash', 'image_url_hash', { unique: false });
        store.createIndex('created_at', 'created_at', { unique: false });
        log.info('[OCR Cache] Object store created');
      }
    };
  });
}

/**
 * 生成 image URL ?hash（简单版?
 */
async function hashImageUrl(imageUrl) {
  const encoder = new TextEncoder();
  const data = encoder.encode(imageUrl);
  const hashBuffer = await crypto.subtle.digest('SHA-256', data);
  const hashArray = Array.from(new Uint8Array(hashBuffer));
  return hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
}

/**
 * 检查缓存是否过?
 * v1.2.0: 使用远程配置的过期天?
 */
function isExpired(createdAt) {
  const now = Date.now();
  const expiryDays = getCacheExpiryDays();
  const expiryTime = createdAt + (expiryDays * 24 * 60 * 60 * 1000);
  return now > expiryTime;
}

/**
 * 从缓存中获取 OCR 结果
 * @param {string} creativeId - 广告创意 ID
 * @param {string} imageUrl - 图片 URL（可选，用于 hash 匹配?
 * @returns {Promise<Object|null>} - 缓存?OCR 结果?null
 */
export async function getCachedOCR(creativeId, imageUrl = null) {
  try {
    const db = await openDB();
    
    return new Promise((resolve, reject) => {
      const transaction = db.transaction([STORE_NAME], 'readonly');
      const store = transaction.objectStore(STORE_NAME);
      const request = store.get(creativeId);
      
      request.onsuccess = () => {
        const cached = request.result;
        
        if (!cached) {
          resolve(null);
          return;
        }
        
        // 检查是否过?
        if (isExpired(cached.created_at)) {
          log.info('[OCR Cache] Cache expired for:', creativeId);
          // 异步删除过期缓存
          deleteCachedOCR(creativeId).catch(() => {});
          resolve(null);
          return;
        }
        
        // 如果提供?imageUrl，验?hash 是否匹配
        if (imageUrl) {
          hashImageUrl(imageUrl).then(hash => {
            if (hash !== cached.image_url_hash) {
              log.info('[OCR Cache] Image URL hash mismatch for:', creativeId);
              resolve(null);
              return;
            }
            
            log.info('[OCR Cache] Cache HIT:', creativeId);
            resolve({
              ad_domain: cached.ad_domain,
              ad_title: cached.ad_title,
              ad_description: cached.ad_description,
              confidence: cached.confidence,
              cached: true
            });
          }).catch(() => resolve(null));
        } else {
          log.info('[OCR Cache] Cache HIT (no URL validation):', creativeId);
          resolve({
            ad_domain: cached.ad_domain,
            ad_title: cached.ad_title,
            ad_description: cached.ad_description,
            confidence: cached.confidence,
            cached: true
          });
        }
      };
      
      request.onerror = () => reject(request.error);
    });
  } catch (e) {
    log.error('[OCR Cache] Get error:', e);
    return null;
  }
}

/**
 * 保存 OCR 结果到缓?
 * @param {string} creativeId - 广告创意 ID
 * @param {string} imageUrl - 图片 URL
 * @param {Object} ocrResult - OCR 结果
 */
export async function setCachedOCR(creativeId, imageUrl, ocrResult) {
  try {
    const db = await openDB();
    const imageUrlHash = await hashImageUrl(imageUrl);
    
    return new Promise((resolve, reject) => {
      const transaction = db.transaction([STORE_NAME], 'readwrite');
      const store = transaction.objectStore(STORE_NAME);
      
      const cacheEntry = {
        creative_id: creativeId,
        image_url_hash: imageUrlHash,
        ad_domain: ocrResult.ad_domain || '',
        ad_title: ocrResult.ad_title || '',
        ad_description: ocrResult.ad_description || '',
        confidence: ocrResult.confidence || 0,
        created_at: Date.now()
      };
      
      const request = store.put(cacheEntry);
      
      request.onsuccess = () => {
        log.info('[OCR Cache] Cached result for:', creativeId);
        resolve(true);
      };
      
      request.onerror = () => {
        log.error('[OCR Cache] Save error:', request.error);
        reject(request.error);
      };
    });
  } catch (e) {
    log.error('[OCR Cache] Set error:', e);
    return false;
  }
}

/**
 * 删除缓存条目
 */
export async function deleteCachedOCR(creativeId) {
  try {
    const db = await openDB();
    
    return new Promise((resolve, reject) => {
      const transaction = db.transaction([STORE_NAME], 'readwrite');
      const store = transaction.objectStore(STORE_NAME);
      const request = store.delete(creativeId);
      
      request.onsuccess = () => resolve(true);
      request.onerror = () => reject(request.error);
    });
  } catch (e) {
    log.error('[OCR Cache] Delete error:', e);
    return false;
  }
}

/**
 * 清理过期缓存（后台任务）
 */
export async function cleanupExpiredCache() {
  try {
    const db = await openDB();
    
    return new Promise((resolve, reject) => {
      const transaction = db.transaction([STORE_NAME], 'readwrite');
      const store = transaction.objectStore(STORE_NAME);
      const request = store.openCursor();
      
      let deletedCount = 0;
      
      request.onsuccess = (event) => {
        const cursor = event.target.result;
        
        if (cursor) {
          const cached = cursor.value;
          
          if (isExpired(cached.created_at)) {
            cursor.delete();
            deletedCount++;
          }
          
          cursor.continue();
        } else {
          log.info(`[OCR Cache] Cleanup complete: ${deletedCount} expired entries deleted`);
          resolve(deletedCount);
        }
      };
      
      request.onerror = () => reject(request.error);
    });
  } catch (e) {
    log.error('[OCR Cache] Cleanup error:', e);
    return 0;
  }
}

/**
 * 获取缓存统计信息
 */
export async function getCacheStats() {
  try {
    const db = await openDB();
    
    return new Promise((resolve, reject) => {
      const transaction = db.transaction([STORE_NAME], 'readonly');
      const store = transaction.objectStore(STORE_NAME);
      const countRequest = store.count();
      
      countRequest.onsuccess = () => {
        const total = countRequest.result;
        
        // 计算过期数量
        const cursorRequest = store.openCursor();
        let expiredCount = 0;
        
        cursorRequest.onsuccess = (event) => {
          const cursor = event.target.result;
          
          if (cursor) {
            if (isExpired(cursor.value.created_at)) {
              expiredCount++;
            }
            cursor.continue();
          } else {
            resolve({
              total,
              valid: total - expiredCount,
              expired: expiredCount
            });
          }
        };
        
        cursorRequest.onerror = () => reject(cursorRequest.error);
      };
      
      countRequest.onerror = () => reject(countRequest.error);
    });
  } catch (e) {
    log.error('[OCR Cache] Stats error:', e);
    return { total: 0, valid: 0, expired: 0 };
  }
}
