/**
 * Ads Insight Pro - OCR Service Module
 * v1.2.0: 本地 Tesseract.js OCR via Content Script 注入
 * 
 * 策略：
 * 1. 主方案：通过 Content Script 在 adstransparency.google.com 页面注入 Tesseract.js
 *    - 该网站 CSP 允许 unsafe-eval，所以 Tesseract.js 可以正常运行
 *    - 完全本地化，无 API 依赖
 * 2. 备选方案：Offscreen Document + OCR.space API
 *    - 当 Content Script 不可用时使用
 */

import { createLogger, DEBUG_CONFIG } from './debug-config.js';
import { permissionsManager } from './permissions.js';
import { getCachedOCR, setCachedOCR } from './ocr-cache.js';
import { RemoteConfig } from './remote-config.js';

// OCR 调试时开启，生产环境设为 false
const OCR_DEBUG = true;
const log = createLogger('[AdsInsightPro][OCR]', OCR_DEBUG || DEBUG_CONFIG.serviceWorker);

// v1.2.0: OCR 配置从远程获取，本地默认值作为回退
const DEFAULT_OCR_TIMEOUT_MS = 15000;
const DEFAULT_MAX_RETRIES = 2;

/**
 * v1.2.0: 获取 OCR 超时配置
 */
function getOCRTimeout() {
  return RemoteConfig.get('ocr_config.timeout_ms', DEFAULT_OCR_TIMEOUT_MS);
}

/**
 * v1.2.0: 获取 OCR 最大重试次数
 */
function getOCRMaxRetries() {
  return RemoteConfig.get('ocr_config.max_retries', DEFAULT_MAX_RETRIES);
}

/**
 * v1.2.0: 检查 OCR 功能是否启用
 */
function isOCREnabled() {
  return RemoteConfig.isOCREnabled();
}

// OCR 方法状态
let contentScriptOCRAvailable = null; // null = unknown, true/false = tested
let offscreenCreating = null;

/**
 * 确保 Offscreen Document 已创建
 */
async function ensureOffscreenDocument() {
  const offscreenUrl = chrome.runtime.getURL('offscreen/offscreen.html');
  
  // Check if offscreen document already exists
  const existingContexts = await chrome.runtime.getContexts({
    contextTypes: ['OFFSCREEN_DOCUMENT'],
    documentUrls: [offscreenUrl]
  });
  
  if (existingContexts.length > 0) {
    log.info('Offscreen document already exists');
    return true;
  }
  
  // Avoid creating multiple documents simultaneously
  if (offscreenCreating) {
    await offscreenCreating;
    return true;
  }
  
  try {
    log.info('Creating offscreen document for OCR...');
    offscreenCreating = chrome.offscreen.createDocument({
      url: offscreenUrl,
      reasons: ['DOM_PARSER'],
      justification: 'OCR text extraction from ad images using Tesseract.js'
    });
    await offscreenCreating;
    log.success('Offscreen document created');
    return true;
  } catch (e) {
    log.error('Failed to create offscreen document:', e);
    return false;
  } finally {
    offscreenCreating = null;
  }
}

/**
 * 关闭 Offscreen Document
 */
async function closeOffscreenDocument() {
  try {
    await chrome.offscreen.closeDocument();
    log.info('Offscreen document closed');
  } catch (e) {
    // Ignore errors (document might not exist)
  }
}

/**
 * 对单张图片执行 OCR（不消耗配额，供内部使用）
 * @param {string} imageUrl - 图片 URL
 * @returns {Promise<Object>} - OCR 结果
 */
export async function performOCR(imageUrl) {
  const result = {
    ad_domain: '',
    ad_title: '',
    ad_description: '',
    confidence: 0,
    raw_text: ''
  };
  
  try {
    if (!imageUrl || !imageUrl.startsWith('http')) {
      return result;
    }
    
    const ready = await ensureOffscreenDocument();
    if (!ready) {
      log.warn('Offscreen document not available');
      return result;
    }
    
    log.info('Sending OCR request to offscreen document...');
    
    // v1.1.0: 获取 OCR 配置并传递给 offscreen
    const ocrConfig = await permissionsManager.getOCRConfig();
    
    const response = await chrome.runtime.sendMessage({
      type: 'OCR_SINGLE',
      imageUrl: imageUrl,
      ocrConfig: ocrConfig // 传递 API 配置
    });
    
    if (response && response.success) {
      return response.result;
    }
    
  } catch (e) {
    log.error('OCR failed:', e);
  }
  
  return result;
}

/**
 * v1.1.0: 带配额管理的单张图片 OCR
 * @param {string} imageUrl - 图片 URL
 * @param {string} creativeId - 广告创意 ID（用于缓存）
 * @returns {Promise<Object>} - OCR 结果
 */
export async function performOCRWithQuota(imageUrl, creativeId = null) {
  const emptyResult = {
    ad_domain: '',
    ad_title: '',
    ad_description: '',
    confidence: 0,
    raw_text: '',
    ocr_status: 'skipped'
  };
  
  try {
    if (!imageUrl || !imageUrl.startsWith('http')) {
      return emptyResult;
    }
    
    // 1. 检查配额
    const quotaCheck = await permissionsManager.checkOCRQuota(1);
    if (!quotaCheck.allowed) {
      log.warn('OCR quota exceeded:', quotaCheck.message);
      return { ...emptyResult, ocr_status: 'quota_exceeded', quota_message: quotaCheck.message };
    }
    
    // 2. 先扣配额（用户视角：无论缓存命中与否，都消耗配额）
    const consumeResult = await permissionsManager.consumeOCRQuota(1);
    if (!consumeResult.success) {
      log.warn('Failed to consume OCR quota:', consumeResult.message);
      return { ...emptyResult, ocr_status: 'consume_failed' };
    }
    
    const transactionId = consumeResult.transaction_id;
    log.info('OCR quota consumed, transaction:', transactionId);
    
    // 3. 检查本地缓存（运营者视角：节省 API 调用）
    if (creativeId) {
      try {
        const cachedResult = await getCachedOCR(creativeId, imageUrl);
        if (cachedResult) {
          log.info('💾 OCR cache HIT - saved API call for:', creativeId);
          // 标记交易完成（即使是缓存命中，配额仍然已消耗）
          await permissionsManager.completeOCRTransaction(transactionId);
          return {
            ...cachedResult,
            ocr_status: 'success',
            from_cache: true
          };
        }
      } catch (cacheError) {
        log.warn('Cache check failed, proceeding with OCR:', cacheError.message);
      }
    }
    
    // 4. 执行实际 OCR（缓存未命中）
    try {
      const ocrResult = await performOCRWithRetry(imageUrl);
      
      // 5. 标记交易完成
      await permissionsManager.completeOCRTransaction(transactionId);
      
      // 6. 保存到本地缓存（用于下次节省 API 调用）
      if (creativeId && ocrResult.confidence > 0) {
        setCachedOCR(creativeId, imageUrl, ocrResult).catch(e => {
          log.warn('Failed to cache OCR result:', e.message);
        });
      }
      
      return {
        ...ocrResult,
        ocr_status: ocrResult.confidence > 0 ? 'success' : 'no_text'
      };
    } catch (ocrError) {
      // 7. OCR 失败，标记失败并尝试退款
      log.error('OCR failed, attempting refund:', ocrError.message);
      
      const failResult = await permissionsManager.failOCRTransaction(transactionId, ocrError.message);
      
      if (failResult.can_refund) {
        const refundResult = await permissionsManager.refundOCRQuota(transactionId);
        log.info('OCR quota refund:', refundResult.success ? 'success' : 'failed');
      }
      
      return { ...emptyResult, ocr_status: 'failed', error: ocrError.message };
    }
  } catch (e) {
    log.error('performOCRWithQuota error:', e);
    return { ...emptyResult, ocr_status: 'error', error: e.message };
  }
}

/**
 * v1.1.0: 带重试的 OCR 执行
 * @param {string} imageUrl - 图片 URL
 * @param {number} maxRetries - 最大重试次数
 * @returns {Promise<Object>}
 */
async function performOCRWithRetry(imageUrl, maxRetries = null) {
  // v1.2.0: 从远程配置获取超时和重试次数
  const timeoutMs = getOCRTimeout();
  const retries = maxRetries !== null ? maxRetries : getOCRMaxRetries();
  let lastError = null;
  
  for (let attempt = 0; attempt <= retries; attempt++) {
    try {
      // 使用超时包装
      const result = await Promise.race([
        performOCR(imageUrl),
        new Promise((_, reject) => 
          setTimeout(() => reject(new Error('OCR timeout')), timeoutMs)
        )
      ]);
      
      return result;
    } catch (e) {
      lastError = e;
      log.warn(`OCR attempt ${attempt + 1} failed:`, e.message);
      
      if (attempt < retries) {
        // 等待后重试
        await new Promise(r => setTimeout(r, 1000 * (attempt + 1)));
      }
    }
  }
  
  throw lastError || new Error('OCR failed after retries');
}

/**
 * 获取当前活动标签页的 ID
 */
async function getActiveTabId() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (tab && tab.url && tab.url.includes('adstransparency.google.com')) {
      return tab.id;
    }
  } catch (e) {
    log.warn('Failed to get active tab:', e);
  }
  return null;
}

/**
 * 通过 Content Script 执行本地 OCR
 * @param {number} tabId - 标签页 ID
 * @param {Array} adsToProcess - 要处理的广告数组
 * @returns {Promise<Array|null>} - OCR 结果或 null（失败）
 */
async function performOCRViaContentScript(tabId, adsToProcess) {
  try {
    log.info('Attempting local OCR via Content Script, tabId:', tabId);
    
    // 首先检查 Content Script 是否已加载
    let isReady = false;
    let tesseractReady = false;
    
    try {
      // 尝试获取 Content Script 状态
      const readyCheck = await chrome.tabs.sendMessage(tabId, { type: 'OCR_CHECK_READY' });
      log.info('Content Script status:', JSON.stringify(readyCheck));
      
      if (readyCheck) {
        isReady = readyCheck.injected === true;
        tesseractReady = readyCheck.ready === true;
      }
    } catch (e) {
      // 如果消息发送失败，可能是 Content Script 没有加载
      log.warn('Content Script not responding:', e.message);
      
      // 尝试重新注入 Content Script
      try {
        log.info('Attempting to inject OCR content script...');
        await chrome.scripting.executeScript({
          target: { tabId: tabId },
          files: ['content/ocr-injector.js']
        });
        log.info('OCR content script injected');
        
        // 等待脚本初始化
        await new Promise(r => setTimeout(r, 2000));
        
        const retryCheck = await chrome.tabs.sendMessage(tabId, { type: 'OCR_CHECK_READY' });
        log.info('After injection, status:', JSON.stringify(retryCheck));
        if (retryCheck) {
          isReady = retryCheck.injected === true;
          tesseractReady = retryCheck.ready === true;
        }
      } catch (injectError) {
        log.error('Failed to inject content script:', injectError.message);
        return null;
      }
    }
    
    if (!isReady) {
      log.warn('OCR Injector not injected');
      return null;
    }
    
    // 如果 Tesseract 还没准备好，等待
    if (!tesseractReady) {
      log.info('Tesseract not ready, waiting up to 15s...');
      
      for (let i = 0; i < 15; i++) {
        await new Promise(r => setTimeout(r, 1000));
        try {
          const retryCheck = await chrome.tabs.sendMessage(tabId, { type: 'OCR_CHECK_READY' });
          if (retryCheck && retryCheck.ready) {
            log.success('Tesseract ready after', i + 1, 'seconds');
            tesseractReady = true;
            break;
          }
        } catch (e) {
          // 继续等待
        }
      }
      
      if (!tesseractReady) {
        log.warn('Tesseract still not ready after 15s');
        return null;
      }
    }
    
    log.info('Sending OCR_BATCH request with', adsToProcess.length, 'images');
    
    // 使用 Promise 包装以添加超时
    const response = await Promise.race([
      chrome.tabs.sendMessage(tabId, {
        type: 'OCR_BATCH',
        ads: adsToProcess
      }),
      new Promise((_, reject) => 
        setTimeout(() => reject(new Error('Content Script OCR timeout (90s)')), 90000)
      )
    ]);
    
    if (response && response.success && response.results) {
      log.success('Local OCR completed:', response.results.length, 'results');
      contentScriptOCRAvailable = true;
      return response.results;
    }
    
    log.warn('Content Script OCR returned invalid response:', response);
    return null;
    
  } catch (e) {
    log.warn('Content Script OCR failed:', e.message);
    contentScriptOCRAvailable = false;
    return null;
  }
}

/**
 * v1.2.0: 获取 OCR 并发配置
 */
function getOCRConcurrencyConfig() {
  return {
    concurrent: RemoteConfig.get('ocr_config.concurrent', 2),
    max_retries: RemoteConfig.get('ocr_config.max_retries', 2),
    retry_delay_ms: RemoteConfig.get('ocr_config.retry_delay_ms', 1500),
    request_delay_ms: RemoteConfig.get('ocr_config.request_delay_ms', 500)
  };
}

/**
 * 通过 Offscreen Document + OCR.space API 执行 OCR（备选方案）
 * @param {Array} adsToProcess - 要处理的广告数组
 * @returns {Promise<Array|null>} - OCR 结果或 null（失败）
 */
async function performOCRViaOffscreen(adsToProcess) {
  try {
    log.info('Using Offscreen + OCR.space API as fallback...');
    
    const ready = await ensureOffscreenDocument();
    if (!ready) {
      log.warn('Offscreen document not available');
      return null;
    }
    
    // v1.1.0: 获取 OCR 配置并传递给 offscreen
    const ocrConfig = await permissionsManager.getOCRConfig();
    
    // v1.2.1: 获取并发配置
    const concurrencyConfig = getOCRConcurrencyConfig();
    log.info('OCR concurrency config:', concurrencyConfig);
    
    const response = await chrome.runtime.sendMessage({
      type: 'OCR_PROCESS_ADS',
      ads: adsToProcess,
      ocrConfig: ocrConfig,
      concurrencyConfig: concurrencyConfig // v1.2.1: 传递并发配置
    });
    
    if (response && response.success && response.results) {
      log.success('Offscreen OCR completed:', response.results.length, 'results');
      return response.results;
    }
    
    log.warn('Offscreen OCR returned invalid response');
    return null;
    
  } catch (e) {
    log.error('Offscreen OCR failed:', e);
    return null;
  }
}

/**
 * v1.2.1: 带配额管理的批量 OCR 处理（并行版本）
 * @param {Array} ads - 广告数组
 * @param {Object} options - 选项
 * @param {Function} onProgress - 进度回调
 * @returns {Promise<Object>} - { ads: [], ocrStats: {} }
 */
export async function processAdsWithOCRAndQuota(ads, options = {}, onProgress) {
  const { maxOCR = Infinity, abortSignal = null } = options;
  
  // 🔧 FIX: 筛选需要 OCR 的广告（支持 _temp_image_url 以兼容 Free 用户）
  const imageAds = ads.filter(ad => {
    const imageUrl = ad.image_url || ad._temp_image_url;
    return imageUrl && 
           imageUrl.startsWith('http') &&
           !ad.ad_title;
  });
  
  const stats = {
    total: imageAds.length,
    processed: 0,
    successful: 0,
    skipped: 0,
    failed: 0,
    quota_exceeded: 0
  };
  
  if (imageAds.length === 0) {
    log.info('No image ads need OCR processing');
    return { ads, ocrStats: stats };
  }
  
  // 检查配额
  const quotaCheck = await permissionsManager.checkOCRQuota(Math.min(imageAds.length, maxOCR));
  const effectiveMax = Math.min(maxOCR, quotaCheck.effective_remaining);
  
  if (effectiveMax === 0) {
    log.warn('No OCR quota available');
    for (const ad of imageAds) {
      ad.ocr_status = 'quota_exceeded';
      stats.quota_exceeded++;
    }
    return { ads, ocrStats: stats };
  }
  
  // 🔧 FIX: 准备要处理的广告（限制数量，支持 _temp_image_url）
  const adsToProcess = imageAds.slice(0, effectiveMax).map(ad => ({
    creative_id: ad.creative_id,
    image_url: ad.image_url || ad._temp_image_url
  }));
  
  // 标记超出配额的广告
  for (let i = effectiveMax; i < imageAds.length; i++) {
    imageAds[i].ocr_status = 'skipped';
    stats.skipped++;
  }
  
  log.info(`Starting batch OCR for ${adsToProcess.length} of ${imageAds.length} image ads`);
  
  try {
    // v1.2.1: 使用批量并行处理
    const ready = await ensureOffscreenDocument();
    if (!ready) {
      log.warn('Offscreen document not available');
      for (const ad of imageAds.slice(0, effectiveMax)) {
        ad.ocr_status = 'failed';
        stats.failed++;
      }
      return { ads, ocrStats: stats };
    }
    
    // 获取 OCR 配置
    const ocrConfig = await permissionsManager.getOCRConfig();
    const concurrencyConfig = getOCRConcurrencyConfig();
    
    log.info('Sending batch OCR request, concurrency:', concurrencyConfig.concurrent);
    
    // 发送批量请求到 offscreen document
    const response = await chrome.runtime.sendMessage({
      type: 'OCR_PROCESS_ADS',
      ads: adsToProcess,
      ocrConfig: ocrConfig,
      concurrencyConfig: concurrencyConfig
    });
    
    if (response && response.success && response.results) {
      log.info('Batch OCR completed:', response.results.length, 'results');
      
      // 处理结果
      const resultsMap = new Map(response.results.map(r => [r.creative_id, r]));
      
      for (const ad of imageAds.slice(0, effectiveMax)) {
        const ocrResult = resultsMap.get(ad.creative_id);
        
        if (ocrResult && (ocrResult.confidence > 0.1 || ocrResult.ad_title || ocrResult.raw_text)) {
          ad.ad_domain = ocrResult.ad_domain || ad.ad_domain || '';
          ad.ad_title = ocrResult.ad_title || ad.ad_title || '';
          ad.ad_description = ocrResult.ad_description || ad.ad_description || '';
          ad.confidence = ocrResult.confidence || 0;
          ad.ocr_status = 'success';
          stats.successful++;
        } else if (ocrResult && ocrResult.error) {
          ad.ocr_status = 'failed';
          stats.failed++;
        } else {
          ad.ocr_status = 'no_text';
          stats.successful++;
        }
        stats.processed++;
      }
      
      // 保存到缓存
      for (const result of response.results) {
        if (result.confidence > 0) {
          setCachedOCR(result.creative_id, adsToProcess.find(a => a.creative_id === result.creative_id)?.image_url, result).catch(() => {});
        }
      }
    } else {
      log.warn('Batch OCR failed');
      for (const ad of imageAds.slice(0, effectiveMax)) {
        ad.ocr_status = 'failed';
        stats.failed++;
        stats.processed++;
      }
    }
    
    // 关闭 offscreen document
    await closeOffscreenDocument();
    
  } catch (e) {
    log.error('Batch OCR error:', e);
    for (const ad of imageAds.slice(0, effectiveMax)) {
      if (!ad.ocr_status) {
        ad.ocr_status = 'failed';
        stats.failed++;
        stats.processed++;
      }
    }
  }
  
  // 最终进度回调
  if (onProgress) {
    onProgress({
      processed: stats.processed,
      total: adsToProcess.length,
      successful: stats.successful,
      percentage: 100
    });
  }
  
  // 🔧 FIX: 消费 OCR 配额（只计算成功处理的数量）
  if (stats.successful > 0) {
    try {
      const consumeResult = await permissionsManager.consumeOCRQuota(stats.successful);
      if (consumeResult.success) {
        log.info(`OCR quota consumed: ${stats.successful}, transaction: ${consumeResult.transaction_id}`);
        // 标记交易完成
        await permissionsManager.completeOCRTransaction(consumeResult.transaction_id);
      } else {
        log.warn('Failed to consume OCR quota:', consumeResult.message);
      }
    } catch (e) {
      log.error('OCR quota consumption error:', e);
    }
  }
  
  log.success(`OCR complete: ${stats.successful}/${stats.processed} successful, ${stats.skipped} skipped, ${stats.failed} failed`);
  
  return { ads, ocrStats: stats };
}

/**
 * 批量处理广告的 OCR（原版本，不消耗配额，内部使用）
 * 优先使用 Content Script 本地 Tesseract.js，回退到 Offscreen + OCR.space API
 * 
 * @param {Array} ads - 广告数组
 * @param {Function} onProgress - 进度回调
 * @returns {Promise<Array>} - 处理后的广告数组
 */
export async function processAdsWithOCR(ads, onProgress) {
  // 筛选需要 OCR 的广告（所有有图片 URL 的广告）
  const imageAds = ads.filter(ad => 
    ad.image_url && 
    ad.image_url.startsWith('http') &&
    !ad.ad_title
  );
  
  if (imageAds.length === 0) {
    log.info('No image ads need OCR processing');
    return ads;
  }
  
  log.info('Starting OCR for', imageAds.length, 'image ads');
  
  // 准备要发送的数据
  const adsToProcess = imageAds.map(ad => ({
    creative_id: ad.creative_id,
    image_url: ad.image_url
  }));
  
  let ocrResults = null;
  let ocrMethod = 'none';
  
  // 🎯 方案 A: 尝试通过 Content Script 执行本地 OCR（首选）
  if (contentScriptOCRAvailable !== false) {
    const tabId = await getActiveTabId();
    if (tabId) {
      ocrResults = await performOCRViaContentScript(tabId, adsToProcess);
      if (ocrResults) {
        ocrMethod = 'local-tesseract';
      }
    } else {
      log.warn('No active adstransparency tab found for local OCR');
    }
  }
  
  // 🔄 方案 F: 回退到 Offscreen + OCR.space API
  if (!ocrResults) {
    ocrResults = await performOCRViaOffscreen(adsToProcess);
    if (ocrResults) {
      ocrMethod = 'ocr-space-api';
    }
  }
  
  // 处理结果
  if (ocrResults && ocrResults.length > 0) {
    log.info('OCR results received via', ocrMethod, ':', ocrResults.length, 'results');
    const resultsMap = new Map(ocrResults.map(r => [r.creative_id, r]));
    
    let successful = 0;
    for (const ad of ads) {
      const ocrResult = resultsMap.get(ad.creative_id);
      
      // 降低置信度阈值到 0.1，并且即使 confidence 为 0 也保存有效文本
      if (ocrResult && (ocrResult.confidence > 0.1 || ocrResult.ad_title || ocrResult.raw_text)) {
        ad.ad_domain = ocrResult.ad_domain || ad.ad_domain || '';
        ad.ad_title = ocrResult.ad_title || ad.ad_title || '';
        ad.ad_description = ocrResult.ad_description || ad.ad_description || '';
        ad.confidence = ocrResult.confidence || 0;
        successful++;
      }
    }
    
    log.success(`OCR complete [${ocrMethod}]:`, successful + '/' + imageAds.length, 'successful');
    
    if (onProgress) {
      onProgress({
        processed: imageAds.length,
        total: imageAds.length,
        successful,
        percentage: 100,
        method: ocrMethod
      });
    }
  } else {
    log.error('All OCR methods failed');
  }
  
  return ads;
}

// 域名正则
const DOMAIN_REGEX = /(?:https?:\/\/)?(?:www\.)?([a-zA-Z0-9][-a-zA-Z0-9]*(?:\.[a-zA-Z0-9][-a-zA-Z0-9]*)+)/gi;

/**
 * 从广告数据中提取文本内容（备用方案，从 HTML 提取）
 * @param {Object} ad - 广告对象
 * @param {Object} rawItem - API 返回的原始数据
 * @returns {Object} - 提取的文本内容
 */
export function extractTextFromAdData(ad, rawItem) {
  const result = {
    ad_domain: '',
    ad_title: '',
    ad_description: '',
    confidence: 0
  };
  
  try {
    // 尝试从 creative HTML 中提取
    const creative = rawItem?.["3"] || rawItem?.[3];
    if (creative) {
      const htmlContent = creative?.["3"]?.["2"] || creative?.[3]?.[2] || '';
      
      if (htmlContent && typeof htmlContent === 'string') {
        // 提取 href 中的域名
        const hrefMatch = htmlContent.match(/href=["']([^"']+)["']/i);
        if (hrefMatch) {
          const domainMatch = hrefMatch[1].match(DOMAIN_REGEX);
          if (domainMatch) {
            const domain = domainMatch[0]
              .replace(/^https?:\/\//i, '')
              .replace(/^www\./i, '')
              .split('/')[0];
            if (domain && !domain.includes('google')) {
              result.ad_domain = domain;
              result.confidence = 0.7;
            }
          }
        }
      }
    }
    
    // 使用 target_domain 作为备选
    if (!result.ad_domain && ad.target_domain) {
      result.ad_domain = ad.target_domain;
      result.confidence = Math.max(result.confidence, 0.9);
    }
    
  } catch (e) {
    // 静默失败
  }
  
  return result;
}

/**
 * 终止 OCR（关闭 Offscreen Document）
 */
export async function terminateOCR() {
  await closeOffscreenDocument();
}

export default {
  performOCR,
  performOCRWithQuota,
  processAdsWithOCR,
  processAdsWithOCRAndQuota,
  extractTextFromAdData,
  terminateOCR
};
