/**
 * Ads Insight Pro - Data Parser Module
 * Handles data transformation and filtering
 */

/**
 * Deduplicate ads by creative_id
 * @param {Array} ads - Array of ad objects
 * @returns {Object} Result with deduplicated ads and stats
 */
export function deduplicateAds(ads) {
  const seen = new Set();
  const unique = [];
  let duplicates = 0;
  
  for (const ad of ads) {
    if (seen.has(ad.creative_id)) {
      duplicates++;
      continue;
    }
    seen.add(ad.creative_id);
    unique.push(ad);
  }
  
  return {
    ads: unique,
    duplicatesRemoved: duplicates,
    originalCount: ads.length
  };
}

/**
 * Filter ads by format
 * @param {Array} ads - Array of ad objects
 * @param {string} format - Format to filter (TEXT, IMAGE, VIDEO)
 * @returns {Array} Filtered ads
 */
export function filterByFormat(ads, format) {
  if (!format) return ads;
  return ads.filter(ad => ad.format === format);
}

/**
 * Filter ads by date range
 * @param {Array} ads - Array of ad objects
 * @param {string} startDate - Start date (YYYY-MM-DD)
 * @param {string} endDate - End date (YYYY-MM-DD)
 * @returns {Array} Filtered ads
 */
export function filterByDateRange(ads, startDate, endDate) {
  return ads.filter(ad => {
    if (!ad.start_date) return true;
    
    if (startDate && ad.start_date < startDate) return false;
    if (endDate && ad.end_date && ad.end_date > endDate) return false;
    
    return true;
  });
}

/**
 * Calculate advertiser statistics
 * @param {Array} ads - Array of ad objects
 * @returns {Object} Statistics object
 */
export function calculateStats(ads) {
  const advertisers = new Map();
  const formats = { TEXT: 0, IMAGE: 0, VIDEO: 0, UNKNOWN: 0 };
  
  for (const ad of ads) {
    // Count by advertiser
    const count = advertisers.get(ad.advertiser_id) || {
      id: ad.advertiser_id,
      name: ad.advertiser_name,
      count: 0
    };
    count.count++;
    advertisers.set(ad.advertiser_id, count);
    
    // Count by format
    formats[ad.format] = (formats[ad.format] || 0) + 1;
  }
  
  // Sort advertisers by count
  const topAdvertisers = Array.from(advertisers.values())
    .sort((a, b) => b.count - a.count)
    .slice(0, 10);
  
  return {
    totalAds: ads.length,
    uniqueAdvertisers: advertisers.size,
    formats,
    topAdvertisers
  };
}

/**
 * Prepare ads for export (select fields based on plan)
 * @param {Array} ads - Array of ad objects
 * @param {Object} options - Export options
 * @param {boolean} options.includeEnhanced - Include enhanced fields (Pro)
 * @returns {Array} Prepared ads
 */
export function prepareForExport(ads, options = {}) {
  const { includeEnhanced = false } = options;
  
  return ads.map(ad => {
    const base = {
      advertiser_id: ad.advertiser_id,
      advertiser_name: ad.advertiser_name,
      creative_id: ad.creative_id,
      target_domain: ad.target_domain,
      format: ad.format,
      days_shown: ad.days_shown
    };
    
    if (includeEnhanced) {
      return {
        ...base,
        image_url: ad.image_url || '',
        preview_url: ad.preview_url || '',
        start_date: ad.start_date || '',
        end_date: ad.end_date || ''
      };
    }
    
    return base;
  });
}
