﻿/**
 * Ads Insight Pro - Permissions Manager
 * v1.1.0: Server-driven permissions system
 * 
 * All permission/quota values are fetched from server.
 * Local PLAN_LIMITS in constants.js are only used as fallback.
 */

import * as storage from './storage.js';
import { getApiUrl } from './config.js';
import { fetchWithTimeout, TIMEOUTS } from './fetch-with-timeout.js';
import { PLAN_LIMITS } from './constants.js';
import { decryptOCRConfig } from './decryption.js';

// Cache TTL: 5 minutes
const CACHE_TTL = 5 * 60 * 1000;

// Storage keys
const STORAGE_KEYS = {
  PERMISSIONS: 'cachedPermissions',
  CACHE_TIME: 'permissionsCacheTime',
  RESERVATION: 'currentReservation',
  OCR_TRANSACTION: 'currentOCRTransaction'  // v1.1.0: OCR 交易记录
};

/**
 * PermissionsManager class
 * Handles server-driven permissions with local caching
 */
class PermissionsManager {
  constructor() {
    this.cache = null;
    this.cacheExpiry = 0;
  }

  /**
   * Get API URL with endpoint
   */
  _getUrl(endpoint) {
    return `${getApiUrl()}${endpoint}`;
  }

  /**
   * Make authenticated API request
   */
  async _apiRequest(endpoint, options = {}) {
    const token = await storage.getAuthToken();
    if (!token) {
      throw new Error('Not authenticated');
    }

    const response = await fetchWithTimeout(
      this._getUrl(endpoint),
      {
        ...options,
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`,
          ...options.headers
        }
      },
      TIMEOUTS.NORMAL
    );

    if (!response.ok) {
      if (response.status === 401) {
        await storage.clearAuth();
        throw new Error('Session expired');
      }
      throw new Error(`API error: ${response.status}`);
    }

    return response.json();
  }

  /**
   * Get current user permissions (with caching)
   * @param {boolean} forceRefresh - Force refresh from server
   * @returns {Promise<Object>} Permissions object
   */
  async getPermissions(forceRefresh = false) {
    // Check memory cache
    if (!forceRefresh && this.cache && Date.now() < this.cacheExpiry) {
      return this.cache;
    }

    // Check storage cache
    if (!forceRefresh) {
      const stored = await chrome.storage.local.get([
        STORAGE_KEYS.PERMISSIONS,
        STORAGE_KEYS.CACHE_TIME
      ]);
      
      if (stored[STORAGE_KEYS.PERMISSIONS] && stored[STORAGE_KEYS.CACHE_TIME]) {
        const cacheAge = Date.now() - stored[STORAGE_KEYS.CACHE_TIME];
        if (cacheAge < CACHE_TTL) {
          this.cache = stored[STORAGE_KEYS.PERMISSIONS];
          this.cacheExpiry = stored[STORAGE_KEYS.CACHE_TIME] + CACHE_TTL;
          return this.cache;
        }
      }
    }

    // Fetch from server
    try {
      const response = await this._apiRequest('/api/quota/permissions');
      
      if (response.success) {
        this.cache = response.data;
        this.cacheExpiry = Date.now() + CACHE_TTL;
        
        // Save to storage
        await chrome.storage.local.set({
          [STORAGE_KEYS.PERMISSIONS]: response.data,
          [STORAGE_KEYS.CACHE_TIME]: Date.now()
        });
        
        return response.data;
      }
      
      throw new Error(response.error?.message || 'Failed to fetch permissions');
    } catch (e) {
      // Try to use cached permissions
      const stored = await chrome.storage.local.get(STORAGE_KEYS.PERMISSIONS);
      if (stored[STORAGE_KEYS.PERMISSIONS]) {
        return stored[STORAGE_KEYS.PERMISSIONS];
      }
      
      // Last resort: use local fallback based on cached plan
      const plan = await storage.get('userPlan', 'free');
      return this._getFallbackPermissions(plan);
    }
  }

  /**
   * Get fallback permissions from local constants
   */
  _getFallbackPermissions(plan) {
    const limits = PLAN_LIMITS[plan] || PLAN_LIMITS.free;
    
    // v1.1.0: OCR 配额默认�?
    const ocrQuotas = {
      free: { daily: 10, monthly: 300 },
      pro: { daily: 100, monthly: 3000 },
      enterprise: { daily: 500, monthly: 15000 }
    };
    const ocrLimits = ocrQuotas[plan] || ocrQuotas.free;
    
    return {
      plan,
      quota: {
        daily_limit: limits.dailyQuota,
        per_request_limit: limits.perRequestLimit,
        used_today: 0,
        remaining: limits.dailyQuota,
        resets_at: new Date(Date.now() + 86400000).toISOString()
      },
      // v1.1.0: OCR 配额
      ocr_quota: {
        daily_limit: ocrLimits.daily,
        monthly_limit: ocrLimits.monthly,
        daily_used: 0,
        monthly_used: 0,
        daily_remaining: ocrLimits.daily,
        monthly_remaining: ocrLimits.monthly,
        daily_resets_at: new Date(Date.now() + 86400000).toISOString(),
        monthly_resets_at: new Date(new Date().setMonth(new Date().getMonth() + 1, 1)).toISOString()
      },
      features: {
        export_formats: limits.features.filter(f => ['csv', 'json', 'excel'].includes(f)),
        video_extraction: limits.features.includes('video_extraction'),
        ad_content_export: limits.features.includes('ad_content_export'),
        campaign_info_export: limits.features.includes('campaign_info_export'),
        advertiser_export: limits.features.includes('advertiser_export'),
        enhanced_fields: limits.features.includes('enhanced_fields'),
        ocr_export: plan !== 'free' // Pro/Enterprise 可以导出 OCR 结果
      },
      limits: {
        max_pages: limits.maxPages,
        max_devices: limits.maxDevices
      },
      _fallback: true // Indicate this is fallback data
    };
  }

  /**
   * Check if a feature is available
   * @param {string} feature - Feature name
   * @returns {Promise<boolean>}
   */
  async canUseFeature(feature) {
    const perms = await this.getPermissions();
    const value = perms.features[feature];
    
    if (Array.isArray(value)) {
      return value.length > 0;
    }
    return value === true;
  }

  /**
   * Get available export formats
   * @returns {Promise<string[]>}
   */
  async getExportFormats() {
    const perms = await this.getPermissions();
    return perms.features.export_formats || ['csv'];
  }

  /**
   * Check if quota is sufficient for export
   * @param {number} requestedAds - Number of ads to export
   * @returns {Promise<Object>} Quota check result
   */
  async checkQuota(requestedAds) {
    // Force refresh to get accurate quota
    const perms = await this.getPermissions(true);
    
    const result = {
      allowed: true,
      remaining: perms.quota.remaining,
      limit: perms.quota.daily_limit,
      perRequestLimit: perms.quota.per_request_limit,
      needed: requestedAds,
      plan: perms.plan
    };

    // Check per-request limit
    if (requestedAds > perms.quota.per_request_limit) {
      result.allowed = false;
      result.reason = 'per_request_exceeded';
      result.message = `Maximum ${perms.quota.per_request_limit} ads per export for ${perms.plan} plan`;
      return result;
    }

    // Check daily quota
    if (requestedAds > perms.quota.remaining) {
      result.allowed = false;
      result.reason = 'quota_exceeded';
      result.message = `Insufficient quota. You need ${requestedAds} ads but only have ${perms.quota.remaining} remaining`;
      return result;
    }

    return result;
  }

  /**
   * Reserve quota before export
   * @param {number} count - Number of ads to reserve
   * @returns {Promise<Object>} Reservation result
   */
  async reserveQuota(count) {
    try {
      const response = await this._apiRequest('/api/quota/reserve', {
        method: 'POST',
        body: JSON.stringify({ count })
      });

      if (response.success && response.data.success) {
        // Store reservation for later confirm/cancel
        await chrome.storage.local.set({
          [STORAGE_KEYS.RESERVATION]: {
            id: response.data.reservation_id,
            count,
            expires_at: response.data.expires_at,
            created_at: Date.now()
          }
        });

        // Update cached remaining
        if (this.cache) {
          this.cache.quota.remaining = response.data.remaining;
        }

        return {
          success: true,
          reservationId: response.data.reservation_id,
          expiresAt: response.data.expires_at,
          remaining: response.data.remaining
        };
      }

      return {
        success: false,
        reason: response.data?.reason || 'unknown',
        message: response.data?.message || 'Failed to reserve quota',
        remaining: response.data?.remaining,
        limit: response.data?.limit
      };
    } catch (e) {
      log.error('Reserve quota error:', e);
      return {
        success: false,
        reason: 'network_error',
        message: e.message
      };
    }
  }

  /**
   * Confirm quota consumption after successful export
   * @param {string} reservationId - Reservation ID
   * @param {number} actualCount - Actual number of ads exported
   * @returns {Promise<Object>} Confirm result
   */
  async confirmQuota(reservationId, actualCount) {
    try {
      const response = await this._apiRequest('/api/quota/confirm', {
        method: 'POST',
        body: JSON.stringify({
          reservation_id: reservationId,
          actual_count: actualCount
        })
      });

      // Clear stored reservation
      await chrome.storage.local.remove(STORAGE_KEYS.RESERVATION);

      // Update cached remaining
      if (response.success && response.data.remaining !== undefined) {
        if (this.cache) {
          this.cache.quota.remaining = response.data.remaining;
          this.cache.quota.used_today += actualCount;
        }
      }

      // Invalidate cache to force refresh on next check
      this.cacheExpiry = 0;

      return {
        success: response.success,
        confirmedCount: response.data?.confirmed_count,
        remaining: response.data?.remaining
      };
    } catch (e) {
      log.error('Confirm quota error:', e);
      // Non-critical - the export already succeeded
      return { success: false, error: e.message };
    }
  }

  /**
   * Cancel quota reservation (on export failure)
   * @param {string} reservationId - Reservation ID
   * @returns {Promise<Object>} Cancel result
   */
  async cancelQuota(reservationId) {
    try {
      await this._apiRequest('/api/quota/cancel', {
        method: 'POST',
        body: JSON.stringify({ reservation_id: reservationId })
      });

      // Clear stored reservation
      await chrome.storage.local.remove(STORAGE_KEYS.RESERVATION);

      // Invalidate cache
      this.cacheExpiry = 0;

      return { success: true };
    } catch (e) {
      log.error('Cancel quota error:', e);
      // Non-critical - reservation will auto-expire
      return { success: false, error: e.message };
    }
  }

  /**
   * Get current reservation (if any)
   * @returns {Promise<Object|null>}
   */
  async getCurrentReservation() {
    const stored = await chrome.storage.local.get(STORAGE_KEYS.RESERVATION);
    const reservation = stored[STORAGE_KEYS.RESERVATION];
    
    if (!reservation) return null;
    
    // Check if expired
    const now = Math.floor(Date.now() / 1000);
    if (reservation.expires_at < now) {
      await chrome.storage.local.remove(STORAGE_KEYS.RESERVATION);
      return null;
    }
    
    return reservation;
  }

  /**
   * Clear all cached data (call on login/logout to refresh permissions)
   */
  async clearCache() {
    this.cache = null;
    this.cacheExpiry = 0;
    await chrome.storage.local.remove([
      STORAGE_KEYS.PERMISSIONS,
      STORAGE_KEYS.CACHE_TIME,
      STORAGE_KEYS.RESERVATION,
      STORAGE_KEYS.OCR_TRANSACTION
    ]);
  }

  /**
   * Get quota display info for UI
   * @returns {Promise<Object>}
   */
  async getQuotaDisplayInfo() {
    const perms = await this.getPermissions();
    return {
      plan: perms.plan,
      used: perms.quota.used_today,
      limit: perms.quota.daily_limit,
      remaining: perms.quota.remaining,
      perRequestLimit: perms.quota.per_request_limit,
      resetsAt: perms.quota.resets_at,
      percentage: Math.round((perms.quota.used_today / perms.quota.daily_limit) * 100)
    };
  }

  // ==================== v1.1.0 OCR 配额管理 ====================

  /**
   * 检�?OCR 配额是否充足
   * @param {number} count - 需要的 OCR 次数
   * @returns {Promise<Object>} 配额检查结�?
   */
  async checkOCRQuota(count = 1) {
    const perms = await this.getPermissions(true); // 强制刷新
    const ocrQuota = perms.ocr_quota;
    
    if (!ocrQuota) {
      // 没有 OCR 配额信息，可能是旧版本缓�?
      return {
        allowed: false,
        reason: 'ocr_not_available',
        message: 'OCR quota information not available'
      };
    }
    
    const dailyRemaining = ocrQuota.daily_remaining || (ocrQuota.daily_limit - ocrQuota.daily_used);
    const monthlyRemaining = ocrQuota.monthly_remaining || (ocrQuota.monthly_limit - ocrQuota.monthly_used);
    const effectiveRemaining = Math.min(dailyRemaining, monthlyRemaining);
    
    const result = {
      allowed: true,
      daily_remaining: dailyRemaining,
      monthly_remaining: monthlyRemaining,
      effective_remaining: effectiveRemaining,
      blocked_by: dailyRemaining < monthlyRemaining ? 'daily' : 'monthly',
      daily_limit: ocrQuota.daily_limit,
      monthly_limit: ocrQuota.monthly_limit,
      needed: count,
      plan: perms.plan
    };
    
    // 检查日配额
    if (count > dailyRemaining) {
      result.allowed = false;
      result.reason = 'daily_exceeded';
      result.message = `Daily OCR quota exceeded. Need ${count} but only ${dailyRemaining} remaining`;
      return result;
    }
    
    // 检查月配额
    if (count > monthlyRemaining) {
      result.allowed = false;
      result.reason = 'monthly_exceeded';
      result.message = `Monthly OCR quota exceeded. Need ${count} but only ${monthlyRemaining} remaining`;
      return result;
    }
    
    return result;
  }

  /**
   * 消�?OCR 配额
   * @param {number} count - 消耗数�?
   * @param {string} imageUrlHash - 图片 URL 哈希（可选）
   * @returns {Promise<Object>} 消耗结�?
   */
  async consumeOCRQuota(count = 1, imageUrlHash = null) {
    try {
      const response = await this._apiRequest('/api/ocr/consume', {
        method: 'POST',
        body: JSON.stringify({ count, image_url_hash: imageUrlHash })
      });
      
      if (response.success && response.data.success) {
        // 存储交易记录
        await chrome.storage.local.set({
          [STORAGE_KEYS.OCR_TRANSACTION]: {
            id: response.data.transaction_id,
            count,
            created_at: Date.now()
          }
        });
        
        // 更新内存缓存
        if (this.cache && this.cache.ocr_quota) {
          this.cache.ocr_quota.daily_used += count;
          this.cache.ocr_quota.monthly_used += count;
          this.cache.ocr_quota.daily_remaining = Math.max(0, this.cache.ocr_quota.daily_remaining - count);
          this.cache.ocr_quota.monthly_remaining = Math.max(0, this.cache.ocr_quota.monthly_remaining - count);
          
          // 🔧 FIX: 同时更新 chrome.storage.local 中的持久化缓�?
          // 这样 Account 页面就能立即看到更新后的数据
          chrome.storage.local.set({
            [STORAGE_KEYS.PERMISSIONS]: this.cache,
            [STORAGE_KEYS.CACHE_TIME]: Date.now()
          }).catch(() => {});
        }
        
        return {
          success: true,
          transaction_id: response.data.transaction_id,
          daily_remaining: response.data.daily_remaining,
          monthly_remaining: response.data.monthly_remaining
        };
      }
      
      return {
        success: false,
        reason: response.data?.reason || 'unknown',
        message: response.error?.message || 'Failed to consume OCR quota'
      };
    } catch (e) {
      log.error('Consume OCR quota error:', e);
      return {
        success: false,
        reason: 'network_error',
        message: e.message
      };
    }
  }

  /**
   * 标记 OCR 交易完成
   * @param {string} transactionId - 交易 ID
   * @returns {Promise<Object>}
   */
  async completeOCRTransaction(transactionId) {
    try {
      await this._apiRequest('/api/ocr/transaction/complete', {
        method: 'POST',
        body: JSON.stringify({ transaction_id: transactionId })
      });
      
      // 清除存储的交易记�?
      await chrome.storage.local.remove(STORAGE_KEYS.OCR_TRANSACTION);
      
      return { success: true };
    } catch (e) {
      return { success: false, error: e.message };
    }
  }

  /**
   * 标记 OCR 交易失败
   * @param {string} transactionId - 交易 ID
   * @param {string} errorMessage - 错误信息
   * @returns {Promise<Object>}
   */
  async failOCRTransaction(transactionId, errorMessage = 'Unknown error') {
    try {
      const response = await this._apiRequest('/api/ocr/transaction/failed', {
        method: 'POST',
        body: JSON.stringify({ transaction_id: transactionId, error_message: errorMessage })
      });
      
      return {
        success: true,
        can_refund: response.data?.can_refund || false,
        refund_expires_at: response.data?.refund_expires_at
      };
    } catch (e) {
      return { success: false, error: e.message };
    }
  }

  /**
   * 退�?OCR 配额（仅限失败交易，60秒窗口内�?
   * @param {string} transactionId - 交易 ID
   * @returns {Promise<Object>}
   */
  async refundOCRQuota(transactionId) {
    try {
      const response = await this._apiRequest('/api/ocr/refund', {
        method: 'POST',
        body: JSON.stringify({ transaction_id: transactionId })
      });
      
      if (response.success && response.data.success) {
        // 清除存储的交易记�?
        await chrome.storage.local.remove(STORAGE_KEYS.OCR_TRANSACTION);
        
        // 刷新配额缓存
        this.cacheExpiry = 0;
        
        return {
          success: true,
          refunded_count: response.data.refunded_count
        };
      }
      
      return {
        success: false,
        reason: response.error?.code || 'unknown',
        message: response.error?.message || 'Failed to refund OCR quota'
      };
    } catch (e) {
      return { success: false, error: e.message };
    }
  }

  /**
   * 获取 OCR 配额显示信息
   * @returns {Promise<Object>}
   */
  async getOCRQuotaDisplayInfo() {
    const perms = await this.getPermissions();
    const ocrQuota = perms.ocr_quota || {};
    
    return {
      plan: perms.plan,
      daily_used: ocrQuota.daily_used || 0,
      daily_limit: ocrQuota.daily_limit || 10,
      daily_remaining: ocrQuota.daily_remaining || 0,
      monthly_used: ocrQuota.monthly_used || 0,
      monthly_limit: ocrQuota.monthly_limit || 300,
      monthly_remaining: ocrQuota.monthly_remaining || 0,
      daily_percentage: Math.round(((ocrQuota.daily_used || 0) / (ocrQuota.daily_limit || 10)) * 100),
      monthly_percentage: Math.round(((ocrQuota.monthly_used || 0) / (ocrQuota.monthly_limit || 300)) * 100),
      can_export_ocr: perms.features?.ocr_export === true
    };
  }

  /**
   * 获取当前 OCR 交易（如果有�?
   * @returns {Promise<Object|null>}
   */
  async getCurrentOCRTransaction() {
    const stored = await chrome.storage.local.get(STORAGE_KEYS.OCR_TRANSACTION);
    return stored[STORAGE_KEYS.OCR_TRANSACTION] || null;
  }

  /**
   * v1.2.0: 获取 OCR API 配置（从后端下发，加密存储）
   * 🔧 FIX: Free 用户也可以使�?OCR（有配额限制），ocr_export 只控制是否导出字�?
   * @returns {Promise<Object|null>}
   */
  async getOCRConfig() {
    const perms = await this.getPermissions();
    
    // 🔧 FIX: 不再检�?ocr_export，所有用户都可以使用 OCR（有配额限制�?
    // ocr_export 只控制是否在导出文件中包�?OCR 结果字段
    
    const ocrConfig = perms.ocr_config;
    if (!ocrConfig) {
      return null;
    }
    
    // v1.2.0: 支持加密�?API Key
    if (ocrConfig.api_key_encrypted) {
      // 新格式：加密�?API Key，需要解�?
      return await decryptOCRConfig(ocrConfig);
    }
    
    // 旧格式：未加密的 API Key（向后兼容）
    if (ocrConfig.api_key) {
      return {
        api_endpoint: ocrConfig.api_endpoint || 'https://api.ocr.space/parse/image',
        api_key: ocrConfig.api_key,
        engine: ocrConfig.engine || 1,
        languages: ocrConfig.languages || ['eng']
      };
    }
    
    // 没有 OCR 配置
    return null;
  }
}

// Export singleton instance
export const permissionsManager = new PermissionsManager();

// Also export class for testing
export { PermissionsManager };
