/**
 * Ads Insight Pro - Rate Limiter Module
 * Handles request rate limiting and 429 recovery
 */

// Default rate limit configurations
const DEFAULT_CONFIG = {
  direct: {
    normal: 2000,      // 2 seconds for first 5 pages
    cautious: 5000,    // 5 seconds for pages 6-20
    recovery: 30000    // 30 seconds for pages 21+
  },
  proxy: {
    normal: 500,       // 0.5 seconds
    cautious: 1000,    // 1 second
    recovery: 5000     // 5 seconds
  }
};

// Page thresholds for rate limiting
const THRESHOLDS = {
  normal: 5,
  cautious: 20
};

/**
 * Rate Limiter class
 */
export class RateLimiter {
  constructor(options = {}) {
    this.mode = options.mode || 'direct';
    this.config = options.config || DEFAULT_CONFIG[this.mode];
    this.currentPage = 0;
    this.consecutiveErrors = 0;
    this.maxRetries = options.maxRetries || 3;
    this.backoffMultiplier = options.backoffMultiplier || 2;
  }
  
  /**
   * Get delay for current page
   * @returns {number} Delay in milliseconds
   */
  getDelay() {
    // If we've had errors, use recovery delay
    if (this.consecutiveErrors > 0) {
      return this.config.recovery * Math.pow(this.backoffMultiplier, this.consecutiveErrors - 1);
    }
    
    // Progressive delay based on page number
    if (this.currentPage < THRESHOLDS.normal) {
      return this.config.normal;
    }
    
    if (this.currentPage < THRESHOLDS.cautious) {
      return this.config.cautious;
    }
    
    return this.config.recovery;
  }
  
  /**
   * Wait for the appropriate delay
   * @returns {Promise<void>}
   */
  async wait() {
    const delay = this.getDelay();
    await this.sleep(delay);
  }
  
  /**
   * Record successful request
   */
  onSuccess() {
    this.currentPage++;
    this.consecutiveErrors = 0;
  }
  
  /**
   * Record failed request
   * @param {Error} error - The error that occurred
   * @returns {boolean} Whether to retry
   */
  onError(error) {
    this.consecutiveErrors++;
    
    // Check if we should retry
    if (this.consecutiveErrors <= this.maxRetries) {
      return true;
    }
    
    return false;
  }
  
  /**
   * Check if rate limited (429 error)
   * @param {Error} error - The error to check
   * @returns {boolean}
   */
  isRateLimited(error) {
    return error.message.includes('429') || 
           error.message.toLowerCase().includes('rate limit');
  }
  
  /**
   * Reset the rate limiter
   */
  reset() {
    this.currentPage = 0;
    this.consecutiveErrors = 0;
  }
  
  /**
   * Set mode (direct or proxy)
   * @param {string} mode - 'direct' or 'proxy'
   */
  setMode(mode) {
    this.mode = mode;
    this.config = DEFAULT_CONFIG[mode] || DEFAULT_CONFIG.direct;
  }
  
  /**
   * Sleep for specified milliseconds
   * @param {number} ms - Milliseconds to sleep
   * @returns {Promise<void>}
   */
  sleep(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
  }
  
  /**
   * Get current status
   * @returns {Object} Status object
   */
  getStatus() {
    return {
      mode: this.mode,
      currentPage: this.currentPage,
      consecutiveErrors: this.consecutiveErrors,
      currentDelay: this.getDelay()
    };
  }
}

// Export default instance
export default new RateLimiter();
