/**
 * Ads Insight Pro - Rating Prompt Module
 * Prompts users to rate the extension after successful usage
 */

import { createLogger, DEBUG_CONFIG } from './debug-config.js';

const log = createLogger('[Rating]', DEBUG_CONFIG.background);

const RATING_CONFIG = {
  minDownloads: 5,        // Minimum downloads before prompting
  minDaysInstalled: 7,    // Minimum days since install
  maxPrompts: 3,          // Maximum times to show prompt
  promptCooldown: 7,      // Days between prompts
  chromeStoreUrl: 'https://chrome.google.com/webstore/detail/ads-insight-pro/YOUR_EXTENSION_ID/reviews'
};

/**
 * Check if we should show rating prompt
 * @returns {Promise<boolean>}
 */
export async function shouldShowRatingPrompt() {
  try {
    const data = await chrome.storage.local.get([
      'ratingDismissed',
      'ratingCompleted',
      'ratingPromptCount',
      'lastRatingPrompt',
      'totalDownloads',
      'installDate'
    ]);
    
    // User already rated or permanently dismissed
    if (data.ratingCompleted || data.ratingDismissed === 'never') {
      return false;
    }
    
    // Max prompts reached
    if ((data.ratingPromptCount || 0) >= RATING_CONFIG.maxPrompts) {
      return false;
    }
    
    // Check cooldown
    if (data.lastRatingPrompt) {
      const daysSinceLastPrompt = (Date.now() - data.lastRatingPrompt) / (1000 * 60 * 60 * 24);
      if (daysSinceLastPrompt < RATING_CONFIG.promptCooldown) {
        return false;
      }
    }
    
    // Check minimum downloads
    if ((data.totalDownloads || 0) < RATING_CONFIG.minDownloads) {
      return false;
    }
    
    // Check minimum days installed
    if (data.installDate) {
      const daysInstalled = (Date.now() - data.installDate) / (1000 * 60 * 60 * 24);
      if (daysInstalled < RATING_CONFIG.minDaysInstalled) {
        return false;
      }
    }
    
    return true;
  } catch (e) {
    log.error('Rating check error:', e);
    return false;
  }
}

/**
 * Record that rating prompt was shown
 */
export async function recordRatingPromptShown() {
  const data = await chrome.storage.local.get('ratingPromptCount');
  await chrome.storage.local.set({
    ratingPromptCount: (data.ratingPromptCount || 0) + 1,
    lastRatingPrompt: Date.now()
  });
}

/**
 * Handle user rating action
 * @param {string} action - 'rate', 'later', 'never'
 */
export async function handleRatingAction(action) {
  switch (action) {
    case 'rate':
      await chrome.storage.local.set({ ratingCompleted: true });
      chrome.tabs.create({ url: RATING_CONFIG.chromeStoreUrl });
      break;
    case 'later':
      // Just record the prompt was shown (handled by recordRatingPromptShown)
      break;
    case 'never':
      await chrome.storage.local.set({ ratingDismissed: 'never' });
      break;
  }
}

/**
 * Increment total downloads counter
 */
export async function incrementDownloadCount() {
  const data = await chrome.storage.local.get('totalDownloads');
  await chrome.storage.local.set({
    totalDownloads: (data.totalDownloads || 0) + 1
  });
}

/**
 * Create rating prompt HTML
 * @returns {string} HTML string
 */
export function createRatingPromptHTML() {
  return `
    <div class="rating-prompt" id="rating-prompt">
      <div class="rating-content">
        <div class="rating-icon">?/div>
        <h4>Enjoying Ads Insight Pro?</h4>
        <p>Help us grow by leaving a review!</p>
        <div class="rating-actions">
          <button class="btn-primary btn-small" id="btn-rate">Rate Now</button>
          <button class="btn-secondary btn-small" id="btn-rate-later">Maybe Later</button>
        </div>
        <button class="btn-link" id="btn-rate-never">Don't ask again</button>
      </div>
    </div>
  `;
}

/**
 * Create rating prompt CSS
 * @returns {string} CSS string
 */
export function createRatingPromptCSS() {
  return `
    .rating-prompt {
      position: fixed;
      bottom: 0;
      left: 0;
      right: 0;
      background: white;
      border-top: 1px solid #eee;
      padding: 16px;
      box-shadow: 0 -4px 12px rgba(0,0,0,0.1);
      z-index: 1000;
      animation: slideUp 0.3s ease;
    }
    
    @keyframes slideUp {
      from { transform: translateY(100%); }
      to { transform: translateY(0); }
    }
    
    .rating-content {
      text-align: center;
    }
    
    .rating-icon {
      font-size: 32px;
      margin-bottom: 8px;
    }
    
    .rating-content h4 {
      font-size: 15px;
      font-weight: 600;
      color: #333;
      margin-bottom: 4px;
    }
    
    .rating-content p {
      font-size: 13px;
      color: #666;
      margin-bottom: 12px;
    }
    
    .rating-actions {
      display: flex;
      gap: 8px;
      justify-content: center;
      margin-bottom: 8px;
    }
    
    .rating-actions button {
      min-width: 100px;
    }
    
    .btn-small {
      padding: 8px 16px;
      font-size: 13px;
    }
  `;
}
