/**
 * Ads Insight Pro - Remote Configuration Module (v1.2.0)
 * 
 * Handles fetching, caching, validating and merging remote configuration.
 * Implements security measures including XSS filtering, URL validation,
 * and version downgrade protection.
 * 
 * Priority: Remote Config > Cached Config > Local Defaults
 */

import { LOCAL_DEFAULTS } from './constants.js';
import { getApiUrl, CONFIG } from './config.js';
import { createLogger, DEBUG_CONFIG } from './debug-config.js';

const log = createLogger('[RemoteConfig]', DEBUG_CONFIG.background);

// ============== CONSTANTS ==============

const LOG_PREFIX = '[RemoteConfig]';

// Cache configuration
const CACHE_CONFIG = {
  client: {
    storage_key: 'remote_config_client',
    ttl_ms: 60 * 60 * 1000,              // 1 hour valid
    stale_ttl_ms: 24 * 60 * 60 * 1000    // 24 hours stale but usable
  },
  history: {
    storage_key: 'remote_config_history',
    max_versions: 3                       // Keep last 3 versions for rollback
  }
};

// Allowed domains for URLs in configuration
const ALLOWED_DOMAINS = [
  'adsinsightpro.com',
  'www.adsinsightpro.com',
  'chromewebstore.google.com',
  'addons.mozilla.org'
];

// Numeric value limits for validation
const NUMERIC_LIMITS = {
  'rate_limits.normal_delay_ms': { min: 500, max: 60000 },
  'rate_limits.cautious_delay_ms': { min: 1000, max: 120000 },
  'rate_limits.recovery_delay_ms': { min: 5000, max: 300000 },
  'rate_limits.safe_hourly_limit': { min: 100, max: 2000 },
  'rate_limits.safe_daily_limit': { min: 1000, max: 20000 },
  'ocr_config.timeout_ms': { min: 5000, max: 120000 },
  'ocr_config.max_retries': { min: 0, max: 5 },
  'ocr_config.cache_expiry_days': { min: 1, max: 90 },
  'timeouts.short_ms': { min: 5000, max: 30000 },
  'timeouts.normal_ms': { min: 10000, max: 60000 },
  'timeouts.long_ms': { min: 30000, max: 120000 },
  'timeouts.very_long_ms': { min: 60000, max: 300000 },
  'video_extraction.concurrent': { min: 1, max: 10 },
  'video_extraction.timeout_ms': { min: 10000, max: 120000 }
};

// Array field limits
const ARRAY_LIMITS = {
  'ocr_config.supported_languages': { max: 10, pattern: /^[a-z]{2,5}$/ },
  'field_metadata.export_order': { max: 50, pattern: /^[a-z_]+$/ },
  'field_metadata.single_ad_exclude': { max: 20, pattern: /^[a-z_]+$/ },
  'announcements': { max: 10 }
};

// Boolean fields that must be strictly boolean
const BOOLEAN_FIELDS = [
  'feature_flags.ocr_enabled',
  'feature_flags.video_extraction_enabled',
  'feature_flags.cache_enabled',
  'feature_flags.debug_mode',
  'feature_flags.maintenance_mode',
  'version_control.force_update'
];

// ============== STATE ==============

let runtimeConfig = null;
let configVersion = 0;
let isInitialized = false;

// ============== SECURITY FUNCTIONS ==============

/**
 * Enhanced XSS sanitization for text content
 */
function sanitizeText(text) {
  if (typeof text !== 'string') return '';
  
  return text
    // Decode and remove HTML entities
    .replace(/&#(\d+);/g, '')
    .replace(/&#x([0-9a-f]+);/gi, '')
    .replace(/&[a-z]+;/gi, '')
    // Remove HTML tags
    .replace(/<[^>]*>/g, '')
    // Remove dangerous protocols (expanded list)
    .replace(/\b(javascript|data|vbscript|file|blob):/gi, '')
    // Remove event handlers
    .replace(/\bon\w+\s*=/gi, '')
    // Remove expression()
    .replace(/expression\s*\(/gi, '')
    // Limit length
    .slice(0, 500)
    .trim();
}

/**
 * Validate URL with domain whitelist
 */
function isValidUrl(url) {
  if (typeof url !== 'string') return false;
  
  try {
    const u = new URL(url);
    
    // HTTPS only
    if (u.protocol !== 'https:') {
      return false;
    }
    
    // Domain whitelist
    const hostname = u.hostname.toLowerCase();
    return ALLOWED_DOMAINS.some(domain => 
      hostname === domain || hostname.endsWith('.' + domain)
    );
  } catch {
    return false;
  }
}

/**
 * Validate semver format
 */
function isValidSemver(version) {
  if (typeof version !== 'string') return false;
  return /^\d+\.\d+\.\d+$/.test(version);
}

/**
 * Clamp numeric value within limits
 */
function clamp(value, min, max) {
  return Math.max(min, Math.min(max, value));
}

/**
 * Validate and sanitize array field
 */
function validateArray(path, array) {
  const limit = ARRAY_LIMITS[path];
  if (!limit) return array;
  
  if (!Array.isArray(array)) return [];
  
  let result = array.slice(0, limit.max);
  
  if (limit.pattern) {
    result = result.filter(item => 
      typeof item === 'string' && limit.pattern.test(item)
    );
  }
  
  return result;
}

/**
 * Sanitize UI text object (multilingual)
 */
function sanitizeUIText(obj) {
  if (typeof obj === 'string') {
    return sanitizeText(obj);
  }
  if (typeof obj === 'object' && obj !== null) {
    const result = {};
    const allowedLangs = ['en', 'zh', 'zh-CN', 'zh-TW'];
    for (const [key, value] of Object.entries(obj)) {
      if (allowedLangs.includes(key) && typeof value === 'string') {
        result[key] = sanitizeText(value);
      }
    }
    return result;
  }
  return {};
}

// ============== VALIDATION ==============

/**
 * Validate and clean remote configuration
 */
function validateConfig(config) {
  if (!config || typeof config !== 'object') {
    log.warn(LOG_PREFIX, 'Invalid config object');
    return {};
  }
  
  const validated = {};
  
  // Rate limits
  if (config.rate_limits && typeof config.rate_limits === 'object') {
    validated.rate_limits = {};
    const rl = config.rate_limits;
    
    if (typeof rl.normal_delay_ms === 'number') {
      validated.rate_limits.normal_delay_ms = clamp(rl.normal_delay_ms, 500, 60000);
    }
    if (typeof rl.cautious_delay_ms === 'number') {
      validated.rate_limits.cautious_delay_ms = clamp(rl.cautious_delay_ms, 1000, 120000);
    }
    if (typeof rl.recovery_delay_ms === 'number') {
      validated.rate_limits.recovery_delay_ms = clamp(rl.recovery_delay_ms, 5000, 300000);
    }
    if (typeof rl.safe_hourly_limit === 'number') {
      validated.rate_limits.safe_hourly_limit = clamp(rl.safe_hourly_limit, 100, 2000);
    }
    if (typeof rl.safe_daily_limit === 'number') {
      validated.rate_limits.safe_daily_limit = clamp(rl.safe_daily_limit, 1000, 20000);
    }
  }
  
  // OCR config (from ocr_defaults in API response)
  if (config.ocr_defaults && typeof config.ocr_defaults === 'object') {
    validated.ocr_config = {};
    const ocr = config.ocr_defaults;
    
    if (typeof ocr.timeout_ms === 'number') {
      validated.ocr_config.timeout_ms = clamp(ocr.timeout_ms, 5000, 120000);
    }
    if (typeof ocr.max_retries === 'number') {
      validated.ocr_config.max_retries = clamp(ocr.max_retries, 0, 5);
    }
    if (typeof ocr.cache_expiry_days === 'number') {
      validated.ocr_config.cache_expiry_days = clamp(ocr.cache_expiry_days, 1, 90);
    }
    if (Array.isArray(ocr.supported_languages)) {
      validated.ocr_config.supported_languages = validateArray(
        'ocr_config.supported_languages', 
        ocr.supported_languages
      );
    }
  }
  
  // Timeouts
  if (config.timeouts && typeof config.timeouts === 'object') {
    validated.timeouts = {};
    const t = config.timeouts;
    
    if (typeof t.short_ms === 'number') {
      validated.timeouts.short_ms = clamp(t.short_ms, 5000, 30000);
    }
    if (typeof t.normal_ms === 'number') {
      validated.timeouts.normal_ms = clamp(t.normal_ms, 10000, 60000);
    }
    if (typeof t.long_ms === 'number') {
      validated.timeouts.long_ms = clamp(t.long_ms, 30000, 120000);
    }
    if (typeof t.very_long_ms === 'number') {
      validated.timeouts.very_long_ms = clamp(t.very_long_ms, 60000, 300000);
    }
  }
  
  // Feature flags
  if (config.feature_flags && typeof config.feature_flags === 'object') {
    validated.feature_flags = {};
    const ff = config.feature_flags;
    
    const booleanKeys = ['ocr_enabled', 'video_extraction_enabled', 'cache_enabled', 
                         'debug_mode', 'maintenance_mode'];
    booleanKeys.forEach(key => {
      if (typeof ff[key] === 'boolean') {
        validated.feature_flags[key] = ff[key];
      }
    });
  }
  
  // Version control
  if (config.version_control && typeof config.version_control === 'object') {
    validated.version_control = {};
    const vc = config.version_control;
    
    if (isValidSemver(vc.min_supported_version)) {
      validated.version_control.min_supported_version = vc.min_supported_version;
    }
    if (isValidSemver(vc.latest_version)) {
      validated.version_control.latest_version = vc.latest_version;
    }
    if (typeof vc.force_update === 'boolean') {
      validated.version_control.force_update = vc.force_update;
    }
    if (vc.update_url && typeof vc.update_url === 'object') {
      validated.version_control.update_url = {};
      if (isValidUrl(vc.update_url.chrome)) {
        validated.version_control.update_url.chrome = vc.update_url.chrome;
      }
      if (isValidUrl(vc.update_url.firefox)) {
        validated.version_control.update_url.firefox = vc.update_url.firefox;
      }
    }
    if (vc.deprecation_warning && typeof vc.deprecation_warning === 'string') {
      validated.version_control.deprecation_warning = sanitizeText(vc.deprecation_warning);
    }
  }
  
  // UI text
  if (config.ui_text && typeof config.ui_text === 'object') {
    validated.ui_text = {};
    for (const [key, value] of Object.entries(config.ui_text)) {
      if (typeof key === 'string' && key.length < 50) {
        validated.ui_text[key] = sanitizeUIText(value);
      }
    }
  }
  
  // Announcements
  if (Array.isArray(config.announcements)) {
    validated.announcements = config.announcements
      .slice(0, 10) // Max 10 announcements
      .filter(a => a && typeof a.id === 'string' && a.message)
      .map(a => ({
        id: sanitizeText(a.id).slice(0, 50),
        type: ['info', 'warning', 'error'].includes(a.type) ? a.type : 'info',
        message: sanitizeUIText(a.message),
        action_url: isValidUrl(a.action_url) ? a.action_url : null,
        expires_at: typeof a.expires_at === 'string' ? a.expires_at : null,
        dismissible: a.dismissible !== false
      }));
  }
  
  // Error templates
  if (config.error_templates && typeof config.error_templates === 'object') {
    validated.error_templates = {};
    for (const [key, value] of Object.entries(config.error_templates)) {
      if (typeof key === 'string' && key.length < 50) {
        validated.error_templates[key] = sanitizeUIText(value);
      }
    }
  }
  
  // Field metadata
  if (config.field_metadata && typeof config.field_metadata === 'object') {
    validated.field_metadata = {};
    
    if (Array.isArray(config.field_metadata.export_order)) {
      validated.field_metadata.export_order = validateArray(
        'field_metadata.export_order',
        config.field_metadata.export_order
      );
    }
    if (Array.isArray(config.field_metadata.single_ad_exclude)) {
      validated.field_metadata.single_ad_exclude = validateArray(
        'field_metadata.single_ad_exclude',
        config.field_metadata.single_ad_exclude
      );
    }
  }
  
  // Config version
  if (typeof config.config_version === 'number' && config.config_version > 0) {
    validated.config_version = config.config_version;
  }
  
  return validated;
}

/**
 * Health check for configuration
 */
function healthCheck(config) {
  const issues = [];
  
  // Check rate limits logic
  if (config.rate_limits) {
    const rl = config.rate_limits;
    if (rl.normal_delay_ms >= rl.cautious_delay_ms) {
      issues.push('rate_limits: normal_delay_ms should be less than cautious_delay_ms');
    }
    if (rl.cautious_delay_ms >= rl.recovery_delay_ms) {
      issues.push('rate_limits: cautious_delay_ms should be less than recovery_delay_ms');
    }
  }
  
  // Check version control
  if (config.version_control) {
    const vc = config.version_control;
    if (vc.min_supported_version && vc.latest_version) {
      if (compareVersions(vc.min_supported_version, vc.latest_version) > 0) {
        issues.push('version_control: min_supported_version cannot be greater than latest_version');
      }
    }
  }
  
  return issues;
}

/**
 * Compare semver versions
 */
function compareVersions(v1, v2) {
  const parts1 = v1.split('.').map(Number);
  const parts2 = v2.split('.').map(Number);
  
  for (let i = 0; i < 3; i++) {
    if (parts1[i] > parts2[i]) return 1;
    if (parts1[i] < parts2[i]) return -1;
  }
  return 0;
}

// ============== CACHE OPERATIONS ==============

/**
 * Load cached configuration
 */
async function loadCachedConfig() {
  try {
    const { [CACHE_CONFIG.client.storage_key]: cached } = await chrome.storage.local.get(
      CACHE_CONFIG.client.storage_key
    );
    
    if (!cached || !cached.data) return null;
    
    const now = Date.now();
    const age = now - cached.timestamp;
    
    // Check if still usable (within stale TTL)
    if (age < CACHE_CONFIG.client.stale_ttl_ms) {
      return {
        data: cached.data,
        isFresh: age < CACHE_CONFIG.client.ttl_ms,
        age: age
      };
    }
    
    return null;
  } catch (e) {
    log.warn(LOG_PREFIX, 'Failed to load cached config:', e.message);
    return null;
  }
}

/**
 * Save configuration to cache with history
 */
async function saveCachedConfig(config) {
  try {
    const now = Date.now();
    
    // Save current config
    await chrome.storage.local.set({
      [CACHE_CONFIG.client.storage_key]: {
        data: config,
        timestamp: now,
        version: config.config_version || 0
      }
    });
    
    // Update history for rollback
    const { [CACHE_CONFIG.history.storage_key]: history = [] } = await chrome.storage.local.get(
      CACHE_CONFIG.history.storage_key
    );
    
    // Add to history (avoid duplicates)
    const newHistory = [
      { config, timestamp: now, version: config.config_version || 0 },
      ...history.filter(h => h.version !== config.config_version)
    ].slice(0, CACHE_CONFIG.history.max_versions);
    
    await chrome.storage.local.set({
      [CACHE_CONFIG.history.storage_key]: newHistory
    });
    
  } catch (e) {
    log.warn(LOG_PREFIX, 'Failed to save config cache:', e.message);
  }
}

/**
 * Rollback to previous configuration
 */
async function rollbackConfig() {
  try {
    const { [CACHE_CONFIG.history.storage_key]: history = [] } = await chrome.storage.local.get(
      CACHE_CONFIG.history.storage_key
    );
    
    if (history.length > 1) {
      const previousConfig = history[1].config;
      runtimeConfig = mergeConfig(LOCAL_DEFAULTS, previousConfig);
      configVersion = previousConfig.config_version || 0;
      log.info(LOG_PREFIX, 'Rolled back to version:', configVersion);
      return true;
    }
    
    // No history, use local defaults
    runtimeConfig = { ...LOCAL_DEFAULTS };
    configVersion = 0;
    log.info(LOG_PREFIX, 'Rolled back to LOCAL_DEFAULTS');
    return true;
  } catch (e) {
    log.error(LOG_PREFIX, 'Rollback failed:', e.message);
    return false;
  }
}

// ============== FETCH OPERATIONS ==============

/**
 * Fetch configuration from server
 */
async function fetchClientConfig() {
  const url = `${getApiUrl()}/api/config/client`;
  
  const response = await fetch(url, {
    method: 'GET',
    headers: {
      'Accept': 'application/json',
      'X-Extension-Version': CONFIG.VERSION || '1.0.0'
    }
  });
  
  if (!response.ok) {
    throw new Error(`HTTP ${response.status}`);
  }
  
  const result = await response.json();
  
  if (!result.success) {
    throw new Error(result.error?.message || 'Failed to fetch config');
  }
  
  return result.data;
}

// ============== CONFIG MERGE ==============

/**
 * Deep merge configuration objects
 */
function mergeConfig(base, override) {
  const result = { ...base };
  
  for (const [key, value] of Object.entries(override)) {
    if (value && typeof value === 'object' && !Array.isArray(value)) {
      result[key] = mergeConfig(result[key] || {}, value);
    } else if (value !== undefined) {
      result[key] = value;
    }
  }
  
  return result;
}

// ============== PUBLIC API ==============

/**
 * Initialize remote configuration
 * @returns {Promise<Object>} Runtime configuration
 */
export async function initRemoteConfig() {
  try {
    // 1. Load cached config first for fast startup
    const cached = await loadCachedConfig();
    
    if (cached) {
      runtimeConfig = mergeConfig(LOCAL_DEFAULTS, cached.data);
      configVersion = cached.data.config_version || 0;
      log.info(LOG_PREFIX, 'Loaded cached config, version:', configVersion, 
                  cached.isFresh ? '(fresh)' : '(stale)');
    } else {
      runtimeConfig = { ...LOCAL_DEFAULTS };
      log.info(LOG_PREFIX, 'Using local defaults');
    }
    
    isInitialized = true;
    
    // 2. Refresh in background if cache is stale or missing
    if (!cached || !cached.isFresh) {
      refreshConfigInBackground();
    }
    
    return runtimeConfig;
  } catch (e) {
    log.warn(LOG_PREFIX, 'Init failed, using local defaults:', e.message);
    runtimeConfig = { ...LOCAL_DEFAULTS };
    isInitialized = true;
    return runtimeConfig;
  }
}

/**
 * Refresh configuration from server (background)
 */
async function refreshConfigInBackground() {
  try {
    log.info(LOG_PREFIX, 'Refreshing config from server...');
    const freshConfig = await fetchClientConfig();
    const validated = validateConfig(freshConfig);
    
    // Version downgrade protection
    if (validated.config_version && validated.config_version < configVersion) {
      log.warn(LOG_PREFIX, 'Rejected config downgrade:', 
                   validated.config_version, '<', configVersion);
      return;
    }
    
    // Health check
    const issues = healthCheck(validated);
    if (issues.length > 0) {
      log.warn(LOG_PREFIX, 'Config health issues:', issues);
      // Continue but log issues
    }
    
    // Save and apply
    await saveCachedConfig(validated);
    runtimeConfig = mergeConfig(LOCAL_DEFAULTS, validated);
    configVersion = validated.config_version || configVersion;
    
    log.info(LOG_PREFIX, 'Config refreshed, version:', configVersion);
  } catch (e) {
    log.warn(LOG_PREFIX, 'Background refresh failed:', e.message);
  }
}

/**
 * Get configuration value by path
 * @param {string} path - Dot-separated path (e.g., 'rate_limits.normal_delay_ms')
 * @param {*} defaultValue - Default value if path not found
 * @returns {*} Configuration value
 */
export function getConfigValue(path, defaultValue = undefined) {
  if (!runtimeConfig) {
    // Not initialized, use local defaults
    return getValueByPath(LOCAL_DEFAULTS, path, defaultValue);
  }
  
  return getValueByPath(runtimeConfig, path, defaultValue);
}

/**
 * Get value from object by dot-separated path
 */
function getValueByPath(obj, path, defaultValue) {
  const keys = path.split('.');
  let value = obj;
  
  for (const key of keys) {
    if (value && typeof value === 'object' && key in value) {
      value = value[key];
    } else {
      return defaultValue;
    }
  }
  
  return value;
}

/**
 * Get entire configuration section
 */
export function getConfigSection(section) {
  if (!runtimeConfig) {
    return LOCAL_DEFAULTS[section] || {};
  }
  return runtimeConfig[section] || LOCAL_DEFAULTS[section] || {};
}

/**
 * Force refresh configuration
 */
export async function forceRefresh() {
  await refreshConfigInBackground();
  return runtimeConfig;
}

/**
 * Check if a feature is enabled
 */
export function isFeatureEnabled(feature) {
  return getConfigValue(`feature_flags.${feature}`, true);
}

/**
 * Check if maintenance mode is active
 */
export function isMaintenanceMode() {
  return getConfigValue('feature_flags.maintenance_mode', false);
}

/**
 * Get active announcements (not expired, not dismissed)
 */
export async function getActiveAnnouncements() {
  const announcements = getConfigValue('announcements', []);
  const now = new Date();
  
  // Get dismissed announcements
  const { dismissed_announcements = [] } = await chrome.storage.local.get('dismissed_announcements');
  
  return announcements.filter(a => {
    // Check if expired
    if (a.expires_at && new Date(a.expires_at) < now) {
      return false;
    }
    // Check if dismissed
    if (dismissed_announcements.includes(a.id)) {
      return false;
    }
    return true;
  });
}

/**
 * Dismiss an announcement
 */
export async function dismissAnnouncement(announcementId) {
  const { dismissed_announcements = [] } = await chrome.storage.local.get('dismissed_announcements');
  
  if (!dismissed_announcements.includes(announcementId)) {
    await chrome.storage.local.set({
      dismissed_announcements: [...dismissed_announcements, announcementId]
    });
  }
}

/**
 * Check version compatibility
 */
export function checkVersionCompatibility() {
  const currentVersion = CONFIG.VERSION || '1.0.0';
  const minVersion = getConfigValue('version_control.min_supported_version', '1.0.0');
  const latestVersion = getConfigValue('version_control.latest_version');
  const forceUpdate = getConfigValue('version_control.force_update', false);
  const deprecationWarning = getConfigValue('version_control.deprecation_warning');
  
  const isOutdated = compareVersions(currentVersion, minVersion) < 0;
  const hasUpdate = latestVersion && compareVersions(currentVersion, latestVersion) < 0;
  
  // Ensure updateUrl is a string (could be object from remote config for i18n)
  let updateUrl = getConfigValue('version_control.update_url');
  if (typeof updateUrl === 'object' && updateUrl !== null) {
    // If it's an i18n object, get English or first available
    updateUrl = updateUrl.en || updateUrl['en-US'] || Object.values(updateUrl)[0] || null;
  }
  if (typeof updateUrl !== 'string') {
    updateUrl = null;
  }
  
  return {
    currentVersion,
    minVersion,
    latestVersion,
    isOutdated,
    hasUpdate,
    forceUpdate: forceUpdate && isOutdated,
    deprecationWarning: isOutdated ? deprecationWarning : null,
    updateUrl
  };
}

// ============== CONVENIENCE EXPORTS ==============

export const RemoteConfig = {
  init: initRemoteConfig,
  get: getConfigValue,
  getSection: getConfigSection,
  refresh: forceRefresh,
  rollback: rollbackConfig,
  
  // Feature checks
  isFeatureEnabled,
  isMaintenanceMode,
  isOCREnabled: () => isFeatureEnabled('ocr_enabled'),
  isVideoExtractionEnabled: () => isFeatureEnabled('video_extraction_enabled'),
  isCacheEnabled: () => isFeatureEnabled('cache_enabled'),
  isDebugMode: () => isFeatureEnabled('debug_mode'),
  
  // Section getters
  getRateLimits: () => getConfigSection('rate_limits'),
  getOCRConfig: () => getConfigSection('ocr_config'),
  getTimeouts: () => getConfigSection('timeouts'),
  getFeatureFlags: () => getConfigSection('feature_flags'),
  getVersionControl: () => getConfigSection('version_control'),
  getFieldMetadata: () => getConfigSection('field_metadata'),
  getVideoExtractionConfig: () => getConfigSection('video_extraction'),
  
  // Announcements
  getAnnouncements: getActiveAnnouncements,
  dismissAnnouncement,
  
  // Version
  checkVersion: checkVersionCompatibility,
  
  // State
  isInitialized: () => isInitialized,
  getConfigVersion: () => configVersion
};

export default RemoteConfig;
