﻿
/**
 * Usage Monitor
 * 
 * 监控下载使用量，检测是否接近或超过限制
 * 提供警告和建议，帮助用户避免触发 Google 限制
 */

import { thresholdManager } from './usage-thresholds.js';
import { createLogger, DEBUG_CONFIG } from './debug-config.js';
import { getApiUrl } from './config.js';

// ============== LOGGER ==============
const log = createLogger('[UsageMonitor]', DEBUG_CONFIG.usageMonitor);

// ============== 使用量监控类 ==============

class UsageMonitor {
  constructor() {
    this.hourlyStats = null;
    this.dailyStats = null;
    this.initialized = false;
  }
  
  /**
   * 初始�?
   */
  async init() {
    await this.loadStats();
    this.resetIfNeeded();
    this.initialized = true;
    log.info('Initialized', this.getStats());
  }
  
  /**
   * 加载统计数据
   */
  async loadStats() {
    const data = await chrome.storage.local.get(['hourlyStats', 'dailyStats']);
    
    // 加载每小时统计，并将 domains 数组转回 Set
    if (data.hourlyStats) {
      this.hourlyStats = {
        ...data.hourlyStats,
        domains: new Set(data.hourlyStats.domains || [])
      };
    } else {
      this.hourlyStats = this.createEmptyHourlyStats();
    }
    
    // 加载每天统计，并�?domains 数组转回 Set
    if (data.dailyStats) {
      this.dailyStats = {
        ...data.dailyStats,
        domains: new Set(data.dailyStats.domains || [])
      };
    } else {
      this.dailyStats = this.createEmptyDailyStats();
    }
    
    log.debug('Loaded stats:', { hourly: this.hourlyStats, daily: this.dailyStats });
  }
  
  /**
   * 创建空的每小时统�?
   */
  createEmptyHourlyStats() {
    const now = Date.now();
    return {
      requests: 0,
      ads: 0,
      domains: new Set(),
      startTime: now,
      resetAt: now + 3600000 // 1小时�?
    };
  }
  
  /**
   * 创建空的每天统计
   */
  createEmptyDailyStats() {
    return {
      downloads: 0,
      ads: 0,
      domains: new Set(),
      date: new Date().toDateString()
    };
  }
  
  /**
   * 检查是否需要重置统�?
   */
  async resetIfNeeded() {
    const now = Date.now();
    const today = new Date().toDateString();
    let changed = false;
    
    // 重置每小时统�?
    if (now >= this.hourlyStats.resetAt) {
      log.info('Resetting hourly stats');
      this.hourlyStats = this.createEmptyHourlyStats();
      changed = true;
    }
    
    // 重置每天统计
    if (this.dailyStats.date !== today) {
      log.info('Resetting daily stats (new day)');
      this.dailyStats = this.createEmptyDailyStats();
      changed = true;
    }
    
    if (changed) {
      await this.saveStats();
    }
  }
  
  /**
   * 更新用户计划并调整阈�?
   */
  async updatePlan(plan) {
    if (thresholdManager && thresholdManager.applyPlanMultiplier) {
      thresholdManager.applyPlanMultiplier(plan);
      log.info('Updated plan thresholds:', plan);
    }
  }

  /**
   * 记录一次下�?
   * @param {number} adsCount - 下载的广告数�?
   * @param {string} domain - 域名（可选）
   * @returns {Array} 警告列表
   */
  async recordDownload(adsCount, domain = null) {
    if (!this.initialized) {
      await this.init();
    }
    
    await this.resetIfNeeded();
    
    // 更新每小时统�?
    this.hourlyStats.requests++;
    this.hourlyStats.ads += adsCount;
    if (domain) {
      this.hourlyStats.domains.add(domain);
    }
    
    // 更新每天统计
    this.dailyStats.downloads++;
    this.dailyStats.ads += adsCount;
    if (domain) {
      this.dailyStats.domains.add(domain);
    }
    
    log.info('Recorded download:', {
      adsCount,
      domain,
      hourly: { requests: this.hourlyStats.requests, ads: this.hourlyStats.ads },
      daily: { downloads: this.dailyStats.downloads, ads: this.dailyStats.ads }
    });
    
    // 保存统计
    await this.saveStats();
    
    // 检查阈值并生成警告
    const warnings = this.checkThresholds();
    
    if (warnings.length > 0) {
      log.warn('Warnings generated:', warnings);
    }
    
    return warnings;
  }
  
  /**
   * 同步到服务器
   */
  async syncToServer() {
    try {
      // 获取认证token
      const authData = await chrome.storage.local.get('authToken');
      if (!authData.authToken) {
        log.info('[UsageMonitor] No auth token, skipping server sync');
        return false;
      }
      
      // 准备同步数据
      const syncData = {
        hourly: {
          requests: this.hourlyStats.requests,
          ads: this.hourlyStats.ads,
          domains: this.hourlyStats.domains.size, // Set 的大�?
          domainsList: Array.from(this.hourlyStats.domains), // Set 转数�?
          resetAt: this.hourlyStats.resetAt
        },
        daily: {
          downloads: this.dailyStats.downloads,
          ads: this.dailyStats.ads,
          domains: this.dailyStats.domains.size, // Set 的大�?
          domainsList: Array.from(this.dailyStats.domains), // Set 转数�?
          date: this.dailyStats.date
        }
      };
      
      // 发送到服务�?
      const BACKEND_API = getApiUrl();
      const response = await fetch(`${BACKEND_API}/api/usage/sync`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${authData.authToken}`
        },
        body: JSON.stringify(syncData)
      });
      
      if (response.ok) {
        const result = await response.json();
        log.info('[UsageMonitor] Synced to server:', result);
        return true;
      } else {
        log.warn('[UsageMonitor] Server sync failed:', response.status);
        return false;
      }
    } catch (error) {
      log.error('[UsageMonitor] Server sync error:', error);
      return false;
    }
  }
  
  /**
   * 从服务器恢复统计
   */
  async restoreFromServer() {
    try {
      // 获取认证token
      const authData = await chrome.storage.local.get('authToken');
      if (!authData.authToken) {
        log.info('[UsageMonitor] No auth token, skipping server restore');
        return false;
      }
      
      // 从服务器获取
      const BACKEND_API = getApiUrl();
      const response = await fetch(`${BACKEND_API}/api/usage/stats`, {
        headers: {
          'Authorization': `Bearer ${authData.authToken}`
        }
      });
      
      if (response.ok) {
        const result = await response.json();
        if (result.success && result.data) {
          const { hourly, daily } = result.data;
          
          // 恢复数据
          this.hourlyStats = {
            requests: hourly.requests || 0,
            ads: hourly.ads || 0,
            domains: new Set(hourly.domainsList || []),
            startTime: hourly.startTime || Date.now(),
            resetAt: hourly.resetAt || Date.now() + 3600000
          };
          
          this.dailyStats = {
            downloads: daily.downloads || 0,
            ads: daily.ads || 0,
            domains: new Set(daily.domainsList || []),
            date: daily.date ? new Date(daily.date).toDateString() : new Date().toDateString()
          };
          
          log.info('[UsageMonitor] Restored from server:', result.data);
          
          // 保存到本�?
          await this.saveStats();
          return true;
        }
      }
      
      return false;
    } catch (error) {
      log.error('[UsageMonitor] Server restore error:', error);
      return false;
    }
  }
  
  /**
   * 保存统计数据
   */
  async saveStats() {
    // 转换 Set �?Array 以便存储
    const hourlyToSave = {
      ...this.hourlyStats,
      domains: Array.from(this.hourlyStats.domains)
    };
    
    const dailyToSave = {
      ...this.dailyStats,
      domains: Array.from(this.dailyStats.domains)
    };
    
    await chrome.storage.local.set({
      hourlyStats: hourlyToSave,
      dailyStats: dailyToSave
    });
    
    log.debug('Stats saved');
  }
  
  /**
   * 检查阈值并生成警告
   * @returns {Array} 警告对象数组
   */
  checkThresholds() {
    const warnings = [];
    const thresholds = thresholdManager.getThresholds();
    
    // 检查每小时限制
    const hourlyLevel = thresholdManager.checkLevel('hourly', 'ads', this.hourlyStats.ads);
    
    if (hourlyLevel === 'danger') {
      warnings.push({
        level: 'danger',
        type: 'hourly',
        title: '🔴 Rate Limit Warning',
        message: `You have downloaded ${this.hourlyStats.ads.toLocaleString()} ads in the past hour. Google may temporarily restrict access.`,
        suggestions: [
          'Stop downloading for 30-60 minutes',
          'Reduce download frequency',
          'Spread downloads throughout the day',
          'Contact support if you need higher limits'
        ],
        stats: {
          current: this.hourlyStats.ads,
          threshold: thresholds.hourly.danger.ads,
          percentage: thresholdManager.getProgress('hourly', 'ads', this.hourlyStats.ads)
        }
      });
    } else if (hourlyLevel === 'warning') {
      warnings.push({
        level: 'warning',
        type: 'hourly',
        title: '⚠️ High Usage Detected',
        message: `You have downloaded ${this.hourlyStats.ads.toLocaleString()} ads in the past hour. Consider slowing down.`,
        suggestions: [
          'Wait 10-15 minutes before next download',
          'Reduce pages per download',
          'Take a short break'
        ],
        stats: {
          current: this.hourlyStats.ads,
          threshold: thresholds.hourly.warning.ads,
          percentage: thresholdManager.getProgress('hourly', 'ads', this.hourlyStats.ads)
        }
      });
    }
    
    // 检查每天限�?
    const dailyLevel = thresholdManager.checkLevel('daily', 'ads', this.dailyStats.ads);
    
    if (dailyLevel === 'danger') {
      warnings.push({
        level: 'danger',
        type: 'daily',
        title: '🔴 Daily Limit Exceeded',
        message: `You have downloaded ${this.dailyStats.ads.toLocaleString()} ads today. This may trigger Google rate limits.`,
        suggestions: [
          'Stop downloading for today',
          'Resume tomorrow',
          'Consider spreading downloads across multiple days',
          'Upgrade to Enterprise for higher limits'
        ],
        stats: {
          current: this.dailyStats.ads,
          threshold: thresholds.daily.danger.ads,
          percentage: thresholdManager.getProgress('daily', 'ads', this.dailyStats.ads)
        }
      });
    } else if (dailyLevel === 'warning') {
      warnings.push({
        level: 'warning',
        type: 'daily',
        title: '⚠️ Daily Limit Approaching',
        message: `You have downloaded ${this.dailyStats.ads.toLocaleString()} ads today. You are approaching the safe daily limit.`,
        suggestions: [
          'Slow down download frequency',
          'Consider continuing tomorrow',
          'Monitor your usage in Settings'
        ],
        stats: {
          current: this.dailyStats.ads,
          threshold: thresholds.daily.warning.ads,
          percentage: thresholdManager.getProgress('daily', 'ads', this.dailyStats.ads)
        }
      });
    }
    
    return warnings;
  }
  
  /**
   * 获取当前统计数据
   */
  getStats() {
    // Ensure stats are initialized before accessing
    if (!this.hourlyStats) {
      this.hourlyStats = this.createEmptyHourlyStats();
    }
    if (!this.dailyStats) {
      this.dailyStats = this.createEmptyDailyStats();
    }
    
    return {
      hourly: {
        requests: this.hourlyStats.requests || 0,
        ads: this.hourlyStats.ads || 0,
        domains: this.hourlyStats.domains?.size || this.hourlyStats.domains?.length || 0,
        resetIn: Math.max(0, (this.hourlyStats.resetAt || Date.now()) - Date.now()),
        level: thresholdManager.checkLevel('hourly', 'ads', this.hourlyStats.ads || 0),
        progress: thresholdManager.getProgress('hourly', 'ads', this.hourlyStats.ads || 0)
      },
      daily: {
        downloads: this.dailyStats.downloads || 0,
        ads: this.dailyStats.ads || 0,
        domains: this.dailyStats.domains?.size || this.dailyStats.domains?.length || 0,
        date: this.dailyStats.date || new Date().toDateString(),
        level: thresholdManager.checkLevel('daily', 'ads', this.dailyStats.ads || 0),
        progress: thresholdManager.getProgress('daily', 'ads', this.dailyStats.ads || 0)
      }
    };
  }
  
  /**
   * 获取详细报告（用于设置页面）
   */
  getDetailedReport() {
    const stats = this.getStats();
    const thresholds = thresholdManager.getThresholds();
    
    return {
      stats,
      thresholds,
      recommendations: this.getRecommendations(stats)
    };
  }
  
  /**
   * 根据当前使用情况生成建议
   */
  getRecommendations(stats) {
    const recommendations = [];
    
    // 每小时建�?
    if (stats.hourly.level === 'safe') {
      recommendations.push({
        type: 'hourly',
        level: 'safe',
        message: '�?Your hourly usage is within safe limits'
      });
    } else if (stats.hourly.level === 'warning') {
      recommendations.push({
        type: 'hourly',
        level: 'warning',
        message: '⚠️ Consider slowing down to avoid rate limits'
      });
    } else {
      recommendations.push({
        type: 'hourly',
        level: 'danger',
        message: '🔴 Take a break to avoid Google restrictions'
      });
    }
    
    // 每天建议
    if (stats.daily.level === 'safe') {
      recommendations.push({
        type: 'daily',
        level: 'safe',
        message: '�?Your daily usage is within safe limits'
      });
    } else if (stats.daily.level === 'warning') {
      recommendations.push({
        type: 'daily',
        level: 'warning',
        message: '⚠️ Approaching daily limit, consider continuing tomorrow'
      });
    } else {
      recommendations.push({
        type: 'daily',
        level: 'danger',
        message: '🔴 Daily limit exceeded, resume tomorrow'
      });
    }
    
    return recommendations;
  }
  
  /**
   * 重置统计（用于测试）
   */
  async reset() {
    this.hourlyStats = this.createEmptyHourlyStats();
    this.dailyStats = this.createEmptyDailyStats();
    await this.saveStats();
    log.info('Stats reset');
  }
  
  /**
   * 手动触发检查（用于测试�?
   */
  async checkNow() {
    this.resetIfNeeded();
    return this.checkThresholds();
  }
}

// ============== 导出单例 ==============

export const usageMonitor = new UsageMonitor();

// ============== 使用示例 ==============

/*
// 1. 初始化（�?Service Worker 启动时）
import { usageMonitor } from './usage-monitor.js';
await usageMonitor.init();

// 2. 记录下载
const warnings = await usageMonitor.recordDownload(1000, 'example.com');
if (warnings.length > 0) {
  // 显示警告
  warnings.forEach(warning => {
    log.info(warning.title, warning.message);
  });
}

// 3. 获取当前统计
const stats = usageMonitor.getStats();
log.info('Hourly:', stats.hourly.ads, 'ads');
log.info('Daily:', stats.daily.ads, 'ads');

// 4. 获取详细报告（用于设置页面）
const report = usageMonitor.getDetailedReport();
log.info(report);

// 5. 重置统计（用于测试）
await usageMonitor.reset();
*/
