/**
 * Offscreen Document for OCR Processing
 * v1.2.1: Using OCR.space API with parallel batch processing
 */

// Simple logger for offscreen document
const log = {
  info: (...args) => console.log('[OCR Offscreen]', ...args),
  warn: (...args) => console.warn('[OCR Offscreen]', ...args),
  error: (...args) => console.error('[OCR Offscreen]', ...args),
  debug: (...args) => console.log('[OCR Offscreen][DEBUG]', ...args)
};
const DOMAIN_REGEX = /(?:https?:\/\/)?(?:www\.)?([a-zA-Z0-9][-a-zA-Z0-9]*(?:\.[a-zA-Z0-9][-a-zA-Z0-9]*)+)/gi;

// OCR API 配置（从后端下发�?
let ocrApiConfig = null;

/**
 * Send log to service worker
 */
function sendLog(level, ...args) {
  const message = args.map(a => typeof a === 'object' ? JSON.stringify(a) : String(a)).join(' ');
  log.info(`[OCR Offscreen][${level}]`, ...args);
  chrome.runtime.sendMessage({ type: 'OCR_LOG', level, message }).catch(() => {});
}

/**
 * Perform OCR using OCR.space API
 */
async function performOCRSpaceAPI(imageUrl) {
  // 检�?API 配置
  if (!ocrApiConfig || !ocrApiConfig.api_key) {
    throw new Error('OCR API key not configured');
  }
  
  sendLog('INFO', 'Using OCR.space API for:', imageUrl);
  
  const response = await fetch(imageUrl);
  if (!response.ok) throw new Error(`Failed to fetch image: ${response.status}`);
  
  const blob = await response.blob();
  
  let ext = 'png';
  if (blob.type === 'image/jpeg' || blob.type === 'image/jpg') ext = 'jpg';
  else if (blob.type === 'image/gif') ext = 'gif';
  else if (blob.type === 'image/webp') ext = 'webp';
  
  const arrayBuffer = await blob.arrayBuffer();
  const uint8Array = new Uint8Array(arrayBuffer);
  let binary = '';
  for (let i = 0; i < uint8Array.length; i++) {
    binary += String.fromCharCode(uint8Array[i]);
  }
  const base64Image = `data:${blob.type};base64,${btoa(binary)}`;
  
  const formData = new FormData();
  formData.append('base64Image', base64Image);
  formData.append('filetype', ext.toUpperCase());
  formData.append('language', ocrApiConfig.languages?.[0] || 'eng');
  formData.append('isOverlayRequired', 'false');
  formData.append('detectOrientation', 'true');
  formData.append('scale', 'true');
  formData.append('OCREngine', String(ocrApiConfig.engine || 1));
  
  const apiEndpoint = ocrApiConfig.api_endpoint || 'https://api.ocr.space/parse/image';
  const ocrResponse = await fetch(apiEndpoint, {
    method: 'POST',
    headers: { 'apikey': ocrApiConfig.api_key },
    body: formData
  });
  
  const ocrResult = await ocrResponse.json();
  
  if (ocrResult.ParsedResults?.length > 0) {
    const parsed = ocrResult.ParsedResults[0];
    return {
      text: parsed.ParsedText || '',
      confidence: parsed.ParsedText?.trim() ? 80 : 0
    };
  }
  
  // 检查错�?
  if (ocrResult.ErrorMessage) {
    sendLog('WARN', 'OCR.space API error:', ocrResult.ErrorMessage);
  }
  
  return { text: '', confidence: 0 };
}

/**
 * Perform OCR (auto-selects method)
 */
async function performOCR(imageUrl) {
  const result = {
    ad_domain: '',
    ad_title: '',
    ad_description: '',
    confidence: 0,
    raw_text: ''
  };
  
  try {
    if (!imageUrl?.startsWith('http')) {
      sendLog('WARN', 'Invalid image URL:', imageUrl);
      return result;
    }
    
    sendLog('INFO', 'Processing image:', imageUrl);
    
    // v1.2.1: Direct OCR.space API call (sandbox removed)
    const ocrData = await performOCRSpaceAPI(imageUrl);
    
    sendLog('INFO', 'OCR result:', {
      hasText: !!ocrData.text,
      textLength: ocrData.text?.length || 0,
      confidence: ocrData.confidence,
      textPreview: ocrData.text?.substring(0, 200) || '(empty)'
    });
    
    if (ocrData.text) {
      result.raw_text = ocrData.text.trim();
      result.confidence = ocrData.confidence / 100;
      
      const parsed = parseOCRText(ocrData.text);
      result.ad_domain = parsed.domain;
      result.ad_title = parsed.title;
      result.ad_description = parsed.description;
      
      sendLog('SUCCESS', 'OCR completed, confidence:', result.confidence.toFixed(2));
    } else {
      sendLog('WARN', 'No text found in image');
    }
    
  } catch (e) {
    sendLog('ERROR', 'OCR failed:', String(e));
  }
  
  return result;
}

/**
 * Parse OCR text
 */
function parseOCRText(text) {
  const result = { domain: '', title: '', description: '' };
  if (!text) return result;
  
  const AD_LABELS = [
    'sponsored', 'ad', 'advertisement', 'promoted', '广告', '推广',
    'learn more', 'shop now', 'buy now', 'sign up', 'get started',
    'click here', 'download', 'install', 'subscribe'
  ];
  
  const lines = text.split(/[\n\r]+/).map(l => l.trim()).filter(l => l.length > 2);
  
  for (const line of lines) {
    const domainMatch = line.match(DOMAIN_REGEX);
    if (domainMatch) {
      const domain = domainMatch[0]
        .replace(/^https?:\/\//i, '')
        .replace(/^www\./i, '')
        .split('/')[0]
        .toLowerCase();
      
      if (domain && !domain.includes('google') && !domain.includes('gstatic') && domain.includes('.')) {
        result.domain = domain;
        break;
      }
    }
  }
  
  const textLines = lines.filter(l => {
    const lower = l.toLowerCase();
    if (DOMAIN_REGEX.test(l)) return false;
    if (/^[0-9\s\-\.\,\!\?\@\#\$\%\^\&\*\(\)]+$/.test(l)) return false;
    if (l.length < 3) return false;
    if (AD_LABELS.some(label => lower === label)) return false;
    return true;
  });
  
  let titleIndex = 0;
  if (result.domain) {
    const brandName = result.domain.split('.')[0].toLowerCase();
    for (let i = 0; i < textLines.length; i++) {
      if (textLines[i].toLowerCase().includes(brandName) && textLines[i].length > 10) {
        titleIndex = i;
        break;
      }
    }
  }
  
  if (textLines.length > titleIndex) {
    result.title = textLines[titleIndex].substring(0, 200);
  }
  
  const descLines = textLines.filter((_, i) => i !== titleIndex);
  if (descLines.length > 0) {
    result.description = descLines.join(' ').substring(0, 500);
  }
  
  return result;
}

// v1.2.1: OCR 并发配置（从消息中获取或使用默认值）
let ocrConcurrencyConfig = {
  concurrent: 2,           // 默认并发�?
  max_retries: 2,          // 最大重试次�?
  retry_delay_ms: 1500,    // 重试延迟
  request_delay_ms: 500    // 请求间延�?
};

/**
 * Sleep helper
 */
function sleep(ms) {
  return new Promise(resolve => setTimeout(resolve, ms));
}

/**
 * Perform OCR with retry logic
 * @param {string} imageUrl - Image URL
 * @param {number} retryCount - Current retry count
 */
async function performOCRWithRetry(imageUrl, retryCount = 0) {
  try {
    return await performOCR(imageUrl);
  } catch (e) {
    const maxRetries = ocrConcurrencyConfig.max_retries;
    const retryDelay = ocrConcurrencyConfig.retry_delay_ms;
    
    // Check if it's a rate limit error (429) or temporary failure
    const isRetryable = e.message?.includes('429') || 
                        e.message?.includes('rate') ||
                        e.message?.includes('timeout') ||
                        e.message?.includes('Failed to fetch');
    
    if (isRetryable && retryCount < maxRetries) {
      sendLog('WARN', `OCR failed, retrying (${retryCount + 1}/${maxRetries}) after ${retryDelay}ms:`, e.message);
      await sleep(retryDelay * (retryCount + 1)); // Exponential backoff
      return performOCRWithRetry(imageUrl, retryCount + 1);
    }
    
    // Return empty result on final failure
    sendLog('ERROR', 'OCR failed after retries:', e.message);
    return {
      ad_domain: '',
      ad_title: '',
      ad_description: '',
      confidence: 0,
      raw_text: '',
      error: e.message
    };
  }
}

/**
 * Process multiple ads with parallel execution
 * v1.2.1: Supports configurable concurrency with retry
 */
async function processAds(ads) {
  const concurrent = ocrConcurrencyConfig.concurrent || 2;
  const requestDelay = ocrConcurrencyConfig.request_delay_ms || 500;
  
  sendLog('INFO', `processAds called with ${ads.length} ads, concurrency: ${concurrent}`);
  
  const results = [];
  let processedCount = 0;
  
  // Process in batches
  for (let i = 0; i < ads.length; i += concurrent) {
    const batch = ads.slice(i, Math.min(i + concurrent, ads.length));
    const batchStartIndex = i;
    
    sendLog('INFO', `Processing batch ${Math.floor(i / concurrent) + 1}, items ${i + 1}-${i + batch.length}`);
    
    // Process batch in parallel
    const batchPromises = batch.map(async (ad, batchIdx) => {
      const globalIndex = batchStartIndex + batchIdx;
      sendLog('INFO', `[${globalIndex + 1}/${ads.length}] Processing:`, ad.image_url?.substring(0, 80));
      
      const ocrResult = await performOCRWithRetry(ad.image_url);
      return { creative_id: ad.creative_id, ...ocrResult };
    });
    
    // Wait for all in batch to complete
    const batchResults = await Promise.all(batchPromises);
    results.push(...batchResults);
    
    // Update progress
    processedCount += batch.length;
    chrome.runtime.sendMessage({
      type: 'OCR_PROGRESS_FROM_OFFSCREEN',
      processed: processedCount,
      total: ads.length
    }).catch(() => {});
    
    sendLog('SUCCESS', `Batch complete: ${processedCount}/${ads.length} processed`);
    
    // Delay between batches to avoid rate limiting
    if (i + concurrent < ads.length) {
      await sleep(requestDelay);
    }
  }
  
  sendLog('SUCCESS', `All OCR complete: ${results.length} results`);
  return results;
}

async function terminate() {
  log.info('[OCR Offscreen] Cleanup complete');
}

// Startup
log.info('[OCR Offscreen] Script loaded');

chrome.runtime.sendMessage({
  type: 'OCR_LOG',
  level: 'INFO',
  message: 'Offscreen started. Sandbox OCR with API fallback.'
}).catch(() => {});

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  log.info('[OCR Offscreen] Received message:', message.type);
  
  // v1.1.0: 配置 OCR API
  if (message.type === 'OCR_SET_CONFIG') {
    ocrApiConfig = message.config;
    sendLog('INFO', 'OCR API config updated, hasKey:', !!ocrApiConfig?.api_key);
    sendResponse({ success: true });
    return true;
  }
  
  if (message.type === 'OCR_PROCESS_ADS') {
    // v1.1.0: 支持通过消息传�?API 配置
    if (message.ocrConfig) {
      ocrApiConfig = message.ocrConfig;
    }
    // v1.2.1: 支持并发配置
    if (message.concurrencyConfig) {
      Object.assign(ocrConcurrencyConfig, message.concurrencyConfig);
      sendLog('INFO', 'Concurrency config updated:', ocrConcurrencyConfig);
    }
    sendLog('INFO', 'Processing OCR request...');
    processAds(message.ads).then(results => {
      sendResponse({ success: true, results });
      terminate();
    }).catch(err => {
      sendLog('ERROR', 'processAds error:', err.message);
      sendResponse({ success: false, error: err.message });
      terminate();
    });
    return true;
  }
  
  if (message.type === 'OCR_SINGLE') {
    // v1.1.0: 支持通过消息传�?API 配置
    if (message.ocrConfig) {
      ocrApiConfig = message.ocrConfig;
    }
    performOCR(message.imageUrl).then(result => {
      sendResponse({ success: true, result });
    }).catch(err => {
      sendResponse({ success: false, error: err.message });
    });
    return true;
  }
  
  if (message.type === 'OCR_TERMINATE') {
    terminate().then(() => sendResponse({ success: true }));
    return true;
  }
});

log.info('[OCR Offscreen] Message listener registered');
