/**
 * Ads Insight Pro - Popup UI Controller
 * v1.1.0: Server-driven permissions
 */

import { TIME, API, UI, timestampToDaysAgo, PLAN_LIMITS } from '../lib/constants.js';
import { permissionsManager } from '../lib/permissions.js';
import { initWarningListener, showWarningBanner } from './usage-warning-ui.js';
import { RemoteConfig, initRemoteConfig } from '../lib/remote-config.js';
import { createLogger, DEBUG_CONFIG } from '../lib/debug-config.js';

// Logger
const log = createLogger('[Popup]', DEBUG_CONFIG.popup);

// State
let state = {
  isOnPage: false,
  searchMode: null,
  domain: null,
  advertiserId: null,
  adsCount: 0,
  // 🆕 Brand mode fields
  brandName: null,
  advertiserIds: null,
  // 🆕 Single ad mode fields
  creativeId: null,
  // Download state
  isDownloading: false,
  isPaused: false,
  currentPage: 0,
  totalPages: 0,
  fetchedAds: [],
  // v1.1.0: Server-driven permissions
  userPlan: 'free',
  dailyUsed: 0,
  dailyLimit: 100,     // Will be updated from server
  maxPages: 5,         // Will be updated from server
  perRequestLimit: 100, // Will be updated from server
  // v1.2.1: OCR quota
  ocrDailyUsed: 0,
  ocrDailyLimit: 10,
  ocrDailyRemaining: 10
};

// DOM Elements
const elements = {
  statusBadge: document.getElementById('status-badge'),
  statusText: document.getElementById('status-text'),
  pageDetails: document.getElementById('page-details'),
  domainLabel: document.getElementById('domain-label'),
  currentDomain: document.getElementById('current-domain'),
  adsCount: document.getElementById('ads-count'),
  downloadSettings: document.getElementById('download-settings'),
  actionSection: document.getElementById('action-section'),
  progressSection: document.getElementById('progress-section'),
  completeSection: document.getElementById('complete-section'),
  errorSection: document.getElementById('error-section'),
  notOnPage: document.getElementById('not-on-page'),
  pagesInput: document.getElementById('pages'),
  pagesInfo: document.getElementById('pages-info'),
  formatSelect: document.getElementById('format'),
  exportFormat: document.getElementById('export-format'),
  fetchVideoLinks: document.getElementById('fetch-video-links'),
  videoLinksNote: document.getElementById('video-links-note'),
  videoProgress: document.getElementById('video-progress'),
  videoProgressText: document.getElementById('video-progress-text'),
  videoProgressFill: document.getElementById('video-progress-fill'),
  ocrProgress: document.getElementById('ocr-progress'),
  ocrProgressText: document.getElementById('ocr-progress-text'),
  ocrProgressFill: document.getElementById('ocr-progress-fill'),
  mainProgressInfo: document.getElementById('main-progress-info'),
  mainProgressBar: document.getElementById('main-progress-bar'),
  btnDownload: document.getElementById('btn-download'),
  btnPause: document.getElementById('btn-pause'),
  btnCancel: document.getElementById('btn-cancel'),
  progressText: document.getElementById('progress-text'),
  progressPercent: document.getElementById('progress-percent'),
  progressFill: document.getElementById('progress-fill'),
  fetchedCount: document.getElementById('fetched-count'),
  advertisersCount: document.getElementById('advertisers-count'),
  totalAds: document.getElementById('total-ads'),
  uniqueAdvertisers: document.getElementById('unique-advertisers'),
  duplicatesRemoved: document.getElementById('duplicates-removed'),
  usageInfo: document.getElementById('usage-info'),
  upgradeLink: document.getElementById('upgrade-link'),
  // Current progress module elements
  currentProgressSection: document.getElementById('current-progress-section'),
  currentProgressFill: document.getElementById('current-progress-fill'),
  currentProgressText: document.getElementById('current-progress-text'),
  currentProgressAds: document.getElementById('current-progress-ads'),
  currentVideoProgress: document.getElementById('current-video-progress'),
  currentVideoText: document.getElementById('current-video-text'),
  currentVideoFill: document.getElementById('current-video-fill'),
  btnViewProgress: document.getElementById('btn-view-progress'),
  // Upgrade modal elements
  upgradeModal: document.getElementById('upgrade-modal'),
  modalIcon: document.getElementById('modal-icon'),
  modalTitle: document.getElementById('modal-title'),
  modalMessage: document.getElementById('modal-message'),
  btnUpgrade: document.getElementById('btn-upgrade'),
  btnModalCancel: document.getElementById('btn-modal-cancel'),
  // Feature labels
  labelFetchVideoLinks: document.getElementById('label-fetch-video-links'),
  labelIncludeImages: document.getElementById('label-include-images'),
  labelIncludePreview: document.getElementById('label-include-preview'),
  labelIncludeAdContent: document.getElementById('label-include-ad-content'),
  includeImages: document.getElementById('include-images'),
  includePreview: document.getElementById('include-preview'),
  includeAdContent: document.getElementById('include-ad-content'),
  adContentNote: document.getElementById('ad-content-note')
};

// Initialize
document.addEventListener('DOMContentLoaded', init);

async function init() {
  // Check if onboarding needed
  const onboardingNeeded = await checkOnboarding();
  if (onboardingNeeded) {
    window.location.href = 'onboarding.html';
    return;
  }
  
  // Check login status
  const isLoggedIn = await checkLoginStatus();
  if (!isLoggedIn) {
    // Not logged in, redirect to account page
    window.location.href = 'account.html';
    return;
  }
  
  // v1.2.0: Initialize remote configuration
  try {
    await initRemoteConfig();
    log.info('RemoteConfig initialized, version:', RemoteConfig.getConfigVersion());
  } catch (e) {
    log.warn('RemoteConfig init failed:', e.message);
  }
  
  // v1.2.0: Check maintenance mode
  if (RemoteConfig.isMaintenanceMode()) {
    showMaintenanceMode();
    return;
  }
  
  // v1.2.0: Check version compatibility
  const versionCheck = RemoteConfig.checkVersion();
  if (versionCheck.forceUpdate) {
    showForceUpdateModal(versionCheck);
    return;
  }
  
  // User is logged in, show the UI
  showMainUI();
  
  await loadUserState();
  await checkCurrentPage();
  
  // 恢复保存的进度（如果有）
  await restoreProgress();
  
  setupEventListeners();
  updateUI();
  
  // Initialize usage warning listener
  initWarningListener();
  
  // v1.2.0: Show announcements and update notification
  await showAnnouncements();
  if (versionCheck.hasUpdate) {
    showUpdateAvailable(versionCheck);
  }
  
  // Check for rating prompt after successful downloads
  await checkRatingPrompt();
}

// Show main UI after login check
function showMainUI() {
  const loadingScreen = document.getElementById('loading-screen');
  const header = document.querySelector('.header');
  const mainContent = document.getElementById('main-content');
  
  if (loadingScreen) loadingScreen.style.display = 'none';
  if (header) header.style.display = 'flex';
  if (mainContent) mainContent.style.display = 'block';
}

// Check if onboarding is needed
async function checkOnboarding() {
  try {
    const data = await chrome.storage.local.get('onboardingComplete');
    return !data.onboardingComplete;
  } catch (e) {
    return false;
  }
}

// Check if user is logged in
async function checkLoginStatus() {
  try {
    const data = await chrome.storage.local.get(['authToken', 'authExpires']);
    
    // Check if token exists and not expired
    if (data.authToken && data.authExpires > Date.now()) {
      return true;
    }
    
    return false;
  } catch (e) {
    log.error('Login check failed:', e);
    return false;
  }
}

// Load user state from server (v1.1.0: server-driven permissions)
async function loadUserState() {
  try {
    // Load settings from local storage
    const data = await chrome.storage.local.get(['settings']);
    
    // Load default settings and apply to UI
    if (data.settings) {
      if (data.settings.defaultPages) {
        elements.pagesInput.value = data.settings.defaultPages;
      }
      if (data.settings.defaultFormat) {
        elements.exportFormat.value = data.settings.defaultFormat;
      }
    }
    
    // Fetch permissions from server
    await updatePlanLimitsFromServer();
  } catch (e) {
    log.error('Failed to load user state:', e);
    // Use fallback values on error
    updatePlanLimitsFallback();
  }
}

// v1.1.0: Fetch permissions from server
async function updatePlanLimitsFromServer() {
  try {
    const permissions = await permissionsManager.getPermissions();
    
    state.userPlan = permissions.plan;
    state.dailyLimit = permissions.quota.daily_limit;
    state.dailyUsed = permissions.quota.used_today;
    state.maxPages = permissions.limits.max_pages;
    state.perRequestLimit = permissions.quota.per_request_limit;
    
    // v1.2.1: Load OCR quota
    if (permissions.ocr_quota) {
      state.ocrDailyUsed = permissions.ocr_quota.daily_used || 0;
      state.ocrDailyLimit = permissions.ocr_quota.daily_limit || 10;
      state.ocrDailyRemaining = permissions.ocr_quota.daily_remaining || 0;
    }
    
    log.info(`Permissions loaded from server: plan=${state.userPlan}, quota=${state.dailyUsed}/${state.dailyLimit}, ocr=${state.ocrDailyUsed}/${state.ocrDailyLimit}`);
  } catch (e) {
    log.warn('Failed to load permissions from server, using fallback:', e.message);
    updatePlanLimitsFallback();
  }
}

// Fallback to local constants if server unavailable
function updatePlanLimitsFallback() {
  const limits = PLAN_LIMITS[state.userPlan] || PLAN_LIMITS.free;
  state.dailyLimit = limits.dailyQuota;
  state.maxPages = limits.maxPages;
  state.perRequestLimit = limits.perRequestLimit;
  log.info('Using fallback permissions');
}

// Check if we're on Google Ads Transparency page
async function checkCurrentPage() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    
    // 检查是否在 Google Ads Transparency 页面
    if (!tab?.url?.includes('adstransparency.google.com')) {
      state.isOnPage = false;
      return;
    }
    
    // 尝试发送消息到 content script（带重试机制）
    const maxRetries = 3;
    const retryDelay = 500; // ms
    
    for (let attempt = 1; attempt <= maxRetries; attempt++) {
      try {
        // 首先尝试注入 content script（如果还没加载）
        if (attempt > 1) {
          try {
            await chrome.scripting.executeScript({
              target: { tabId: tab.id },
              files: ['content/content.js']
            });
            log.debug(`Injected content script on attempt ${attempt}`);
            await new Promise(r => setTimeout(r, 200)); // 等待脚本初始化
          } catch (injectError) {
            // 忽略注入错误（可能已经加载了）
            log.debug('Content script injection skipped:', injectError.message);
          }
        }
        
        const response = await chrome.tabs.sendMessage(tab.id, { type: 'GET_PAGE_INFO' });
        
        if (response?.success) {
          state.isOnPage = true;
          state.searchMode = response.mode;
          state.domain = response.domain;
          state.advertiserId = response.advertiserId;
          state.adsCount = response.adsCount || 0;
          state.pageRegion = response.region || '';
          // 🆕 Brand mode fields
          state.brandName = response.brandName;
          state.advertiserIds = response.advertiserIds;
          // 🆕 Single ad mode fields
          state.creativeId = response.creativeId;
          state.advertiserName = response.advertiserName;
          log.debug(`Page info received on attempt ${attempt}:`, response.mode);
          return; // 成功，退出
        } else {
          // Content script 响应但没有成功
          state.isOnPage = false;
          return;
        }
      } catch (messageError) {
        log.debug(`Attempt ${attempt}/${maxRetries} failed:`, messageError.message);
        
        if (attempt < maxRetries) {
          // 等待后重试
          await new Promise(r => setTimeout(r, retryDelay));
        } else {
          // 最后一次尝试失败
          log.warn('Content script not ready after all retries');
          state.isOnPage = false;
        }
      }
    }
  } catch (e) {
    // 查询标签页失败
    log.error('Failed to query tab:', e);
    state.isOnPage = false;
  }
}

// Setup event listeners
function setupEventListeners() {
  elements.pagesInput.addEventListener('input', onPagesChange);
  elements.fetchVideoLinks.addEventListener('change', onFetchVideoLinksChange);
  elements.includeAdContent?.addEventListener('change', onIncludeAdContentChange);
  elements.btnDownload.addEventListener('click', startDownload);
  elements.btnPause.addEventListener('click', togglePause);
  elements.btnCancel.addEventListener('click', cancelDownload);
  document.getElementById('btn-open-transparency').addEventListener('click', openTransparencyPage);
  document.getElementById('btn-retry').addEventListener('click', startDownload);
  document.getElementById('btn-dismiss').addEventListener('click', dismissError);
  document.getElementById('btn-download-again').addEventListener('click', resetToStart);
  document.getElementById('btn-settings').addEventListener('click', openSettings);
  document.getElementById('btn-account').addEventListener('click', openAccount);
  
  // View Details 按钮 - 切换到详细进度页面
  if (elements.btnViewProgress) {
    elements.btnViewProgress.addEventListener('click', () => {
      // 隐藏当前进度模块，显示详细进度页面
      elements.currentProgressSection.style.display = 'none';
      hideAllSections();
      elements.progressSection.style.display = 'block';
    });
  }
  
  // Feature click handlers - show upgrade modal for locked features
  elements.labelFetchVideoLinks?.addEventListener('click', (e) => {
    if (elements.fetchVideoLinks.disabled) {
      e.preventDefault();
      handleFeatureClick('fetch-video-links', 'enterprise');
    }
  });
  
  elements.labelIncludeImages?.addEventListener('click', (e) => {
    if (elements.includeImages.disabled) {
      e.preventDefault();
      handleFeatureClick('include-images', 'pro');
    }
  });
  
  elements.labelIncludePreview?.addEventListener('click', (e) => {
    if (elements.includePreview.disabled) {
      e.preventDefault();
      handleFeatureClick('include-preview', 'pro');
    }
  });
  
  // Note: Ad Content Analysis is available to all users with quota limits
  // No upgrade prompt needed - just let them use it
  
  // Export format change - check permissions
  elements.exportFormat?.addEventListener('change', (e) => {
    const selectedOpt = e.target.options[e.target.selectedIndex];
    if (selectedOpt.disabled) {
      e.target.value = 'csv';
      const feature = selectedOpt.value === 'json' ? 'export-json' : 'export-excel';
      handleFeatureClick(feature, 'pro');
    }
  });
  
  // Upgrade modal buttons
  elements.btnModalCancel?.addEventListener('click', hideUpgradeModal);
  elements.upgradeModal?.querySelector('.modal-backdrop')?.addEventListener('click', hideUpgradeModal);
  elements.btnUpgrade?.addEventListener('click', () => {
    hideUpgradeModal();
    // Open pricing page on official website
    chrome.tabs.create({ url: 'https://adsinsightpro.com/pricing' });
  });
}

function onFetchVideoLinksChange() {
  const isChecked = elements.fetchVideoLinks.checked;
  elements.videoLinksNote.style.display = isChecked ? 'block' : 'none';
}

function onIncludeAdContentChange() {
  // v1.2.1: Check OCR quota when user tries to enable
  if (elements.includeAdContent?.checked && state.ocrDailyRemaining <= 0) {
    // Quota exhausted - uncheck and show warning
    elements.includeAdContent.checked = false;
    showWarningBanner({
      level: 'warning',
      type: 'ocr_quota',
      title: 'OCR Quota Exhausted',
      message: `You have used all ${state.ocrDailyLimit} OCR analyses for today. Quota resets at midnight UTC.`,
      suggestions: ['Wait until tomorrow for quota reset', 'Upgrade your plan for higher limits']
    });
  }
}

function onPagesChange() {
  let pages = parseInt(elements.pagesInput.value) || 1;
  pages = Math.min(pages, state.maxPages);
  pages = Math.max(pages, 1);
  elements.pagesInput.value = pages;
  
  const pagesFormGroup = elements.pagesInput?.closest('.form-group');
  const pagesLabel = pagesFormGroup?.querySelector('label');
  const inputWithInfo = pagesFormGroup?.querySelector('.input-with-info');
  
  // 🔧 FIX: Show smarter UI based on actual ads count
  if (state.adsCount > 0) {
    const actualMaxPages = Math.ceil(state.adsCount / API.ESTIMATED_ADS_PER_PAGE);
    
    // If all ads fit in one page, simplify the UI
    if (actualMaxPages <= 1) {
      // Hide pages input, show simple "Download all X ads" message
      if (pagesLabel) pagesLabel.textContent = 'Ads to download:';
      if (inputWithInfo) inputWithInfo.style.display = 'none';
      elements.pagesInfo.style.display = 'inline';
      elements.pagesInfo.textContent = `All ${state.adsCount} ads`;
      elements.pagesInput.value = 1;
      elements.pagesInput.max = 1;
    } else {
      // Multiple pages available, show normal pages selector
      if (pagesLabel) pagesLabel.textContent = 'Pages to download:';
      if (inputWithInfo) inputWithInfo.style.display = 'flex';
      elements.pagesInfo.style.display = '';
      
      const effectiveMaxPages = Math.min(actualMaxPages, state.maxPages);
      elements.pagesInput.max = effectiveMaxPages;
      
      if (pages > effectiveMaxPages) {
        pages = effectiveMaxPages;
        elements.pagesInput.value = pages;
      }
      
      const estimatedAds = Math.min(state.adsCount, pages * API.ESTIMATED_ADS_PER_PAGE);
      if (state.adsCount <= pages * API.ESTIMATED_ADS_PER_PAGE) {
        elements.pagesInfo.textContent = `= all ${estimatedAds} ads`;
      } else {
        elements.pagesInfo.textContent = `= ~${estimatedAds} / ${state.adsCount.toLocaleString()} ads`;
      }
    }
  } else {
    // Unknown ads count - show normal pages selector
    if (pagesLabel) pagesLabel.textContent = 'Pages to download:';
    if (inputWithInfo) inputWithInfo.style.display = 'flex';
    elements.pagesInfo.style.display = '';
    elements.pagesInput.max = state.maxPages;
    elements.pagesInfo.textContent = `= ~${pages * API.ESTIMATED_ADS_PER_PAGE} ads`;
  }
}

// ============== Feature Permissions ==============

/**
 * 根据用户计划更新功能权限
 */
function updateFeaturePermissions() {
  const plan = state.userPlan;
  const isPro = plan === 'pro' || plan === 'enterprise';
  const isEnterprise = plan === 'enterprise';
  
  // 1. Fetch Video Links - Enterprise only
  if (elements.fetchVideoLinks) {
    elements.fetchVideoLinks.disabled = !isEnterprise;
    elements.labelFetchVideoLinks?.classList.toggle('feature-locked', !isEnterprise);
    elements.labelFetchVideoLinks?.classList.toggle('feature-unlocked', isEnterprise);
    // 默认不勾选（即使有权限）
    if (!isEnterprise) {
      elements.fetchVideoLinks.checked = false;
    }
  }
  
  // 2. Include Image URLs - Pro+
  if (elements.includeImages) {
    elements.includeImages.disabled = !isPro;
    elements.labelIncludeImages?.classList.toggle('feature-locked', !isPro);
    elements.labelIncludeImages?.classList.toggle('feature-unlocked', isPro);
    // Pro+ 默认勾选
    if (isPro) {
      elements.includeImages.checked = true;
    }
  }
  
  // 3. Include Preview Links - Pro+
  if (elements.includePreview) {
    elements.includePreview.disabled = !isPro;
    elements.labelIncludePreview?.classList.toggle('feature-locked', !isPro);
    elements.labelIncludePreview?.classList.toggle('feature-unlocked', isPro);
    // Pro+ 默认勾选
    if (isPro) {
      elements.includePreview.checked = true;
    }
  }
  
  // 4. Include Ad Content (OCR) - Available to all users with quota limits
  // Free: 10/day, Pro: 100/day, Enterprise: unlimited
  if (elements.includeAdContent) {
    const ocrQuotaExhausted = state.ocrDailyRemaining <= 0;
    
    // OCR is available to all users, but disabled if quota exhausted
    elements.includeAdContent.disabled = ocrQuotaExhausted;
    elements.labelIncludeAdContent?.classList.toggle('feature-locked', ocrQuotaExhausted);
    elements.labelIncludeAdContent?.classList.toggle('feature-unlocked', !ocrQuotaExhausted);
    
    // Update the note to show quota status - 显示已用/总量
    if (elements.adContentNote) {
      if (ocrQuotaExhausted) {
        elements.adContentNote.innerHTML = `<span style="color: var(--warning-color);">⚠️ Daily quota exhausted (${state.ocrDailyUsed}/${state.ocrDailyLimit}). Resets at midnight UTC.</span>`;
        elements.includeAdContent.checked = false;
      } else {
        elements.adContentNote.innerHTML = `🔍 Extracts ad domain, title and description from image ads <span style="color: var(--text-secondary);">(${state.ocrDailyUsed}/${state.ocrDailyLimit} used today)</span>`;
      }
    }
  }
  
  // 4. Export Format - JSON/Excel for Pro+
  if (elements.exportFormat) {
    elements.exportFormat.querySelectorAll('option').forEach(opt => {
      const minPlan = opt.dataset.minPlan;
      if (minPlan === 'pro') {
        opt.disabled = !isPro;
        // 更新显示文本
        if (isPro) {
          opt.textContent = opt.value === 'json' ? 'JSON' : 'Excel (.xls)';
        } else {
          opt.textContent = opt.value === 'json' ? 'JSON (Pro)' : 'Excel (Pro)';
        }
      }
    });
    // 如果当前选中的是禁用的选项，重置为 CSV
    const selectedOpt = elements.exportFormat.options[elements.exportFormat.selectedIndex];
    if (selectedOpt && selectedOpt.disabled) {
      elements.exportFormat.value = 'csv';
    }
  }
}

/**
 * 显示升级提示模态框
 */
function showUpgradeModal(feature, requiredPlan) {
  const messages = {
    'fetch-video-links': {
      icon: '🎬',
      title: 'Enterprise Feature',
      message: 'Extract YouTube video links from video ads. This powerful feature helps you analyze video ad campaigns.',
      plan: 'Enterprise'
    },
    'include-images': {
      icon: '🖼️',
      title: 'Pro Feature',
      message: 'Include direct image URLs in your export. Download ad creatives for competitive analysis.',
      plan: 'Pro'
    },
    'include-preview': {
      icon: '👁️',
      title: 'Pro Feature',
      message: 'Include ad preview links in your export. View ads exactly as they appear to users.',
      plan: 'Pro'
    },
    'export-json': {
      icon: '📄',
      title: 'Pro Feature',
      message: 'Export data in JSON format for easy integration with your tools and workflows.',
      plan: 'Pro'
    },
    'export-excel': {
      icon: '📊',
      title: 'Pro Feature',
      message: 'Export data in Excel format for advanced analysis and reporting.',
      plan: 'Pro'
    },
    'include-ad-content': {
      icon: '🔍',
      title: 'Pro Feature',
      message: 'Ad Content Analysis extracts domain, title and description from image ads automatically.',
      plan: 'Pro'
    }
  };
  
  const config = messages[feature] || {
    icon: '🔒',
    title: `${requiredPlan} Feature`,
    message: `This feature is available for ${requiredPlan} users.`,
    plan: requiredPlan
  };
  
  elements.modalIcon.textContent = config.icon;
  elements.modalTitle.textContent = config.title;
  elements.modalMessage.textContent = config.message;
  elements.btnUpgrade.textContent = `Upgrade to ${config.plan}`;
  elements.upgradeModal.style.display = 'flex';
}

/**
 * 隐藏升级提示模态框
 */
function hideUpgradeModal() {
  elements.upgradeModal.style.display = 'none';
}

/**
 * 处理功能点击（检查权限）
 */
function handleFeatureClick(feature, requiredPlan) {
  const plan = state.userPlan;
  const planOrder = { free: 0, pro: 1, enterprise: 2 };
  
  if (planOrder[plan] < planOrder[requiredPlan]) {
    showUpgradeModal(feature, requiredPlan);
    return false;
  }
  return true;
}

/**
 * 恢复保存的下载进度 - 在主页面显示"当前进度"模块
 */
async function restoreProgress() {
  try {
    // 从 service worker 获取保存的进度
    const response = await chrome.runtime.sendMessage({ type: 'GET_PROGRESS' });
    
    if (!response || !response.isActive) {
      log.debug('No active progress found');
      // 隐藏当前进度模块
      if (elements.currentProgressSection) {
        elements.currentProgressSection.style.display = 'none';
      }
      return;
    }
    
    log.debug('Progress loaded:', response);
    log.debug('Progress state:', {
      isActive: response.isActive,
      currentPage: response.currentPage,
      totalPages: response.totalPages,
      videoProcessing: response.videoProcessing
    });
    
    // 恢复状态
    state.isDownloading = response.isActive;
    state.currentPage = response.currentPage;
    state.totalPages = response.totalPages;
    state.fetchedAds = Array(response.adsCount).fill(null);
    
    // 显示"当前进度"模块（在主页面）
    if (elements.currentProgressSection) {
      elements.currentProgressSection.style.display = 'block';
      
      // 更新下载进度
      const percent = response.totalPages > 0 ? 
        Math.round((response.currentPage / response.totalPages) * 100) : 0;
      elements.currentProgressFill.style.width = `${percent}%`;
      elements.currentProgressText.textContent = `Page ${response.currentPage}/${response.totalPages}`;
      elements.currentProgressAds.textContent = `${response.adsCount} ads`;
      
      log.debug('Current progress module updated');
      
      // 显示视频处理进度（如果有）
      if (response.videoProcessing && response.videoProcessing.isActive) {
        elements.currentVideoProgress.style.display = 'block';
        const videoPercent = response.videoProcessing.total > 0 ? 
          Math.round((response.videoProcessing.processed / response.videoProcessing.total) * 100) : 0;
        elements.currentVideoText.textContent = 
          `${response.videoProcessing.processed}/${response.videoProcessing.total}`;
        elements.currentVideoFill.style.width = `${videoPercent}%`;
        
        // 如果正在处理视频，还要更新详细进度页面的状态（以防用户点击 View Details）
        if (elements.videoProgress) {
          elements.videoProgress.style.display = 'block';
          elements.videoProgressText.textContent = `${response.videoProcessing.processed}/${response.videoProcessing.total} (${response.videoProcessing.found} found)`;
          elements.videoProgressFill.style.width = `${videoPercent}%`;
        }
        
        // 隐藏详细页面中的主进度条
        if (elements.mainProgressInfo) elements.mainProgressInfo.style.display = 'none';
        if (elements.mainProgressBar) elements.mainProgressBar.style.display = 'none';
        
        log.debug('Video progress updated:', videoPercent, '%');
      } else {
        elements.currentVideoProgress.style.display = 'none';
        // 如果没有处理视频，确保主进度条是可见的（为了下次）
        if (elements.mainProgressInfo) elements.mainProgressInfo.style.display = 'flex'; // flex for info
        if (elements.mainProgressBar) elements.mainProgressBar.style.display = 'block';
        if (elements.videoProgress) elements.videoProgress.style.display = 'none';
      }
    }
    
    // Note: Don't update status badge here - let updateUI() handle it based on state.isOnPage
    // This ensures consistent status display whether user is on the page or not
    
  } catch (error) {
    log.warn('Failed to restore progress:', error);
  }
}

/**
 * 更新当前进度模块（实时更新）
 */
function updateCurrentProgressModule(data) {
  if (!elements.currentProgressSection) return;
  
  // 如果详细进度页面是可见的，不要显示这个小模块
  if (elements.progressSection && elements.progressSection.style.display !== 'none') {
    elements.currentProgressSection.style.display = 'none';
    return;
  }
  
  // 显示模块
  elements.currentProgressSection.style.display = 'block';
  
  if (data.currentPage !== undefined && data.totalPages !== undefined) {
    const percent = data.totalPages > 0 ? 
      Math.round((data.currentPage / data.totalPages) * 100) : 0;
    elements.currentProgressFill.style.width = `${percent}%`;
    elements.currentProgressText.textContent = `Page ${data.currentPage}/${data.totalPages}`;
  }
  
  if (data.adsCount !== undefined) {
    elements.currentProgressAds.textContent = `${data.adsCount} ads`;
  }
}

/**
 * 更新当前视频进度模块
 */
function updateCurrentVideoProgress(processed, total) {
  if (!elements.currentVideoProgress) return;
  
  elements.currentVideoProgress.style.display = 'block';
  const percent = total > 0 ? Math.round((processed / total) * 100) : 0;
  elements.currentVideoText.textContent = `${processed}/${total}`;
  elements.currentVideoFill.style.width = `${percent}%`;
}

// Update UI based on state
function updateUI() {
  // Update usage info (v1.1.0: quota-based, ads/day)
  const limitText = state.dailyLimit === Infinity ? '∞' : state.dailyLimit.toLocaleString();
  elements.usageInfo.textContent = `Today: ${state.dailyUsed.toLocaleString()}/${limitText} ads`;
  elements.upgradeLink.textContent = state.userPlan === 'free' ? 'Free Tier' : 
                                      state.userPlan === 'pro' ? 'Pro' : 'Enterprise';
  
  // Update pages input max
  elements.pagesInput.max = state.maxPages;
  if (parseInt(elements.pagesInput.value) > state.maxPages) {
    elements.pagesInput.value = state.maxPages;
  }
  // 🔧 FIX: Always update pages info to reflect actual ads count
  onPagesChange();
  
  // Update feature permissions based on plan
  updateFeaturePermissions();
  
  // Show appropriate section based on state
  hideAllSections();
  
  if (!state.isOnPage) {
    // Not on Ads Transparency page
    elements.notOnPage.style.display = 'block';
    
    // If download is in progress, show downloading status instead of error
    if (state.isDownloading) {
      elements.statusBadge.className = 'status-badge downloading';
      elements.statusText.textContent = 'Downloading in background...';
    } else {
      elements.statusBadge.className = 'status-badge error';
      elements.statusText.textContent = 'Not on Ads Transparency page';
    }
    
    // The currentProgressSection is managed by restoreProgress() separately
    return;
  }
  
  // If on page but download is in progress, show downloading status
  if (state.isDownloading) {
    elements.statusBadge.className = 'status-badge downloading';
    elements.statusText.textContent = 'Downloading...';
    // Don't return - continue to show page details
  }
  
  // 🆕 Single ad mode display
  if (state.searchMode === 'single_ad') {
    elements.statusBadge.className = 'status-badge ready';
    elements.statusText.textContent = 'Single Ad Detail Page';
    elements.pageDetails.style.display = 'block';
    
    elements.domainLabel.textContent = 'Creative:';
    elements.currentDomain.textContent = state.creativeId || 'Unknown';
    elements.adsCount.textContent = '1 ad';
    
    // Simplified UI for single ad
    if (!state.isDownloading) {
      elements.downloadSettings.style.display = 'block';
      elements.actionSection.style.display = 'block';
      
      // Hide pages selector (only 1 ad)
      const pagesGroup = elements.pagesInput?.closest('.form-group');
      if (pagesGroup) pagesGroup.style.display = 'none';
      
      // Hide format filter (not relevant for single ad)
      const formatGroup = elements.formatSelect?.closest('.form-group');
      if (formatGroup) formatGroup.style.display = 'none';
      
      // Show checkboxes but hide irrelevant ones
      const checkboxesGroup = document.querySelector('.form-group.checkboxes');
      if (checkboxesGroup) {
        checkboxesGroup.style.display = 'block';
        // Hide "Include image URLs" and "Include preview links" (not useful for single ad)
        if (elements.labelIncludeImages) elements.labelIncludeImages.style.display = 'none';
        if (elements.labelIncludePreview) elements.labelIncludePreview.style.display = 'none';
        // Keep video extraction and ad content analysis visible
        if (elements.labelFetchVideoLinks) elements.labelFetchVideoLinks.style.display = 'flex';
        if (elements.labelIncludeAdContent) elements.labelIncludeAdContent.style.display = 'flex';
      }
    }
    
    // Check quota
    if (state.dailyUsed >= state.dailyLimit) {
      elements.btnDownload.disabled = true;
      elements.btnDownload.innerHTML = '<span>Daily limit reached</span>';
    } else {
      elements.btnDownload.disabled = false;
      elements.btnDownload.innerHTML = '<span class="btn-icon">📥</span><span>Download This Ad</span>';
    }
    
    return;
  }
  
  // 🆕 Brand mode display
  if (state.searchMode === 'brand') {
    elements.statusBadge.className = 'status-badge ready';
    elements.statusText.textContent = 'Brand Aggregation Page';
    elements.pageDetails.style.display = 'block';
    
    // 设置标签为 "Advertiser:"
    elements.domainLabel.textContent = 'Advertiser:';
    
    // 显示广告商名字
    const displayName = (state.brandName && state.brandName !== 'Unknown Brand') 
      ? state.brandName 
      : (state.advertiserIds && state.advertiserIds.length > 0 
          ? state.advertiserIds[0].substring(0, 20) 
          : 'Multiple Advertisers');
    
    elements.currentDomain.textContent = displayName;
    elements.adsCount.textContent = state.adsCount > 0 
      ? `~${state.adsCount.toLocaleString()} ads (${state.advertiserIds?.length || 0} accounts)`
      : `${state.advertiserIds?.length || 0} accounts`;
    
    // Show download settings (只在未下载时显示)
    if (!state.isDownloading) {
      elements.downloadSettings.style.display = 'block';
      elements.actionSection.style.display = 'block';
      // Reset all form groups visibility
      resetFormGroupsVisibility();
    }
    
    // Check quota
    if (state.dailyUsed >= state.dailyLimit) {
      elements.btnDownload.disabled = true;
      elements.btnDownload.innerHTML = '<span>Daily limit reached</span>';
    } else {
      elements.btnDownload.disabled = false;
      elements.btnDownload.innerHTML = '<span class="btn-icon">📥</span><span>Start Download</span>';
    }
    
    return;  // 提前返回，不执行后面的 domain/advertiser 逻辑
  }
  
  // Update status (domain/advertiser mode)
  elements.statusBadge.className = 'status-badge ready';
  elements.statusText.textContent = 'Ready to download';
  elements.pageDetails.style.display = 'block';
  
  // 根据模式设置标签
  if (state.domain) {
    elements.domainLabel.textContent = 'Domain:';
    elements.currentDomain.textContent = state.domain;
  } else if (state.advertiserId) {
    elements.domainLabel.textContent = 'Advertiser:';
    // 优先显示 advertiserName，没有则显示 ID
    elements.currentDomain.textContent = state.advertiserName || state.advertiserId;
  } else {
    elements.domainLabel.textContent = 'Domain:';
    elements.currentDomain.textContent = '-';
  }
  
  elements.adsCount.textContent = state.adsCount > 0 ? 
    `${state.adsCount.toLocaleString()} ads` : 'Unknown';
  
  // Show download settings (只在未下载时显示)
  if (!state.isDownloading) {
    elements.downloadSettings.style.display = 'block';
    elements.actionSection.style.display = 'block';
    // Reset all form groups visibility
    resetFormGroupsVisibility();
  }
  
  // Check quota
  if (state.dailyUsed >= state.dailyLimit) {
    elements.btnDownload.disabled = true;
    elements.btnDownload.innerHTML = '<span>Daily limit reached</span>';
  } else {
    elements.btnDownload.disabled = false;
    elements.btnDownload.innerHTML = '<span class="btn-icon">📥</span><span>Start Download</span>';
  }
}

/**
 * Reset form groups visibility for non-single-ad modes
 */
function resetFormGroupsVisibility() {
  // Show all form groups
  const pagesGroup = elements.pagesInput?.closest('.form-group');
  if (pagesGroup) pagesGroup.style.display = 'block';
  
  const formatGroup = elements.formatSelect?.closest('.form-group');
  if (formatGroup) formatGroup.style.display = 'block';
  
  const checkboxesGroup = document.querySelector('.form-group.checkboxes');
  if (checkboxesGroup) checkboxesGroup.style.display = 'block';
  
  // Show all checkbox labels
  if (elements.labelFetchVideoLinks) elements.labelFetchVideoLinks.style.display = 'flex';
  if (elements.labelIncludeImages) elements.labelIncludeImages.style.display = 'flex';
  if (elements.labelIncludePreview) elements.labelIncludePreview.style.display = 'flex';
  if (elements.labelIncludeAdContent) elements.labelIncludeAdContent.style.display = 'flex';
}

function hideAllSections() {
  elements.pageDetails.style.display = 'none';
  elements.downloadSettings.style.display = 'none';
  elements.actionSection.style.display = 'none';
  elements.progressSection.style.display = 'none';
  elements.completeSection.style.display = 'none';
  elements.errorSection.style.display = 'none';
  elements.notOnPage.style.display = 'none';
}

// Download functions
async function startDownload() {
  // Validate we have search criteria
  if (!state.domain && !state.advertiserId && !state.brandName && !state.creativeId) {
    showError('No Search', 'Please search for a domain or advertiser first.');
    return;
  }
  
  // Check usage limits before starting
  const shouldProceed = await checkUsageLimitsBeforeDownload();
  if (!shouldProceed) {
    return; // User cancelled
  }
  
  state.isDownloading = true;
  state.isPaused = false;
  state.currentPage = 0;
  state.totalPages = parseInt(elements.pagesInput.value);
  state.fetchedAds = [];
  
  hideAllSections();
  elements.progressSection.style.display = 'block';
  
  // 动态设置标题
  const titleElement = elements.progressSection.querySelector('h3');
  if (titleElement) {
    titleElement.textContent = state.searchMode === 'single_ad' ? 'Processing Ad...' : 'Downloading...';
  }
  
  // 单个广告模式隐藏暂停按钮（处理很快，不需要暂停）
  if (elements.btnPause) {
    elements.btnPause.style.display = state.searchMode === 'single_ad' ? 'none' : 'inline-block';
  }
  
  updateProgress();
  
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    
    const config = {
      pages: state.searchMode === 'single_ad' ? 1 : state.totalPages,
      region: state.pageRegion || '',  // Use region from page URL
      format: elements.formatSelect.value,
      exportFormat: elements.exportFormat.value,
      includeImages: document.getElementById('include-images').checked,
      includePreview: document.getElementById('include-preview').checked,
      fetchVideoLinks: document.getElementById('fetch-video-links').checked,
      includeAdContent: document.getElementById('include-ad-content')?.checked || false, // v1.1.0: OCR
      domain: state.domain,
      advertiserId: state.advertiserId,
      searchMode: state.searchMode,
      // 🆕 Brand mode fields
      brandName: state.brandName,
      advertiserIds: state.advertiserIds,
      // 🆕 Single ad mode fields
      creativeId: state.creativeId,
      advertiserName: state.advertiserName,
      // 🔧 FIX: Include actual ads count for accurate quota calculation
      adsCount: state.adsCount || 0
    };
    
    log.info('Starting download with config:', config);
    
    // Start download via background script
    chrome.runtime.sendMessage({
      type: 'START_DOWNLOAD',
      config: config
    });
  } catch (e) {
    log.error('Download error:', e);
    showError('Download Failed', e.message);
  }
}

function togglePause() {
  state.isPaused = !state.isPaused;
  elements.btnPause.textContent = state.isPaused ? 'Resume' : 'Pause';
  
  chrome.runtime.sendMessage({
    type: state.isPaused ? 'PAUSE_DOWNLOAD' : 'RESUME_DOWNLOAD'
  });
}

function cancelDownload() {
  state.isDownloading = false;
  state.isPaused = false;
  
  chrome.runtime.sendMessage({ type: 'CANCEL_DOWNLOAD' });
  resetToStart();
}

function updateProgress() {
  // 单个广告模式优化
  if (state.searchMode === 'single_ad') {
    // 简化显示：不显示页数，直接显示状态
    elements.progressText.textContent = 'Fetching ad data...';
    elements.progressPercent.textContent = '';
    elements.progressFill.style.width = '100%'; // 显示动画效果
    
    // 隐藏统计信息（单个广告不需要）
    const statsDiv = elements.progressSection.querySelector('.progress-stats');
    if (statsDiv) statsDiv.style.display = 'none';
    
    return;
  }
  
  // 🔧 FIX: Simplify progress for small ad counts
  const adsCount = Array.isArray(state.fetchedAds) ? state.fetchedAds.length : 0;
  const totalAdsExpected = state.adsCount || (state.totalPages * 100);
  
  // If total ads ≤ 100, show simplified "Fetching X/Y ads" instead of pages
  if (totalAdsExpected <= 100) {
    elements.progressText.textContent = `Fetching ${adsCount}/${totalAdsExpected} ads`;
    const percent = totalAdsExpected > 0 ? Math.round((adsCount / totalAdsExpected) * 100) : 0;
    elements.progressPercent.textContent = `${percent}%`;
    elements.progressFill.style.width = `${percent}%`;
    
    // Hide page-based stats, show simplified info
    const statsDiv = elements.progressSection.querySelector('.progress-stats');
    if (statsDiv) statsDiv.style.display = 'none';
  } else {
    // Normal page-based progress for large downloads
    const percent = state.totalPages > 0 ? 
      Math.round((state.currentPage / state.totalPages) * UI.PROGRESS_PERCENTAGE_MAX) : 0;
    
    elements.progressText.textContent = `Page ${state.currentPage} of ${state.totalPages}`;
    elements.progressPercent.textContent = `${percent}%`;
    elements.progressFill.style.width = `${percent}%`;
    
    // 显示统计信息
    const statsDiv = elements.progressSection.querySelector('.progress-stats');
    if (statsDiv) statsDiv.style.display = 'flex';
    
    elements.fetchedCount.textContent = adsCount.toLocaleString();
    
    // 安全处理：过滤掉 null 值，只统计有效的广告
    const validAds = Array.isArray(state.fetchedAds) ? 
      state.fetchedAds.filter(ad => ad && ad.advertiser_id) : [];
    const uniqueAdvertisers = new Set(validAds.map(ad => ad.advertiser_id)).size;
    elements.advertisersCount.textContent = uniqueAdvertisers.toLocaleString();
  }
}

function showComplete(stats) {
  hideAllSections();
  elements.completeSection.style.display = 'block';
  
  // 安全处理：确保 stats 对象存在
  const safeStats = stats || {};
  
  // 单个广告模式简化显示
  if (state.searchMode === 'single_ad') {
    const titleElement = elements.completeSection.querySelector('h3');
    if (titleElement) titleElement.textContent = 'Export Complete!';
    
    elements.totalAds.textContent = '1';
    
    // 隐藏不相关的统计
    const statsRows = elements.completeSection.querySelectorAll('.stat-row');
    if (statsRows.length >= 2) {
      statsRows[1].style.display = 'none'; // 隐藏 "Unique advertisers"
      statsRows[2].style.display = 'none'; // 隐藏 "Duplicates removed"
    }
  } else {
    // 批量下载模式正常显示
    const titleElement = elements.completeSection.querySelector('h3');
    if (titleElement) titleElement.textContent = 'Download Complete!';
    
    elements.totalAds.textContent = (safeStats.totalAds || 0).toLocaleString();
    elements.uniqueAdvertisers.textContent = (safeStats.uniqueAdvertisers || 0).toLocaleString();
    elements.duplicatesRemoved.textContent = (safeStats.duplicatesRemoved || 0).toLocaleString();
    
    // 显示所有统计
    const statsRows = elements.completeSection.querySelectorAll('.stat-row');
    statsRows.forEach(row => row.style.display = 'flex');
  }
  
  // Update daily usage
  state.dailyUsed++;
  chrome.storage.local.set({ dailyUsed: state.dailyUsed });
}

function showError(title, message) {
  hideAllSections();
  elements.errorSection.style.display = 'block';
  document.getElementById('error-title').textContent = title;
  document.getElementById('error-message').textContent = message;
}

function dismissError() {
  resetToStart();
}

function resetToStart() {
  state.isDownloading = false;
  state.isPaused = false;
  state.currentPage = 0;
  state.fetchedAds = [];
  updateUI();
}

// Navigation functions
function openTransparencyPage() {
  chrome.tabs.create({ url: 'https://adstransparency.google.com/' });
}

function openSettings() {
  window.location.href = 'settings.html';
}

function openAccount() {
  window.location.href = 'account.html';
}

// Rating prompt functions
async function checkRatingPrompt() {
  try {
    const data = await chrome.storage.local.get([
      'ratingDismissed',
      'ratingCompleted',
      'ratingPromptCount',
      'lastRatingPrompt',
      'totalDownloads',
      'installDate'
    ]);
    
    // User already rated or permanently dismissed
    if (data.ratingCompleted || data.ratingDismissed === 'never') {
      return;
    }
    
    // Max prompts reached (3)
    if ((data.ratingPromptCount || 0) >= 3) {
      return;
    }
    
    // Check cooldown
    if (data.lastRatingPrompt) {
      const daysSinceLastPrompt = timestampToDaysAgo(data.lastRatingPrompt);
      if (daysSinceLastPrompt < TIME.RATING_COOLDOWN_DAYS) {
        return;
      }
    }
    
    // Check minimum downloads (5)
    if ((data.totalDownloads || 0) < 5) {
      return;
    }
    
    // Check minimum days installed
    if (data.installDate) {
      const daysInstalled = timestampToDaysAgo(data.installDate);
      if (daysInstalled < TIME.MIN_INSTALL_DAYS_FOR_RATING) {
        return;
      }
    }
    
    // Show rating prompt
    showRatingPrompt();
  } catch (e) {
    log.error('Rating check error:', e);
  }
}

function showRatingPrompt() {
  // Create rating prompt element
  const promptHTML = `
    <div class="rating-prompt" id="rating-prompt">
      <div class="rating-content">
        <div class="rating-icon">⭐</div>
        <h4>Enjoying Ads Insight Pro?</h4>
        <p>Help us grow by leaving a review!</p>
        <div class="rating-actions">
          <button class="btn-primary btn-small" id="btn-rate">Rate Now</button>
          <button class="btn-secondary btn-small" id="btn-rate-later">Maybe Later</button>
        </div>
        <button class="btn-link" id="btn-rate-never">Don't ask again</button>
      </div>
    </div>
  `;
  
  // Add CSS
  const style = document.createElement('style');
  style.textContent = `
    .rating-prompt {
      position: fixed;
      bottom: 0;
      left: 0;
      right: 0;
      background: white;
      border-top: 1px solid #eee;
      padding: 16px;
      box-shadow: 0 -4px 12px rgba(0,0,0,0.1);
      z-index: ${UI.Z_INDEX_RATING_BANNER};
      animation: slideUp 0.3s ease;
    }
    @keyframes slideUp {
      from { transform: translateY(100%); }
      to { transform: translateY(0); }
    }
    .rating-content { text-align: center; }
    .rating-icon { font-size: 32px; margin-bottom: 8px; }
    .rating-content h4 { font-size: 15px; font-weight: 600; color: #333; margin-bottom: 4px; }
    .rating-content p { font-size: 13px; color: #666; margin-bottom: 12px; }
    .rating-actions { display: flex; gap: 8px; justify-content: center; margin-bottom: 8px; }
    .rating-actions button { min-width: 100px; }
    .btn-small { padding: 8px 16px; font-size: 13px; }
    .btn-link { background: none; border: none; color: #888; font-size: 12px; cursor: pointer; text-decoration: underline; }
  `;
  document.head.appendChild(style);
  
  // Add prompt to page
  document.body.insertAdjacentHTML('beforeend', promptHTML);
  
  // Record prompt shown
  chrome.storage.local.get('ratingPromptCount').then(data => {
    chrome.storage.local.set({
      ratingPromptCount: (data.ratingPromptCount || 0) + 1,
      lastRatingPrompt: Date.now()
    });
  });
  
  // Event listeners
  document.getElementById('btn-rate').addEventListener('click', () => {
    chrome.storage.local.set({ ratingCompleted: true });
    // chrome.runtime.id will automatically use the correct extension ID after publishing
    const extensionId = chrome.runtime.id;
    chrome.tabs.create({ url: `https://chrome.google.com/webstore/detail/${extensionId}/reviews` });
    document.getElementById('rating-prompt').remove();
  });
  
  document.getElementById('btn-rate-later').addEventListener('click', () => {
    document.getElementById('rating-prompt').remove();
  });
  
  document.getElementById('btn-rate-never').addEventListener('click', () => {
    chrome.storage.local.set({ ratingDismissed: 'never' });
    document.getElementById('rating-prompt').remove();
  });
}

// Listen for messages from background script
chrome.runtime.onMessage.addListener((message, _sender, _sendResponse) => {
  log.debug('Received message:', message.type, message);
  
  switch (message.type) {
    case 'DOWNLOAD_PROGRESS':
      state.currentPage = message.currentPage;
      state.totalPages = message.totalPages;
      // Update fetched ads count from message
      if (message.adsCount !== undefined) {
        // Create placeholder array for count display
        state.fetchedAds = new Array(message.adsCount);
      }
      if (message.advertisersCount !== undefined) {
        elements.advertisersCount.textContent = message.advertisersCount.toLocaleString();
      }
      updateProgress();
      // 同时更新主页面的当前进度模块
      updateCurrentProgressModule({
        currentPage: message.currentPage,
        totalPages: message.totalPages,
        adsCount: message.adsCount
      });
      break;
    
    // 🆕 Brand progress update
    case 'BRAND_PROGRESS':
      updateBrandProgress(message);
      break;
    
    // 🆕 Brand export warning
    case 'BRAND_EXPORT_WARNING':
      showBrandWarning(message);
      break;
      
    case 'VIDEO_PROCESSING_START':
      // 显示视频处理进度区域
      elements.videoProgress.style.display = 'block';
      elements.videoProgressText.textContent = `0/${message.total}`;
      elements.videoProgressFill.style.width = '0%';
      // 隐藏主进度条，避免混乱
      if (elements.mainProgressInfo) elements.mainProgressInfo.style.display = 'none';
      if (elements.mainProgressBar) elements.mainProgressBar.style.display = 'none';
      
      // 同时更新主页面的视频进度
      updateCurrentVideoProgress(0, message.total);
      break;
      
    case 'VIDEO_PROCESSING_PROGRESS':
      // 更新视频处理进度
      const videoPercent = message.total > 0 ? 
        Math.round((message.processed / message.total) * 100) : 0;
      elements.videoProgressText.textContent = `${message.processed}/${message.total} (${message.found} found)`;
      elements.videoProgressFill.style.width = `${videoPercent}%`;
      // 同时更新主页面的视频进度
      updateCurrentVideoProgress(message.processed, message.total);
      break;
      
    case 'VIDEO_PROCESSING_COMPLETE':
      // 视频处理完成，隐藏进度区域
      setTimeout(() => {
        elements.videoProgress.style.display = 'none';
        // 隐藏主页面的视频进度
        if (elements.currentVideoProgress) {
          elements.currentVideoProgress.style.display = 'none';
        }
      }, 1000);
      break;
    
    case 'OCR_PROCESSING_START':
      // 显示 OCR 处理进度区域
      if (elements.ocrProgress) {
        elements.ocrProgress.style.display = 'block';
        elements.ocrProgressText.textContent = `0/${message.total}`;
        elements.ocrProgressFill.style.width = '0%';
      }
      // v1.2.1: 更新主进度显示为 OCR 模式
      if (elements.progressText) {
        elements.progressText.textContent = `OCR Processing: 0/${message.total}`;
      }
      if (elements.progressFill) {
        elements.progressFill.style.width = '0%';
      }
      if (elements.progressPercent) {
        elements.progressPercent.textContent = '0%';
      }
      break;
      
    case 'OCR_PROCESSING_PROGRESS':
      // 更新 OCR 处理进度
      if (elements.ocrProgress) {
        const ocrPercent = message.total > 0 ? 
          Math.round((message.processed / message.total) * 100) : 0;
        elements.ocrProgressText.textContent = `${message.processed}/${message.total}`;
        elements.ocrProgressFill.style.width = `${ocrPercent}%`;
      }
      // v1.2.1: 同时更新主进度显示
      if (elements.progressText) {
        const ocrPercent = message.total > 0 ? 
          Math.round((message.processed / message.total) * 100) : 0;
        elements.progressText.textContent = `OCR Processing: ${message.processed}/${message.total}`;
        if (elements.progressFill) {
          elements.progressFill.style.width = `${ocrPercent}%`;
        }
        if (elements.progressPercent) {
          elements.progressPercent.textContent = `${ocrPercent}%`;
        }
      }
      break;
      
    case 'OCR_PROCESSING_COMPLETE':
      // OCR 处理完成，隐藏进度区域
      setTimeout(() => {
        if (elements.ocrProgress) {
          elements.ocrProgress.style.display = 'none';
        }
      }, 1000);
      break;
      
    case 'DOWNLOAD_COMPLETE':
      state.isDownloading = false;
      showComplete(message.stats);
      // 隐藏当前进度模块
      if (elements.currentProgressSection) {
        elements.currentProgressSection.style.display = 'none';
      }
      // 显示使用量警告（如果有）
      if (message.warnings && message.warnings.length > 0) {
        message.warnings.forEach(warning => {
          showWarningBanner(warning);
        });
      }
      break;
      
    case 'DOWNLOAD_ERROR':
      state.isDownloading = false;
      showError(message.title || 'Error', message.message);
      break;
  }
});

// ============== Usage Limit Check ==============

/**
 * 在下载前检查使用量限制
 * @returns {boolean} true = 继续下载, false = 取消下载
 */
async function checkUsageLimitsBeforeDownload() {
  try {
    // 获取当前使用量统计
    const response = await chrome.runtime.sendMessage({ type: 'GET_USAGE_STATS' });
    
    if (!response || !response.success) {
      // 如果获取失败，允许继续（不阻止用户）
      log.warn('Failed to get usage stats, allowing download');
      return true;
    }
    
    const { stats } = response;
    // 🔧 FIX: Use actual ads count calculation instead of parsing text
    const pages = parseInt(elements.pagesInput.value) || 1;
    const estimatedAds = state.adsCount > 0 
      ? Math.min(state.adsCount, pages * API.ESTIMATED_ADS_PER_PAGE)
      : pages * API.ESTIMATED_ADS_PER_PAGE;
    
    log.debug('UsageLimitCheck current stats:', stats);
    log.debug('UsageLimitCheck estimated ads:', estimatedAds);
    
    // 安全检查：如果 stats 为空或缺少必要字段，使用默认值
    const currentHourlyAds = stats?.hourly?.ads || 0;
    const currentDailyAds = stats?.daily?.ads || 0;
    
    // 计算下载后的预计总量
    const projectedHourlyAds = currentHourlyAds + estimatedAds;
    const projectedDailyAds = currentDailyAds + estimatedAds;
    
    log.debug('UsageLimitCheck projected hourly:', projectedHourlyAds);
    log.debug('UsageLimitCheck projected daily:', projectedDailyAds);
    
    // 硬编码阈值（与 usage-thresholds.js 保持一致）
    // 基于实际测试: 单IP限制约 12,000 ads
    const HOURLY_WARNING = 8000;
    const HOURLY_DANGER = 10000;
    const DAILY_WARNING = 12000;   // 接近单IP限制
    const DAILY_DANGER = 15000;    // 超过单IP限制
    
    // 检查是否会超过危险阈值
    if (projectedHourlyAds >= HOURLY_DANGER || projectedDailyAds >= DAILY_DANGER) {
      const isDailyLimit = projectedDailyAds >= DAILY_DANGER;
      return await showUsageLimitConfirmation({
        level: 'danger',
        title: '🔴 High Risk Warning',
        message: isDailyLimit 
          ? `This download will bring you to ${projectedDailyAds.toLocaleString()} ads today, exceeding the single-IP limit (~12,000 ads). Google will likely block your IP.`
          : `This download will bring you to ${projectedHourlyAds.toLocaleString()} ads this hour (${projectedDailyAds.toLocaleString()} today), which may trigger Google rate limits.`,
        suggestions: isDailyLimit
          ? [
              'Stop downloading for today',
              'Resume tomorrow (IP limit resets)',
              'Use a proxy service if you need more',
              'Contact support for enterprise solutions'
            ]
          : [
              'Consider downloading fewer pages',
              'Wait 30-60 minutes before downloading',
              'Spread downloads throughout the day'
            ],
        currentHourly: currentHourlyAds,
        currentDaily: currentDailyAds,
        estimatedAds: estimatedAds,
        projectedHourly: projectedHourlyAds,
        projectedDaily: projectedDailyAds
      });
    }
    
    // 检查是否会超过警告阈值
    if (projectedHourlyAds >= HOURLY_WARNING || projectedDailyAds >= DAILY_WARNING) {
      const isDailyWarning = projectedDailyAds >= DAILY_WARNING;
      return await showUsageLimitConfirmation({
        level: 'warning',
        title: '⚠️ Usage Warning',
        message: isDailyWarning
          ? `This download will bring you to ${projectedDailyAds.toLocaleString()} ads today. You are approaching the single-IP limit (~12,000 ads).`
          : `This download will bring you to ${projectedHourlyAds.toLocaleString()} ads this hour (${projectedDailyAds.toLocaleString()} today). You are approaching the safe limit.`,
        suggestions: isDailyWarning
          ? [
              'Consider reducing pages to download',
              'Take a break after this download',
              'You have ~' + (15000 - projectedDailyAds).toLocaleString() + ' ads remaining today'
            ]
          : [
              'Consider reducing pages to download',
              'Take a break after this download',
              'Monitor your usage in Settings'
            ],
        currentHourly: currentHourlyAds,
        currentDaily: currentDailyAds,
        estimatedAds: estimatedAds,
        projectedHourly: projectedHourlyAds,
        projectedDaily: projectedDailyAds
      });
    }
    
    // 安全范围，直接继续
    return true;
    
  } catch (e) {
    log.error('Usage limit check error:', e);
    // 出错时允许继续
    return true;
  }
}

/**
 * 显示使用量限制确认对话框
 * @returns {Promise<boolean>} true = 继续, false = 取消
 */
function showUsageLimitConfirmation(data) {
  return new Promise((resolve) => {
    // 创建确认对话框
    const dialog = document.createElement('div');
    dialog.className = 'usage-limit-dialog-overlay';
    dialog.innerHTML = `
      <div class="usage-limit-dialog ${data.level}">
        <div class="dialog-header">
          <h3>${data.title}</h3>
        </div>
        <div class="dialog-body">
          <p class="dialog-message">${data.message}</p>
          
          <div class="usage-breakdown">
            <div class="breakdown-item">
              <span class="breakdown-label">Current (This Hour):</span>
              <span class="breakdown-value">${data.currentHourly.toLocaleString()} ads</span>
            </div>
            <div class="breakdown-item">
              <span class="breakdown-label">This Download:</span>
              <span class="breakdown-value">+${data.estimatedAds.toLocaleString()} ads</span>
            </div>
            <div class="breakdown-item breakdown-total">
              <span class="breakdown-label">After Download:</span>
              <span class="breakdown-value">${data.projectedHourly.toLocaleString()} ads</span>
            </div>
          </div>
          
          <div class="dialog-suggestions">
            <p><strong>Suggestions:</strong></p>
            <ul>
              ${data.suggestions.map(s => `<li>${s}</li>`).join('')}
            </ul>
          </div>
        </div>
        <div class="dialog-footer">
          <button class="btn-dialog btn-cancel" id="usage-dialog-cancel">Cancel</button>
          <button class="btn-dialog btn-proceed ${data.level}" id="usage-dialog-proceed">
            ${data.level === 'danger' ? 'Download Anyway (Not Recommended)' : 'Continue Download'}
          </button>
        </div>
      </div>
    `;
    
    document.body.appendChild(dialog);
    
    // 事件处理
    const btnCancel = dialog.querySelector('#usage-dialog-cancel');
    const btnProceed = dialog.querySelector('#usage-dialog-proceed');
    
    const cleanup = () => {
      dialog.remove();
    };
    
    btnCancel.addEventListener('click', () => {
      cleanup();
      resolve(false);
    });
    
    btnProceed.addEventListener('click', () => {
      cleanup();
      resolve(true);
    });
    
    // 点击背景关闭
    dialog.addEventListener('click', (e) => {
      if (e.target === dialog) {
        cleanup();
        resolve(false);
      }
    });
  });
}

// ============== 🆕 Brand Export Functions ==============

/**
 * 更新品牌导出进度显示
 * @param {Object} data - 进度数据
 */
function updateBrandProgress(data) {
  const { current, total, advertiserId, collectedAds } = data;
  
  // 更新进度文本
  const progressText = `Exporting account ${current}/${total}`;
  const detailText = `${collectedAds} ads collected`;
  
  // 更新主进度区域
  if (elements.progressText) {
    elements.progressText.textContent = progressText;
  }
  
  // 更新当前进度模块（如果可见）
  if (elements.currentProgressSection && 
      elements.currentProgressSection.style.display !== 'none') {
    if (elements.currentProgressText) {
      elements.currentProgressText.textContent = `Account ${current}/${total}`;
    }
    if (elements.currentProgressAds) {
      elements.currentProgressAds.textContent = detailText;
    }
    
    // 更新进度条
    const percent = total > 0 ? Math.round((current / total) * 100) : 0;
    if (elements.currentProgressFill) {
      elements.currentProgressFill.style.width = `${percent}%`;
    }
  }
  
  // 更新详细进度页面（如果可见）
  if (elements.progressSection && 
      elements.progressSection.style.display !== 'none') {
    if (elements.fetchedCount) {
      elements.fetchedCount.textContent = collectedAds.toLocaleString();
    }
    
    // 更新进度条
    const percent = total > 0 ? Math.round((current / total) * 100) : 0;
    if (elements.progressFill) {
      elements.progressFill.style.width = `${percent}%`;
    }
    if (elements.progressPercent) {
      elements.progressPercent.textContent = `${percent}%`;
    }
  }
  
  log.debug(`Brand progress: ${current}/${total}, ${collectedAds} ads`);
}

/**
 * 显示品牌导出警告对话框
 * @param {Object} data - 警告数据
 */
function showBrandWarning(data) {
  const { failedCount, successCount, totalAds, failedIds, message } = data;
  
  // 构建警告消息
  let warningMessage = message || 
    `Export completed with warnings:\n\n` +
    `✅ ${successCount} accounts succeeded\n` +
    `❌ ${failedCount} accounts failed\n` +
    `📊 Total ads exported: ${totalAds.toLocaleString()}`;
  
  // 如果有失败的 ID，添加到消息中
  if (failedIds && failedIds.length > 0 && failedIds.length <= 5) {
    warningMessage += `\n\nFailed accounts:\n${failedIds.join('\n')}`;
  } else if (failedIds && failedIds.length > 5) {
    const preview = failedIds.slice(0, 5);
    warningMessage += `\n\nFailed accounts (showing first 5 of ${failedIds.length}):\n${preview.join('\n')}\n...and ${failedIds.length - 5} more`;
    log.warn('Failed advertiser IDs:', failedIds);
  }
  
  // 显示警告对话框
  alert(warningMessage);
  
  log.warn('Brand export warning:', data);
}

// ============== v1.2.0: Remote Config UI Functions ==============

/**
 * Show maintenance mode screen
 */
function showMaintenanceMode() {
  const maintenanceMessage = RemoteConfig.get('feature_flags.maintenance_message', 
    'Service is temporarily unavailable for maintenance.');
  
  // Hide main content
  document.querySelector('.container').innerHTML = `
    <header class="header">
      <div class="logo">
        <img src="../icons/icon48.png" alt="Ads Insight Pro" width="32" height="32">
        <span>Ads Insight Pro</span>
      </div>
    </header>
    <main class="main">
      <section class="section" style="text-align: center; padding: 40px 20px;">
        <div style="font-size: 48px; margin-bottom: 16px;">🔧</div>
        <h3 style="color: #333; margin-bottom: 12px;">Maintenance Mode</h3>
        <p style="color: #666; font-size: 14px; line-height: 1.5;">${escapeHtml(maintenanceMessage)}</p>
        <p style="color: #888; font-size: 12px; margin-top: 16px;">Please check back later.</p>
      </section>
    </main>
  `;
}

/**
 * Show force update modal (blocking)
 * @param {Object} versionCheck - Version check result
 */
function showForceUpdateModal(versionCheck) {
  // Ensure updateUrl is a valid string
  let updateUrl = versionCheck.updateUrl;
  if (typeof updateUrl !== 'string' || !updateUrl.startsWith('http')) {
    updateUrl = 'https://chrome.google.com/webstore';
  }
  const updateMessage = RemoteConfig.get('ui_text.update_required', 
    `Version ${versionCheck.latestVersion} is required. Your version (${versionCheck.currentVersion}) is no longer supported.`);
  
  // Create blocking modal
  const modalHTML = `
    <div class="force-update-overlay" id="force-update-overlay">
      <div class="force-update-modal">
        <div class="update-icon">⚠️</div>
        <h3>Update Required</h3>
        <p>${escapeHtml(updateMessage)}</p>
        <button class="btn-primary" id="btn-force-update">Update Now</button>
      </div>
    </div>
  `;
  
  // Add styles
  const style = document.createElement('style');
  style.textContent = `
    .force-update-overlay {
      position: fixed;
      top: 0; left: 0; right: 0; bottom: 0;
      background: rgba(0,0,0,0.8);
      display: flex;
      align-items: center;
      justify-content: center;
      z-index: 10000;
    }
    .force-update-modal {
      background: white;
      border-radius: 12px;
      padding: 32px;
      text-align: center;
      max-width: 300px;
      box-shadow: 0 8px 32px rgba(0,0,0,0.3);
    }
    .force-update-modal .update-icon { font-size: 48px; margin-bottom: 16px; }
    .force-update-modal h3 { font-size: 18px; font-weight: 600; color: #333; margin-bottom: 12px; }
    .force-update-modal p { font-size: 14px; color: #666; line-height: 1.5; margin-bottom: 20px; }
  `;
  document.head.appendChild(style);
  
  document.body.insertAdjacentHTML('beforeend', modalHTML);
  
  document.getElementById('btn-force-update').addEventListener('click', () => {
    chrome.tabs.create({ url: updateUrl });
  });
}

/**
 * Show update available notification (non-blocking)
 * @param {Object} versionCheck - Version check result
 */
function showUpdateAvailable(versionCheck) {
  // Don't show if user has dismissed this version
  chrome.storage.local.get('dismissedUpdateVersion').then(data => {
    if (data.dismissedUpdateVersion === versionCheck.latestVersion) {
      return;
    }
    
    // Ensure updateUrl is a valid string
    let updateUrl = versionCheck.updateUrl;
    if (typeof updateUrl !== 'string' || !updateUrl.startsWith('http')) {
      updateUrl = 'https://chrome.google.com/webstore';
    }
    
    const bannerHTML = `
      <div class="update-banner" id="update-banner">
        <div class="update-content">
          <span class="update-icon">✨</span>
          <span class="update-text">Version ${versionCheck.latestVersion} available</span>
        </div>
        <div class="update-actions">
          <button class="btn-small btn-primary" id="btn-update">Update</button>
          <button class="btn-small btn-link" id="btn-dismiss-update">Later</button>
        </div>
      </div>
    `;
    
    const style = document.createElement('style');
    style.textContent = `
      .update-banner {
        position: fixed;
        bottom: 40px;
        left: 8px; right: 8px;
        background: linear-gradient(135deg, #4285f4, #34a853);
        color: white;
        padding: 10px 12px;
        border-radius: 8px;
        display: flex;
        align-items: center;
        justify-content: space-between;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        z-index: 9000;
        animation: slideUpBanner 0.3s ease;
      }
      @keyframes slideUpBanner {
        from { transform: translateY(100%); opacity: 0; }
        to { transform: translateY(0); opacity: 1; }
      }
      .update-content { display: flex; align-items: center; gap: 8px; }
      .update-icon { font-size: 16px; }
      .update-text { font-size: 13px; font-weight: 500; }
      .update-actions { display: flex; gap: 8px; align-items: center; }
      .update-banner .btn-small { padding: 4px 12px; font-size: 12px; }
      .update-banner .btn-link { color: rgba(255,255,255,0.8); background: none; border: none; cursor: pointer; text-decoration: underline; }
    `;
    document.head.appendChild(style);
    
    document.body.insertAdjacentHTML('beforeend', bannerHTML);
    
    document.getElementById('btn-update').addEventListener('click', () => {
      chrome.tabs.create({ url: updateUrl });
      document.getElementById('update-banner').remove();
    });
    
    document.getElementById('btn-dismiss-update').addEventListener('click', () => {
      chrome.storage.local.set({ dismissedUpdateVersion: versionCheck.latestVersion });
      document.getElementById('update-banner').remove();
    });
  });
}

/**
 * Show announcements from remote config
 */
async function showAnnouncements() {
  try {
    const announcements = await RemoteConfig.getAnnouncements();
    
    if (!announcements || !Array.isArray(announcements) || announcements.length === 0) {
      return;
    }
    
    // Get dismissed announcements from storage
    const data = await chrome.storage.local.get('dismissedAnnouncements');
    const dismissedIds = data.dismissedAnnouncements || [];
    
    // Filter to unshown announcements
    const toShow = announcements.filter(a => !dismissedIds.includes(a.id));
    
    if (toShow.length === 0) {
      return;
    }
    
    // Show first announcement (can be extended to queue multiple)
    const announcement = toShow[0];
    showAnnouncementBanner(announcement);
  } catch (e) {
    log.warn('Failed to show announcements:', e.message);
  }
}

/**
 * Show single announcement banner
 * @param {Object} announcement - Announcement object
 */
function showAnnouncementBanner(announcement) {
  const typeColors = {
    info: { bg: '#e8f0fe', border: '#4285f4', icon: 'ℹ️' },
    warning: { bg: '#fef7e0', border: '#fbbc04', icon: '⚠️' },
    success: { bg: '#e6f4ea', border: '#34a853', icon: '✅' },
    promo: { bg: '#fce8e6', border: '#ea4335', icon: '🎉' }
  };
  
  const colors = typeColors[announcement.type] || typeColors.info;
  
  const bannerHTML = `
    <div class="announcement-banner" id="announcement-${announcement.id}">
      <div class="announcement-content">
        <span class="announcement-icon">${colors.icon}</span>
        <span class="announcement-text">${escapeHtml(announcement.message)}</span>
      </div>
      ${announcement.link ? `<a href="#" class="announcement-link" data-url="${escapeHtml(announcement.link)}">Learn more</a>` : ''}
      ${announcement.dismissible !== false ? `<button class="announcement-close" data-id="${announcement.id}">×</button>` : ''}
    </div>
  `;
  
  const style = document.createElement('style');
  style.id = 'announcement-style';
  if (!document.getElementById('announcement-style')) {
    style.textContent = `
      .announcement-banner {
        background: ${colors.bg};
        border-left: 3px solid ${colors.border};
        padding: 10px 12px;
        margin: 8px;
        border-radius: 4px;
        display: flex;
        align-items: center;
        justify-content: space-between;
        gap: 8px;
        animation: fadeIn 0.3s ease;
      }
      @keyframes fadeIn {
        from { opacity: 0; }
        to { opacity: 1; }
      }
      .announcement-content { display: flex; align-items: center; gap: 8px; flex: 1; }
      .announcement-icon { font-size: 16px; }
      .announcement-text { font-size: 12px; color: #333; line-height: 1.4; }
      .announcement-link { font-size: 11px; color: #1a73e8; text-decoration: none; white-space: nowrap; }
      .announcement-link:hover { text-decoration: underline; }
      .announcement-close {
        background: none; border: none; font-size: 18px; color: #666;
        cursor: pointer; padding: 0 4px; line-height: 1;
      }
    `;
    document.head.appendChild(style);
  }
  
  // Insert after header
  const header = document.querySelector('.header');
  if (header) {
    header.insertAdjacentHTML('afterend', bannerHTML);
    
    // Add event listeners
    const banner = document.getElementById(`announcement-${announcement.id}`);
    
    const closeBtn = banner.querySelector('.announcement-close');
    if (closeBtn) {
      closeBtn.addEventListener('click', () => {
        dismissAnnouncement(announcement.id);
        banner.remove();
      });
    }
    
    const link = banner.querySelector('.announcement-link');
    if (link) {
      link.addEventListener('click', (e) => {
        e.preventDefault();
        chrome.tabs.create({ url: link.dataset.url });
      });
    }
  }
}

/**
 * Dismiss an announcement
 * @param {string} announcementId - Announcement ID
 */
async function dismissAnnouncement(announcementId) {
  const data = await chrome.storage.local.get('dismissedAnnouncements');
  const dismissed = data.dismissedAnnouncements || [];
  if (!dismissed.includes(announcementId)) {
    dismissed.push(announcementId);
    await chrome.storage.local.set({ dismissedAnnouncements: dismissed });
  }
}

/**
 * Escape HTML to prevent XSS
 * @param {string} text - Text to escape
 * @returns {string} Escaped text
 */
function escapeHtml(text) {
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}
