/**
 * OCR Proxy Content Script for Firefox
 * 
 * Firefox 的 background script 无法绕过 CORS 调用 OCR.space API，
 * 但 Content Script 可以通过页面上下文发送跨域请求。
 * 
 * 这个脚本作为代理，接收来自 background 的 OCR 请求，执行后返回结果。
 */

// 防止重复注入
if (window.__ocrProxyInjected) {
  console.log('[OCR Proxy] Already injected, skipping');
} else {
  window.__ocrProxyInjected = true;
  
  console.log('[OCR Proxy] Content script loaded');
  
  /**
   * 调用 OCR.space API
   */
  async function callOCRSpaceAPI(imageUrl, ocrConfig) {
    console.log('[OCR Proxy] Calling OCR.space API for:', imageUrl.substring(0, 80));
    
    if (!ocrConfig || !ocrConfig.api_key) {
      throw new Error('OCR API key not configured');
    }
    
    // 获取图片
    const response = await fetch(imageUrl);
    if (!response.ok) {
      throw new Error(`Failed to fetch image: ${response.status}`);
    }
    
    const blob = await response.blob();
    
    // 确定文件扩展名
    let ext = 'png';
    if (blob.type === 'image/jpeg' || blob.type === 'image/jpg') ext = 'jpg';
    else if (blob.type === 'image/gif') ext = 'gif';
    else if (blob.type === 'image/webp') ext = 'webp';
    
    // 转换为 base64
    const arrayBuffer = await blob.arrayBuffer();
    const uint8Array = new Uint8Array(arrayBuffer);
    let binary = '';
    for (let i = 0; i < uint8Array.length; i++) {
      binary += String.fromCharCode(uint8Array[i]);
    }
    const base64Image = `data:${blob.type};base64,${btoa(binary)}`;
    
    // 构建请求
    const formData = new FormData();
    formData.append('base64Image', base64Image);
    formData.append('filetype', ext.toUpperCase());
    formData.append('language', ocrConfig.languages?.[0] || 'eng');
    formData.append('isOverlayRequired', 'false');
    formData.append('detectOrientation', 'true');
    formData.append('scale', 'true');
    formData.append('OCREngine', String(ocrConfig.engine || 1));
    
    const apiEndpoint = ocrConfig.api_endpoint || 'https://api.ocr.space/parse/image';
    
    const ocrResponse = await fetch(apiEndpoint, {
      method: 'POST',
      headers: { 'apikey': ocrConfig.api_key },
      body: formData
    });
    
    const ocrResult = await ocrResponse.json();
    
    if (ocrResult.ParsedResults?.length > 0) {
      const parsed = ocrResult.ParsedResults[0];
      return {
        text: parsed.ParsedText || '',
        confidence: parsed.ParsedText?.trim() ? 80 : 0
      };
    }
    
    // 检查错误
    if (ocrResult.ErrorMessage) {
      console.warn('[OCR Proxy] OCR.space API error:', ocrResult.ErrorMessage);
      throw new Error(ocrResult.ErrorMessage);
    }
    
    return { text: '', confidence: 0 };
  }
  
  /**
   * 处理单个图片的 OCR
   */
  async function processOCRSingle(imageUrl, ocrConfig) {
    try {
      const ocrData = await callOCRSpaceAPI(imageUrl, ocrConfig);
      
      console.log('[OCR Proxy] OCR result:', {
        hasText: !!ocrData.text,
        textLength: ocrData.text?.length || 0,
        confidence: ocrData.confidence
      });
      
      return {
        success: true,
        text: ocrData.text || '',
        confidence: ocrData.confidence || 0
      };
    } catch (e) {
      console.error('[OCR Proxy] OCR failed:', e.message);
      return {
        success: false,
        error: e.message
      };
    }
  }
  
  /**
   * 批量处理 OCR
   */
  async function processOCRBatch(ads, ocrConfig, concurrency = 2) {
    console.log(`[OCR Proxy] Processing ${ads.length} images with concurrency: ${concurrency}`);
    
    const results = [];
    
    // 分批处理
    for (let i = 0; i < ads.length; i += concurrency) {
      const batch = ads.slice(i, Math.min(i + concurrency, ads.length));
      
      console.log(`[OCR Proxy] Processing batch ${Math.floor(i / concurrency) + 1}, items ${i + 1}-${i + batch.length}`);
      
      // 并行处理批次
      const batchPromises = batch.map(async (ad) => {
        const result = await processOCRSingle(ad.image_url, ocrConfig);
        return {
          creative_id: ad.creative_id,
          ...result
        };
      });
      
      const batchResults = await Promise.all(batchPromises);
      results.push(...batchResults);
      
      console.log(`[OCR Proxy] Batch complete: ${results.length}/${ads.length} processed`);
      
      // 批次间延迟
      if (i + concurrency < ads.length) {
        await new Promise(r => setTimeout(r, 500));
      }
    }
    
    console.log(`[OCR Proxy] All OCR complete: ${results.length} results`);
    return results;
  }
  
  // 监听来自 background 的消息
  browser.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.type === 'OCR_PROXY_PING') {
      // 健康检查
      return Promise.resolve({ success: true, injected: true });
    }
    
    if (message.type === 'OCR_PROXY_SINGLE') {
      // 单个图片 OCR
      return processOCRSingle(message.imageUrl, message.ocrConfig);
    }
    
    if (message.type === 'OCR_PROXY_BATCH') {
      // 批量 OCR
      return processOCRBatch(message.ads, message.ocrConfig, message.concurrency || 2);
    }
    
    return false;
  });
  
  // 通知 background 脚本已注入
  browser.runtime.sendMessage({ type: 'OCR_PROXY_READY' }).catch(() => {
    // 忽略错误（可能 background 还没准备好接收）
  });
  
  console.log('[OCR Proxy] Message listener registered');
}
