﻿/**
 * Ads Insight Pro - API Client
 * 
 * 统一 API 客户端，处理�?SaaS 服务的通信
 * 核心功能�?
 * 1. 数据处理 (Process) - 字段过滤 + 视频提取
 * 2. 配额验证 (Validate) - 预检�?
 * 3. 视频提取 (Video) - Enterprise 专属
 */

import { browserAPI } from './browser-polyfill.js';
import { getApiUrl } from './config.js';
import { getAuthToken } from './storage.js';

// Helper to get API base URL dynamically
const getBaseUrl = () => getApiUrl();

/**
 * 统一 API 响应处理
 */
async function handleResponse(response, operation) {
  if (!response.ok) {
    let errorMessage = `${operation} failed`;
    try {
      const errorData = await response.json();
      if (errorData.error && errorData.error.message) {
        errorMessage = errorData.error.message;
      }
    } catch (e) {
      // 无法解析 JSON 错误，使用状态文�?
      errorMessage = `${operation} failed: ${response.status} ${response.statusText}`;
    }
    throw new Error(errorMessage);
  }
  
  return await response.json();
}

/**
 * API 客户端对�?
 */
export const apiClient = {
  
  /**
   * 核心数据处理服务
   * 发送原始数据到服务器，获取处理后的数据
   * 
   * @param {Array} ads - 原始广告数据数组
   * @param {Object} options - 处理选项
   * @returns {Promise<Object>} - 处理结果 { success: true, data: { ads: [...] } }
   */
  async processAds(ads, options = {}) {
    // 自动分批，每�?50 �?
    const BATCH_SIZE = 50; 
    
    if (ads.length <= BATCH_SIZE) {
      return this._processBatch(ads, options);
    }
    
    // 分批处理
    const results = {
      success: true,
      data: { 
        ads: [], 
        meta: { 
          total: 0, 
          filtered_fields: 0, 
          videos_extracted: 0, 
          quota_remaining: 0,
          plan: null,
          video_extraction_allowed: false
        } 
      }
    };
    
    for (let i = 0; i < ads.length; i += BATCH_SIZE) {
      const batch = ads.slice(i, i + BATCH_SIZE);
      const result = await this._processBatch(batch, options);
      
      if (!result.success) {
        throw new Error(result.error?.message || `Batch processing failed at index ${i}`);
      }
      
      results.data.ads.push(...result.data.ads);
      // Merge meta data
      if (result.data.meta) {
        results.data.meta.total += result.data.meta.total || 0;
        results.data.meta.filtered_fields = result.data.meta.filtered_fields || 0;
        results.data.meta.videos_extracted += result.data.meta.videos_extracted || 0;
        results.data.meta.quota_remaining = result.data.meta.quota_remaining; // Use last batch
        // �?确保 plan �?video_extraction_allowed 被传�?
        results.data.meta.plan = result.data.meta.plan;
        results.data.meta.video_extraction_allowed = result.data.meta.video_extraction_allowed;
      }
    }
    
    return results;
  },

  /**
   * 内部处理单个批次
   */
  async _processBatch(ads, options) {
    const token = await getAuthToken();
    
    if (!token) {
      throw new Error('Please log in to process data');
    }
    
    const response = await fetch(`${getBaseUrl()}/api/services/process`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      },
      body: JSON.stringify({ ads, options })
    });
    
    return handleResponse(response, 'Data processing');
  },
  
  /**
   * 视频提取服务 (Enterprise)
   * 上传 URL 列表到服务器，由服务器解析提�?YouTube ID
   * 
   * @param {Object} payload - { items: [{ creative_id, advertiser_id, urls: [...] }] }
   * @returns {Promise<Object>} - 提取结果
   */
  async extractVideos(payload) {
    const token = await getAuthToken();
    
    if (!token) {
      throw new Error('Please log in to extract videos');
    }
    
    const response = await fetch(`${getBaseUrl()}/api/services/video/extract`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      },
      body: JSON.stringify(payload)
    });
    
    return handleResponse(response, 'Video extraction');
  },
  
  /**
   * 配额验证
   * 预检查用户是否有足够配额执行操作
   * 
   * @param {string} action - 操作类型 ('export', 'process', 'video_extract')
   * @param {number} count - 数量
   * @param {number} pages - 页数
   * @returns {Promise<Object>} - 验证结果
   */
  async validateQuota(action = 'export', count = 1, pages = 1) {
    const token = await getAuthToken();
    
    // 如果未登录，返回基本限制（模�?Free Plan�?
    if (!token) {
      // 获取本地存储的用户计划（用于测试�?
      const { userPlan } = await browserAPI.storage.local.get('userPlan');
      const plan = userPlan || 'free';
      
      return {
        success: true,
        data: {
          allowed: count <= 500 && pages <= 5, // Free limits
          plan: plan,
          quota: { remaining: 2 },
          is_guest: true,
          features: {
            video_extraction: (plan === 'enterprise')  // �?添加 features 字段
          }
        }
      };
    }
    
    const response = await fetch(`${getBaseUrl()}/api/services/validate`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      },
      body: JSON.stringify({ action, count, pages })
    });
    
    return handleResponse(response, 'Quota validation');
  },
  
  /**
   * 获取功能特�?
   * 获取当前用户的可用功能列�?
   */
  async getFeatures() {
    const token = await getAuthToken();
    
    if (!token) {
      // 返回默认 Free 特�?
      return {
        success: true,
        data: {
          plan: 'free',
          features: ['csv'],
          limits: { daily_downloads: 2 },
          is_guest: true
        }
      };
    }
    
    const response = await fetch(`${getBaseUrl()}/api/services/features`, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      }
    });
    
    return handleResponse(response, 'Fetch features');
  },

  // ==================== v1.1.0 Quota System ====================

  /**
   * 获取用户权限配置（包含配额和功能�?
   * @returns {Promise<Object>} 权限配置
   */
  async getPermissions() {
    const token = await getAuthToken();
    
    if (!token) {
      // 返回默认 Free 权限
      return {
        success: true,
        data: {
          plan: 'free',
          quota: {
            daily_limit: 100,
            per_request_limit: 100,
            used_today: 0,
            remaining: 100,
            resets_at: new Date(Date.now() + 86400000).toISOString()
          },
          features: {
            export_formats: ['csv'],
            video_extraction: false,
            ad_content_export: false,
            campaign_info_export: false,
            advertiser_export: true,
            enhanced_fields: false
          },
          limits: {
            max_pages: 5,
            max_devices: 1
          },
          is_guest: true
        }
      };
    }
    
    const response = await fetch(`${getBaseUrl()}/api/quota/permissions`, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      }
    });
    
    return handleResponse(response, 'Get permissions');
  },

  /**
   * 预扣配额（导出前调用�?
   * @param {number} count - 预计导出的广告数�?
   * @returns {Promise<Object>} 预扣结果
   */
  async reserveQuota(count) {
    const token = await getAuthToken();
    
    if (!token) {
      return {
        success: false,
        data: { success: false, reason: 'not_authenticated', message: 'Please log in to export data' }
      };
    }
    
    const response = await fetch(`${getBaseUrl()}/api/quota/reserve`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      },
      body: JSON.stringify({ count })
    });
    
    return handleResponse(response, 'Reserve quota');
  },

  /**
   * 确认配额消耗（导出成功后调用）
   * @param {string} reservationId - 预扣 ID
   * @param {number} actualCount - 实际导出的广告数�?
   * @returns {Promise<Object>} 确认结果
   */
  async confirmQuota(reservationId, actualCount) {
    const token = await getAuthToken();
    
    if (!token) {
      return { success: false, error: 'Not authenticated' };
    }
    
    const response = await fetch(`${getBaseUrl()}/api/quota/confirm`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      },
      body: JSON.stringify({ reservation_id: reservationId, actual_count: actualCount })
    });
    
    return handleResponse(response, 'Confirm quota');
  },

  /**
   * 取消配额预扣（导出失败时调用�?
   * @param {string} reservationId - 预扣 ID
   * @returns {Promise<Object>} 取消结果
   */
  async cancelQuota(reservationId) {
    const token = await getAuthToken();
    
    if (!token) {
      return { success: true }; // 未登录时静默成功
    }
    
    try {
      const response = await fetch(`${getBaseUrl()}/api/quota/cancel`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`
        },
        body: JSON.stringify({ reservation_id: reservationId })
      });
      
      return handleResponse(response, 'Cancel quota');
    } catch (e) {
      // 取消失败不是致命错误，预扣会自动过期
      log.warn('Cancel quota failed:', e);
      return { success: true };
    }
  },

  // ==================== v1.1.0 OCR 配额 API ====================

  /**
   * 消�?OCR 配额
   * @param {number} count - 消耗数�?
   * @param {string} imageUrlHash - 图片 URL 哈希（可选）
   * @returns {Promise<Object>} 消耗结�?
   */
  async consumeOCRQuota(count = 1, imageUrlHash = null) {
    const token = await getAuthToken();
    
    if (!token) {
      return {
        success: false,
        data: { success: false, reason: 'not_authenticated', message: 'Please log in to use OCR' }
      };
    }
    
    const response = await fetch(`${getBaseUrl()}/api/ocr/consume`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      },
      body: JSON.stringify({ count, image_url_hash: imageUrlHash })
    });
    
    return handleResponse(response, 'Consume OCR quota');
  },

  /**
   * 标记 OCR 交易完成
   * @param {string} transactionId - 交易 ID
   * @returns {Promise<Object>}
   */
  async completeOCRTransaction(transactionId) {
    const token = await getAuthToken();
    
    if (!token) {
      return { success: true }; // 静默成功
    }
    
    try {
      const response = await fetch(`${getBaseUrl()}/api/ocr/transaction/complete`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`
        },
        body: JSON.stringify({ transaction_id: transactionId })
      });
      
      return handleResponse(response, 'Complete OCR transaction');
    } catch (e) {
      log.warn('Complete OCR transaction failed:', e);
      return { success: true };
    }
  },

  /**
   * 标记 OCR 交易失败
   * @param {string} transactionId - 交易 ID
   * @param {string} errorMessage - 错误信息
   * @returns {Promise<Object>}
   */
  async failOCRTransaction(transactionId, errorMessage = 'Unknown error') {
    const token = await getAuthToken();
    
    if (!token) {
      return { success: false };
    }
    
    try {
      const response = await fetch(`${getBaseUrl()}/api/ocr/transaction/failed`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`
        },
        body: JSON.stringify({ transaction_id: transactionId, error_message: errorMessage })
      });
      
      return handleResponse(response, 'Fail OCR transaction');
    } catch (e) {
      log.warn('Fail OCR transaction failed:', e);
      return { success: false, error: e.message };
    }
  },

  /**
   * 退�?OCR 配额
   * @param {string} transactionId - 交易 ID
   * @returns {Promise<Object>}
   */
  async refundOCRQuota(transactionId) {
    const token = await getAuthToken();
    
    if (!token) {
      return { success: false };
    }
    
    try {
      const response = await fetch(`${getBaseUrl()}/api/ocr/refund`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`
        },
        body: JSON.stringify({ transaction_id: transactionId })
      });
      
      return handleResponse(response, 'Refund OCR quota');
    } catch (e) {
      log.warn('Refund OCR quota failed:', e);
      return { success: false, error: e.message };
    }
  },

  /**
   * 获取 OCR 配额信息
   * @returns {Promise<Object>}
   */
  async getOCRQuota() {
    const token = await getAuthToken();
    
    if (!token) {
      // 返回默认 Free 配额
      return {
        success: true,
        data: {
          daily_limit: 10,
          monthly_limit: 300,
          daily_used: 0,
          monthly_used: 0,
          daily_remaining: 10,
          monthly_remaining: 300,
          is_guest: true
        }
      };
    }
    
    const response = await fetch(`${getBaseUrl()}/api/ocr/quota`, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      }
    });
    
    return handleResponse(response, 'Get OCR quota');
  }
};
