/**
 * Ads Insight Pro - API Module
 * Handles Google Ads Transparency API calls
 */

import { createLogger, DEBUG_CONFIG } from './debug-config.js';

const log = createLogger('[API]', DEBUG_CONFIG.apiClient);

const API_URL = 'https://adstransparency.google.com/anji/_/rpc/SearchService/SearchCreatives';

// Region codes mapping
export const REGIONS = {
  'US': 2840,
  'DE': 2276,
  'GB': 2826,
  'FR': 2250,
  'JP': 2392,
  'AU': 2036,
  'CA': 2124,
  'IT': 2380,
  'ES': 2724,
  'NL': 2528,
  'BR': 2076,
  'IN': 2356,
  'KR': 2410,
  'MX': 2484,
  'RU': 2643
};

/**
 * Build request body for API call
 * @param {Object} options - Request options
 * @param {string} options.domain - Domain to search
 * @param {string} options.advertiserId - Advertiser ID to search
 * @param {number} options.region - Region code
 * @param {string} options.cursor - Pagination cursor
 * @param {number} options.pageSize - Number of results per page
 * @returns {Object} Request body
 */
export function buildRequestBody(options) {
  const { domain, advertiserId, region, cursor, pageSize = 100 } = options;
  
  const body = {
    "2": pageSize,
    "3": {},
    "7": { "1": 1, "2": 0, "3": 2344 }
  };
  
  // Search by domain
  if (domain) {
    body["3"]["12"] = { "1": domain, "2": true };
  }
  
  // Search by advertiser ID
  if (advertiserId) {
    body["3"]["1"] = advertiserId;
  }
  
  // Region filter
  if (region) {
    body["3"]["8"] = [region];
  }
  
  // Pagination cursor
  if (cursor) {
    body["4"] = cursor;
  }
  
  return body;
}

/**
 * Make API request
 * @param {Object} requestBody - Request body
 * @param {AbortSignal} signal - Abort signal for cancellation
 * @returns {Promise<Object>} Response data
 */
export async function makeRequest(requestBody, signal) {
  const response = await fetch(API_URL, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/x-www-form-urlencoded',
      'Accept': '*/*'
    },
    body: `f.req=${encodeURIComponent(JSON.stringify(requestBody))}`,
    signal
  });
  
  if (!response.ok) {
    if (response.status === 429) {
      throw new Error('429: Rate limited');
    }
    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
  }
  
  const text = await response.text();
  return parseResponse(text);
}

/**
 * Parse API response
 * @param {string} text - Raw response text
 * @returns {Object} Parsed response with ads and cursor
 */
export function parseResponse(text) {
  // Remove XSSI prefix
  const cleanText = text.replace(/^\)\]\}'\n/, '');
  const data = JSON.parse(cleanText);
  
  const ads = [];
  const items = data?.[1]?.[1] || [];
  
  for (const item of items) {
    const ad = parseAdItem(item);
    if (ad) ads.push(ad);
  }
  
  const cursor = data?.[1]?.[2] || null;
  const totalCount = data?.[1]?.[3] || 0;
  
  return { ads, cursor, totalCount };
}

/**
 * Parse single ad item from API response
 * @param {Object} item - Raw ad item
 * @returns {Object|null} Parsed ad object
 */
export function parseAdItem(item) {
  try {
    const ad = {
      advertiser_id: item["1"] || '',
      creative_id: item["2"] || '',
      advertiser_name: item["12"] || '',
      target_domain: item["14"] || '',
      format: getFormatName(item["4"]),
      format_code: item["4"],
      days_shown: item["13"] || 0
    };
    
    // Image URL - parse from HTML content
    try {
      const htmlContent = item["3"]?.["3"]?.["2"];
      if (htmlContent) {
        const imgMatch = htmlContent.match(/src="([^"]+)"/);
        if (imgMatch) {
          ad.image_url = imgMatch[1];
        }
      }
    } catch (e) {}
    
    // Preview URL
    try {
      ad.preview_url = item["3"]?.["1"]?.["4"] || '';
    } catch (e) {}
    
    // Start date
    try {
      const startTs = item["6"]?.["1"];
      if (startTs) {
        ad.start_date = formatTimestamp(startTs);
        ad.start_timestamp = startTs;
      }
    } catch (e) {}
    
    // End date
    try {
      const endTs = item["7"]?.["1"];
      if (endTs) {
        ad.end_date = formatTimestamp(endTs);
        ad.end_timestamp = endTs;
      }
    } catch (e) {}
    
    return ad;
  } catch (e) {
    log.error('Error parsing ad item:', e);
    return null;
  }
}

/**
 * Get format name from code
 * @param {number} code - Format code
 * @returns {string} Format name
 */
function getFormatName(code) {
  switch (code) {
    case 1: return 'TEXT';
    case 2: return 'IMAGE';
    case 3: return 'VIDEO';
    default: return 'UNKNOWN';
  }
}

/**
 * Format Unix timestamp to ISO date string
 * @param {number} timestamp - Unix timestamp in seconds
 * @returns {string} ISO date string (YYYY-MM-DD)
 */
function formatTimestamp(timestamp) {
  const date = new Date(timestamp * 1000);
  return date.toISOString().split('T')[0];
}
