﻿/**
 * Ads Insight Pro - Auth Module
 * Handles user authentication and license verification
 */

import { browserAPI } from './browser-polyfill.js';
import * as storage from './storage.js';
import { getApiUrl } from './config.js';
import { fetchWithTimeout, TIMEOUTS } from './fetch-with-timeout.js';
import { PLAN_LIMITS } from './constants.js';
import { createLogger, DEBUG_CONFIG } from './debug-config.js';

const log = createLogger('[Auth]', DEBUG_CONFIG.auth);

// Helper function for API requests with timeout
async function apiRequest(endpoint, options = {}) {
  const url = `${getApiUrl()}${endpoint}`;
  return fetchWithTimeout(url, options, TIMEOUTS.NORMAL);
}

/**
 * Get plan limits
 * @param {string} plan - Plan name
 * @returns {Object} Plan limits
 */
export function getPlanLimits(plan) {
  return PLAN_LIMITS[plan] || PLAN_LIMITS.free;
}

/**
 * Check if feature is available for plan
 * @param {string} plan - Plan name
 * @param {string} feature - Feature name
 * @returns {boolean}
 */
export function hasFeature(plan, feature) {
  const limits = getPlanLimits(plan);
  return limits.features.includes(feature);
}

/**
 * Verify license with server
 * @returns {Promise<Object>} Verification result
 */
export async function verifyLicense() {
  try {
    const token = await storage.getAuthToken();
    
    if (!token) {
      return { valid: false, plan: 'free', reason: 'no_token' };
    }
    
    const response = await apiRequest('/api/auth/status', {
      method: 'GET',
      headers: {
        'Authorization': `Bearer ${token}`
      }
    });
    
    if (!response.ok) {
      if (response.status === 401) {
        await storage.clearAuth();
        return { valid: false, plan: 'free', reason: 'expired' };
      }
      throw new Error(`HTTP ${response.status}`);
    }
    
    const data = await response.json();
    
    if (!data.success) {
      return { valid: false, plan: 'free', reason: 'invalid' };
    }
    
    // Update cached plan
    await storage.set('userPlan', data.data.user.plan);
    
    return {
      valid: true,
      plan: data.data.user.plan,
      expiresAt: data.data.user.expiresAt,
      quota: data.data.quota,
      features: PLAN_LIMITS[data.data.user.plan]?.features || []
    };
  } catch (e) {
    log.error('License verification failed:', e);
    
    // Offline mode - use cached plan
    const cachedPlan = await storage.get('userPlan', 'free');
    return {
      valid: true,
      plan: cachedPlan,
      offline: true
    };
  }
}

/**
 * Login user
 * @param {string} email - User email
 * @param {string} password - User password
 * @returns {Promise<Object>} Login result
 */
export async function login(email, password) {
  const response = await apiRequest('/api/auth/login', {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json'
    },
    body: JSON.stringify({ email, password, rememberMe: true })
  });
  
  if (!response.ok) {
    const error = await response.json();
    throw new Error(error.message || 'Login failed');
  }
  
  const data = await response.json();
  
  if (!data.success) {
    throw new Error(data.error?.message || 'Login failed');
  }
  
  // Save auth token
  await storage.saveAuthToken(data.data.token);
  await storage.set('userPlan', data.data.user.plan);
  await storage.set('userEmail', email);
  await storage.set('userId', data.data.user.id);
  
  // Bind device after login
  await bindDevice();
  
  // Link install record to user (for conversion tracking)
  try {
    const { installId } = await browserAPI.storage.local.get('installId');
    if (installId) {
      await apiRequest('/api/stats/link', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ installId, userId: data.data.user.id })
      });
    }
  } catch (e) {
    // Non-critical, ignore errors
  }
  
  return {
    success: true,
    plan: data.data.user.plan,
    email: email
  };
}

/**
 * Logout user
 * @returns {Promise<void>}
 */
export async function logout() {
  await storage.clearAuth();
  await storage.set('userPlan', 'free');
  await storage.remove('userEmail');
}

/**
 * Register new user
 * @param {string} email - User email
 * @param {string} password - User password
 * @returns {Promise<Object>} Registration result
 */
export async function register(email, password) {
  const response = await apiRequest('/api/auth/register', {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json'
    },
    body: JSON.stringify({ email, password })
  });
  
  const data = await response.json();
  
  if (!data.success) {
    throw new Error(data.error?.message || 'Registration failed');
  }
  
  return {
    success: true,
    message: data.data?.message || 'Please check your email to verify your account'
  };
}

/**
 * Request password reset
 * @param {string} email - User email
 * @returns {Promise<Object>} Result
 */
export async function requestPasswordReset(email) {
  const response = await apiRequest('/api/auth/forgot-password', {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json'
    },
    body: JSON.stringify({ email })
  });
  
  const data = await response.json();
  
  if (!data.success) {
    throw new Error(data.error?.message || 'Request failed');
  }
  
  return {
    success: true,
    message: 'Password reset link sent to your email'
  };
}

/**
 * Check if user can perform download (legacy - use permissionsManager instead)
 * @deprecated Use permissionsManager.checkQuota() for v1.1.0+
 * @param {number} adsCount - Number of ads to export
 * @returns {Promise<Object>} Check result
 */
export async function canDownload(adsCount = 0) {
  const { plan, dailyUsedAds = 0 } = await storage.getUserPlan();
  const limits = getPlanLimits(plan);
  
  const remaining = limits.dailyQuota - dailyUsedAds;
  
  if (adsCount > remaining) {
    return {
      allowed: false,
      reason: 'quota_exceeded',
      message: `Insufficient quota. You need ${adsCount} ads but only have ${remaining} remaining.`,
      remaining,
      limit: limits.dailyQuota,
      needed: adsCount
    };
  }
  
  if (adsCount > limits.perRequestLimit) {
    return {
      allowed: false,
      reason: 'per_request_exceeded',
      message: `Maximum ${limits.perRequestLimit} ads per export for ${plan} plan.`,
      remaining,
      limit: limits.perRequestLimit,
      needed: adsCount
    };
  }
  
  return {
    allowed: true,
    remaining,
    limit: limits.dailyQuota,
    perRequestLimit: limits.perRequestLimit,
    plan
  };
}

/**
 * Record download usage
 * @param {Object} downloadInfo - Download information
 * @returns {Promise<void>}
 */
export async function recordDownload(downloadInfo) {
  await storage.incrementUsage();
  await storage.addToHistory(downloadInfo);
  
  // Try to sync with server (non-blocking)
  try {
    const token = await storage.getAuthToken();
    if (token) {
      apiRequest('/api/usage/record', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`
        },
        body: JSON.stringify({
          action: 'download',
          domain: downloadInfo.domain,
          pages: downloadInfo.pages,
          adsCount: downloadInfo.adsCount
        })
      }).catch(() => {});
    }
  } catch (e) {
    // Ignore errors - usage is tracked locally
  }
}

/**
 * Generate friendly device name from user agent
 * @param {string} userAgent - User agent string
 * @param {string} platform - Platform string
 * @returns {string} Friendly device name
 */
function generateDeviceName(userAgent, platform) {
  // 识别浏览�?
  let browser = 'Unknown Browser';
  if (userAgent.includes('Edg/')) {
    browser = 'Edge';
  } else if (userAgent.includes('Chrome/') && !userAgent.includes('Edg/')) {
    browser = 'Chrome';
  } else if (userAgent.includes('Firefox/')) {
    browser = 'Firefox';
  } else if (userAgent.includes('Safari/') && !userAgent.includes('Chrome/')) {
    browser = 'Safari';
  } else if (userAgent.includes('OPR/') || userAgent.includes('Opera/')) {
    browser = 'Opera';
  }
  
  // 识别操作系统
  let os = platform || 'Unknown OS';
  if (userAgent.includes('Windows NT 10.0')) {
    os = 'Windows 10/11';
  } else if (userAgent.includes('Windows NT 6.3')) {
    os = 'Windows 8.1';
  } else if (userAgent.includes('Windows NT 6.2')) {
    os = 'Windows 8';
  } else if (userAgent.includes('Windows NT 6.1')) {
    os = 'Windows 7';
  } else if (userAgent.includes('Windows')) {
    os = 'Windows';
  } else if (userAgent.includes('Mac OS X')) {
    os = 'macOS';
  } else if (userAgent.includes('Linux')) {
    os = 'Linux';
  } else if (userAgent.includes('Android')) {
    os = 'Android';
  } else if (userAgent.includes('iOS') || userAgent.includes('iPhone') || userAgent.includes('iPad')) {
    os = 'iOS';
  }
  
  return `${browser} on ${os}`;
}

/**
 * Generate device fingerprint
 * @returns {Promise<Object>} Fingerprint and components
 */
export async function generateFingerprint() {
  // Check if running in Service Worker (no DOM access)
  if (typeof window === 'undefined' || typeof document === 'undefined' || typeof screen === 'undefined') {
    return {
      fingerprint: 'sw-' + Date.now(),
      components: { userAgent: navigator.userAgent, env: 'service-worker' }
    };
  }

  const components = {
    userAgent: navigator.userAgent,
    language: navigator.language,
    platform: navigator.platform || 'unknown',
    screenResolution: `${screen.width}x${screen.height}`,
    colorDepth: String(screen.colorDepth),
    timezone: Intl.DateTimeFormat().resolvedOptions().timeZone,
    timezoneOffset: String(new Date().getTimezoneOffset()),
  };
  
  // Canvas fingerprint
  try {
    const canvas = document.createElement('canvas');
    const ctx = canvas.getContext('2d');
    ctx.textBaseline = 'top';
    ctx.font = '14px Arial';
    ctx.fillText('AdsInsightPro', 2, 2);
    components.canvas = canvas.toDataURL().slice(-50);
  } catch (e) {
    components.canvas = 'unavailable';
  }
  
  // WebGL fingerprint
  try {
    const canvas = document.createElement('canvas');
    const gl = canvas.getContext('webgl');
    if (gl) {
      const debugInfo = gl.getExtension('WEBGL_debug_renderer_info');
      if (debugInfo) {
        components.webglVendor = gl.getParameter(debugInfo.UNMASKED_VENDOR_WEBGL);
        components.webglRenderer = gl.getParameter(debugInfo.UNMASKED_RENDERER_WEBGL);
      }
    }
  } catch (e) {
    components.webglVendor = 'unavailable';
  }
  
  // Generate hash from components
  const str = Object.values(components).join('|');
  const encoder = new TextEncoder();
  const data = encoder.encode(str);
  const hashBuffer = await crypto.subtle.digest('SHA-256', data);
  const hashArray = Array.from(new Uint8Array(hashBuffer));
  const fingerprint = hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
  
  return { fingerprint, components };
}

/**
 * Bind current device to user account
 * @returns {Promise<Object>} Bind result
 */
export async function bindDevice() {
  try {
    const token = await storage.getAuthToken();
    if (!token) {
      return { success: false, reason: 'no_token' };
    }
    
    const { fingerprint, components } = await generateFingerprint();
    
    // 生成更友好的设备名称
    const deviceName = generateDeviceName(components.userAgent, components.platform);
    
    const response = await apiRequest('/api/auth/bind-device', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      },
      body: JSON.stringify({
        deviceFingerprint: fingerprint,
        deviceName: deviceName,
        fingerprintComponents: components
      })
    });
    
    const data = await response.json();
    
    if (!data.success) {
      return { success: false, error: data.error };
    }
    
    // Save device ID locally
    await storage.set('deviceId', data.data.deviceId);
    
    return { success: true, deviceId: data.data.deviceId };
  } catch (e) {
    log.error('Device bind failed:', e);
    return { success: false, error: e.message };
  }
}

/**
 * Get list of bound devices
 * @returns {Promise<Array>} Device list
 */
export async function getDevices() {
  try {
    const token = await storage.getAuthToken();
    if (!token) return [];
    
    const response = await apiRequest('/api/auth/devices', {
      method: 'GET',
      headers: {
        'Authorization': `Bearer ${token}`
      }
    });
    
    const data = await response.json();
    return data.success ? data.data.devices : [];
  } catch (e) {
    log.error('Get devices failed:', e);
    return [];
  }
}

/**
 * Unbind a device
 * @param {string} deviceId - Device ID to unbind
 * @returns {Promise<Object>} Result
 */
export async function unbindDevice(deviceId) {
  try {
    const token = await storage.getAuthToken();
    if (!token) {
      throw new Error('Not logged in');
    }
    
    const response = await apiRequest('/api/auth/unbind-device', {
      method: 'DELETE',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      },
      body: JSON.stringify({ deviceId })
    });
    
    const data = await response.json();
    
    if (!data.success) {
      throw new Error(data.error?.message || 'Unbind failed');
    }
    
    return { success: true };
  } catch (e) {
    log.error('Unbind device failed:', e);
    throw e;
  }
}
