/**
 * Browser API Polyfill for Firefox/Chrome Compatibility
 * Provides unified Promise-based API for cross-browser extensions
 * v1.2.2 - Firefox migration
 */

/**
 * Unified browser API that works in both Firefox (browser.*) and Chrome (chrome.*)
 * Firefox natively supports browser.* with Promises
 * Chrome uses chrome.* with callbacks, so we wrap them
 */
export const browserAPI = (() => {
  // Firefox: Use native browser.* API (Promise-based)
  if (typeof browser !== 'undefined' && browser.runtime) {
    return browser;
  }
  
  // Chrome: Wrap chrome.* API to return Promises
  if (typeof chrome === 'undefined' || !chrome.runtime) {
    throw new Error('Neither browser nor chrome API is available');
  }
  
  const wrapped = {};
  
  // ============== RUNTIME ==============
  wrapped.runtime = {
    sendMessage: (message, options) => {
      return new Promise((resolve) => {
        chrome.runtime.sendMessage(message, options, (response) => {
          // Ignore lastError in sendMessage - it's expected when no listeners
          if (chrome.runtime.lastError) {
            // Silent ignore
          }
          resolve(response);
        });
      });
    },
    
    onMessage: chrome.runtime.onMessage,
    
    getURL: chrome.runtime.getURL.bind(chrome.runtime),
    
    getManifest: chrome.runtime.getManifest.bind(chrome.runtime),
    
    id: chrome.runtime.id,
    
    lastError: chrome.runtime.lastError,
    
    connect: chrome.runtime.connect.bind(chrome.runtime),
    
    onConnect: chrome.runtime.onConnect,
    
    openOptionsPage: () => {
      return new Promise((resolve, reject) => {
        chrome.runtime.openOptionsPage(() => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
          } else {
            resolve();
          }
        });
      });
    }
  };
  
  // ============== STORAGE ==============
  wrapped.storage = {
    local: {
      get: (keys) => {
        return new Promise((resolve, reject) => {
          chrome.storage.local.get(keys, (items) => {
            if (chrome.runtime.lastError) {
              reject(chrome.runtime.lastError);
            } else {
              resolve(items);
            }
          });
        });
      },
      
      set: (items) => {
        return new Promise((resolve, reject) => {
          chrome.storage.local.set(items, () => {
            if (chrome.runtime.lastError) {
              reject(chrome.runtime.lastError);
            } else {
              resolve();
            }
          });
        });
      },
      
      remove: (keys) => {
        return new Promise((resolve, reject) => {
          chrome.storage.local.remove(keys, () => {
            if (chrome.runtime.lastError) {
              reject(chrome.runtime.lastError);
            } else {
              resolve();
            }
          });
        });
      },
      
      clear: () => {
        return new Promise((resolve, reject) => {
          chrome.storage.local.clear(() => {
            if (chrome.runtime.lastError) {
              reject(chrome.runtime.lastError);
            } else {
              resolve();
            }
          });
        });
      }
    },
    
    sync: {
      get: (keys) => {
        return new Promise((resolve, reject) => {
          chrome.storage.sync.get(keys, (items) => {
            if (chrome.runtime.lastError) {
              reject(chrome.runtime.lastError);
            } else {
              resolve(items);
            }
          });
        });
      },
      
      set: (items) => {
        return new Promise((resolve, reject) => {
          chrome.storage.sync.set(items, () => {
            if (chrome.runtime.lastError) {
              reject(chrome.runtime.lastError);
            } else {
              resolve();
            }
          });
        });
      },
      
      remove: (keys) => {
        return new Promise((resolve, reject) => {
          chrome.storage.sync.remove(keys, () => {
            if (chrome.runtime.lastError) {
              reject(chrome.runtime.lastError);
            } else {
              resolve();
            }
          });
        });
      }
    },
    
    onChanged: chrome.storage.onChanged
  };
  
  // ============== TABS ==============
  wrapped.tabs = {
    query: (queryInfo) => {
      return new Promise((resolve, reject) => {
        chrome.tabs.query(queryInfo, (tabs) => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
          } else {
            resolve(tabs);
          }
        });
      });
    },
    
    get: (tabId) => {
      return new Promise((resolve, reject) => {
        chrome.tabs.get(tabId, (tab) => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
          } else {
            resolve(tab);
          }
        });
      });
    },
    
    sendMessage: (tabId, message, options) => {
      return new Promise((resolve) => {
        chrome.tabs.sendMessage(tabId, message, options, (response) => {
          // Ignore lastError in sendMessage
          if (chrome.runtime.lastError) {
            // Silent ignore
          }
          resolve(response);
        });
      });
    },
    
    create: (createProperties) => {
      return new Promise((resolve, reject) => {
        chrome.tabs.create(createProperties, (tab) => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
          } else {
            resolve(tab);
          }
        });
      });
    },
    
    update: (tabId, updateProperties) => {
      return new Promise((resolve, reject) => {
        chrome.tabs.update(tabId, updateProperties, (tab) => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
          } else {
            resolve(tab);
          }
        });
      });
    },
    
    reload: (tabId, reloadProperties) => {
      return new Promise((resolve, reject) => {
        chrome.tabs.reload(tabId, reloadProperties, () => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
          } else {
            resolve();
          }
        });
      });
    },
    
    onUpdated: chrome.tabs.onUpdated,
    onActivated: chrome.tabs.onActivated,
    onRemoved: chrome.tabs.onRemoved
  };
  
  // ============== DOWNLOADS ==============
  wrapped.downloads = {
    download: (options) => {
      return new Promise((resolve, reject) => {
        chrome.downloads.download(options, (downloadId) => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
          } else {
            resolve(downloadId);
          }
        });
      });
    },
    
    search: (query) => {
      return new Promise((resolve, reject) => {
        chrome.downloads.search(query, (items) => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
          } else {
            resolve(items);
          }
        });
      });
    },
    
    cancel: (downloadId) => {
      return new Promise((resolve, reject) => {
        chrome.downloads.cancel(downloadId, () => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
          } else {
            resolve();
          }
        });
      });
    },
    
    onChanged: chrome.downloads.onChanged
  };
  
  // ============== SCRIPTING ==============
  wrapped.scripting = {
    executeScript: (injection) => {
      return new Promise((resolve, reject) => {
        chrome.scripting.executeScript(injection, (results) => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
          } else {
            resolve(results);
          }
        });
      });
    },
    
    insertCSS: (injection) => {
      return new Promise((resolve, reject) => {
        chrome.scripting.insertCSS(injection, () => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
          } else {
            resolve();
          }
        });
      });
    },
    
    removeCSS: (injection) => {
      return new Promise((resolve, reject) => {
        chrome.scripting.removeCSS(injection, () => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
          } else {
            resolve();
          }
        });
      });
    }
  };
  
  // ============== ALARMS ==============
  if (chrome.alarms) {
    wrapped.alarms = {
      create: chrome.alarms.create.bind(chrome.alarms),
      
      get: (name) => {
        return new Promise((resolve, reject) => {
          chrome.alarms.get(name, (alarm) => {
            if (chrome.runtime.lastError) {
              reject(chrome.runtime.lastError);
            } else {
              resolve(alarm);
            }
          });
        });
      },
      
      getAll: () => {
        return new Promise((resolve, reject) => {
          chrome.alarms.getAll((alarms) => {
            if (chrome.runtime.lastError) {
              reject(chrome.runtime.lastError);
            } else {
              resolve(alarms);
            }
          });
        });
      },
      
      clear: (name) => {
        return new Promise((resolve, reject) => {
          chrome.alarms.clear(name, (wasCleared) => {
            if (chrome.runtime.lastError) {
              reject(chrome.runtime.lastError);
            } else {
              resolve(wasCleared);
            }
          });
        });
      },
      
      clearAll: () => {
        return new Promise((resolve, reject) => {
          chrome.alarms.clearAll((wasCleared) => {
            if (chrome.runtime.lastError) {
              reject(chrome.runtime.lastError);
            } else {
              resolve(wasCleared);
            }
          });
        });
      },
      
      onAlarm: chrome.alarms.onAlarm
    };
  }
  
  // ============== ACTION (MV3) ==============
  if (chrome.action) {
    wrapped.action = {
      setIcon: (details) => {
        return new Promise((resolve, reject) => {
          chrome.action.setIcon(details, () => {
            if (chrome.runtime.lastError) {
              reject(chrome.runtime.lastError);
            } else {
              resolve();
            }
          });
        });
      },
      
      setBadgeText: (details) => {
        return new Promise((resolve, reject) => {
          chrome.action.setBadgeText(details, () => {
            if (chrome.runtime.lastError) {
              reject(chrome.runtime.lastError);
            } else {
              resolve();
            }
          });
        });
      },
      
      setBadgeBackgroundColor: (details) => {
        return new Promise((resolve, reject) => {
          chrome.action.setBadgeBackgroundColor(details, () => {
            if (chrome.runtime.lastError) {
              reject(chrome.runtime.lastError);
            } else {
              resolve();
            }
          });
        });
      },
      
      onClicked: chrome.action.onClicked
    };
  }
  
  return wrapped;
})();

/**
 * Helper function to check if we're running in Firefox
 */
export function isFirefox() {
  return typeof browser !== 'undefined' && browser.runtime;
}

/**
 * Helper function to check if we're running in Chrome
 */
export function isChrome() {
  return typeof chrome !== 'undefined' && chrome.runtime && !isFirefox();
}
