/**
 * Ads Insight Pro - Configuration Module
 * Environment-aware configuration
 */

import { browserAPI } from './browser-polyfill.js';
import { createLogger, DEBUG_CONFIG } from './debug-config.js';

const log = createLogger('[Config]', DEBUG_CONFIG.background);

// Detect environment based on manifest name (Dev Mock) or version
const manifest = browserAPI.runtime.getManifest();
const isDevelopment = manifest.name.includes('Dev') || manifest.name.includes('Mock') || 
                      manifest.version.includes('dev') || manifest.version.includes('beta');

// Environment configuration
const ENVIRONMENTS = {
  development: {
    API_URL: 'http://localhost:8787',
    ENVIRONMENT: 'development',
    DEBUG: true,
  },
  production: {
    API_URL: 'https://api.adsinsightpro.com',
    ENVIRONMENT: 'production',
    DEBUG: false,
  },
};

// Select current environment
const currentEnv = isDevelopment ? 'development' : 'production';
const config = ENVIRONMENTS[currentEnv];

/**
 * Get configuration value
 * @param {string} key - Configuration key
 * @returns {*} Configuration value
 */
export function getConfig(key) {
  return config[key];
}

/**
 * Get API base URL
 * @returns {string} API base URL
 */
export function getApiUrl() {
  return config.API_URL;
}

/**
 * Check if in development mode
 * @returns {boolean} True if in development
 */
export function isDev() {
  return config.DEBUG;
}

/**
 * Get current environment name
 * @returns {string} Environment name
 */
export function getEnvironment() {
  return config.ENVIRONMENT;
}

// Export config object for direct access
export const CONFIG = {
  ...config,
  // Extension info
  VERSION: manifest.version,
  NAME: manifest.name,
};

// Log current environment (only in development)
if (isDev()) {
  log.info('[Config] Environment:', getEnvironment());
  log.info('[Config] API URL:', getApiUrl());
  log.info('[Config] Version:', CONFIG.VERSION);
}
