/**
 * Ads Insight Pro - Constants
 * Centralized constants to avoid magic numbers
 */

// ============== PLAN LIMITS ==============
// v1.1.0: Changed from download-count to ads-quota system
// These are fallback values; actual limits are fetched from server

export const PLAN_LIMITS = {
  free: {
    dailyQuota: 100,           // 100 ads per day
    perRequestLimit: 100,      // max 100 ads per export
    maxPages: 5,
    maxDevices: 1,
    features: ['csv', 'advertiser_export']
  },
  pro: {
    dailyQuota: 5000,          // 5,000 ads per day
    perRequestLimit: 1000,     // max 1,000 ads per export
    maxPages: 100,
    maxDevices: 3,
    features: ['csv', 'json', 'excel', 'enhanced_fields', 'ad_content_export', 'campaign_info_export', 'advertiser_export']
  },
  enterprise: {
    dailyQuota: 50000,         // 50,000 ads per day
    perRequestLimit: 3000,     // max 3,000 ads per export
    maxPages: 500,
    maxDevices: 10,
    features: ['csv', 'json', 'excel', 'enhanced_fields', 'ad_content_export', 'campaign_info_export', 'advertiser_export', 'video_extraction']
  }
};

// ============== TIME CONSTANTS ==============

export const TIME = {
  // Milliseconds
  SECOND: 1000,
  MINUTE: 60 * 1000,
  HOUR: 60 * 60 * 1000,
  DAY: 24 * 60 * 60 * 1000,
  WEEK: 7 * 24 * 60 * 60 * 1000,
  
  // Specific durations
  AUTH_TOKEN_EXPIRY: 7 * 24 * 60 * 60 * 1000, // 7 days
  RATING_COOLDOWN_DAYS: 7,
  MIN_INSTALL_DAYS_FOR_RATING: 7,
  PROXY_TEST_FEEDBACK_DELAY: 2000, // 2 seconds
};

// ============== FIELD DEFINITIONS (v1.1.0) ==============

// Basic fields - available to all plans (Free, Pro, Enterprise)
export const BASIC_FIELDS = [
  'advertiser_id',
  'advertiser_name', 
  'creative_id',
  'target_domain',
  'format',
  'query_region'
];

// Enhanced fields - Pro+ only
export const ENHANCED_FIELDS = [
  'image_url',
  'preview_url'
];

// Campaign info fields - Pro+ only (投放信息导出)
export const CAMPAIGN_INFO_FIELDS = [
  'start_date',
  'end_date',
  'days_shown',
  'regions'
];

// Ad Content Analysis fields - Pro+ only
export const AD_CONTENT_FIELDS = [
  'ad_domain',
  'ad_title',
  'ad_description',
  'confidence'
];

// Video extraction fields - Enterprise only
export const VIDEO_FIELDS = [
  'video_url'
];

// All fields combined
export const ALL_FIELDS = [
  ...BASIC_FIELDS,
  ...ENHANCED_FIELDS,
  ...CAMPAIGN_INFO_FIELDS,
  ...AD_CONTENT_FIELDS,
  ...VIDEO_FIELDS
];

// Field descriptions (bilingual)
export const FIELD_DESCRIPTIONS = {
  // Basic fields
  advertiser_id: 'Advertiser ID - Unique identifier / 广告主 ID',
  advertiser_name: 'Advertiser Name - Company name / 广告主名称',
  creative_id: 'Creative ID - Ad unique identifier / 创意 ID',
  target_domain: 'Target Domain - Ad destination website / 目标域名',
  format: 'Format - TEXT/IMAGE/VIDEO / 广告格式',
  query_region: 'Query Region - Search region filter / 搜索地区',
  // Enhanced fields (Pro+)
  image_url: 'Image URL - Static ad image link / 图片 URL',
  preview_url: 'Preview URL - Dynamic ad preview link / 预览 URL',
  // Campaign info (Pro+)
  start_date: 'Start Date - First shown date / 开始日期',
  end_date: 'End Date - Last shown date / 结束日期',
  days_shown: 'Days Shown - Duration of ad display / 展示天数',
  regions: 'Regions - Targeted regions / 投放地区',
  // Ad Content Analysis (Pro+)
  ad_domain: 'Ad Domain - Domain extracted from ad / 广告内域名',
  ad_title: 'Ad Title - Title text from ad / 广告标题',
  ad_description: 'Ad Description - Description text / 广告描述',
  confidence: 'Confidence - Extraction confidence score / 置信度',
  // Video extraction (Enterprise)
  video_url: 'Video URL - YouTube video link / 视频链接'
};

// Field permission mapping
// v1.2.0: AD_CONTENT_FIELDS available to all users (OCR with quota limits)
export const FIELD_PERMISSIONS = {
  free: [...BASIC_FIELDS, ...AD_CONTENT_FIELDS],
  pro: [...BASIC_FIELDS, ...ENHANCED_FIELDS, ...CAMPAIGN_INFO_FIELDS, ...AD_CONTENT_FIELDS],
  enterprise: ALL_FIELDS
};

// ============== API CONSTANTS ==============

export const API = {
  PAGE_SIZE: 100, // Ads per page from Google API
  ESTIMATED_ADS_PER_PAGE: 100,
  MAX_HISTORY_ITEMS: 50,
};

// ============== LOCAL DEFAULTS (v1.2.0) ==============
// These are fallback values for remote configuration
// Ensures extension works offline or when remote config unavailable

export const LOCAL_DEFAULTS = {
  // Rate limiting configuration
  rate_limits: {
    normal_delay_ms: 2000,       // Normal request interval
    cautious_delay_ms: 5000,     // Cautious mode interval
    recovery_delay_ms: 30000,    // Recovery mode interval after 429
    safe_hourly_limit: 500,      // Safe requests per hour
    safe_daily_limit: 5000       // Safe requests per day
  },
  
  // OCR configuration
  ocr_config: {
    timeout_ms: 15000,           // OCR request timeout
    max_retries: 2,              // Max retry attempts
    retry_delay_ms: 1500,        // Delay between retries (to avoid rate limit)
    concurrent: 2,               // Concurrent OCR requests (2-3 safe for free tier)
    request_delay_ms: 500,       // Delay between batch requests
    cache_expiry_days: 30,       // Local cache expiry
    refund_window_seconds: 60,   // Quota refund window
    supported_languages: ['eng', 'chs', 'cht']
  },
  
  // Fetch timeout configuration
  timeouts: {
    short_ms: 10000,             // Short operations
    normal_ms: 30000,            // Normal API requests
    long_ms: 60000,              // Large data transfers
    very_long_ms: 120000         // Export operations
  },
  
  // Feature flags
  feature_flags: {
    ocr_enabled: true,
    video_extraction_enabled: true,
    cache_enabled: true,
    debug_mode: false,
    maintenance_mode: false
  },
  
  // Version control
  version_control: {
    min_supported_version: '1.0.0',
    force_update: false
  },
  
  // Field metadata
  field_metadata: {
    export_order: ALL_FIELDS,
    single_ad_exclude: ['target_domain', 'preview_url', 'ocr_status', 'confidence']
  },
  
  // Video extraction
  video_extraction: {
    concurrent: 3,               // Concurrent video processing
    timeout_ms: 30000            // Video extraction timeout
  },
  
  // Cache configuration
  cache_config: {
    permissions_ttl_ms: 5 * 60 * 1000,        // 5 minutes
    client_config_ttl_ms: 60 * 60 * 1000,     // 1 hour
    stale_config_ttl_ms: 24 * 60 * 60 * 1000  // 24 hours (stale but usable)
  },
  
  // History limits
  history: {
    max_items: 50
  }
};

// ============== UI CONSTANTS ==============

export const UI = {
  PROGRESS_PERCENTAGE_MAX: 100,
  Z_INDEX_RATING_BANNER: 1000,
};

// ============== TIME INTERVALS ==============

export const INTERVALS = {
  MINUTE_MS: 60000,
  HOUR_MS: 3600000,
  DAY_MS: 86400000,
};

// ============== CONVERSION HELPERS ==============

/**
 * Convert days to milliseconds
 */
export function daysToMs(days) {
  return days * TIME.DAY;
}

/**
 * Convert milliseconds to days
 */
export function msToDays(ms) {
  return ms / TIME.DAY;
}

/**
 * Convert timestamp to days ago
 */
export function timestampToDaysAgo(timestamp) {
  return (Date.now() - timestamp) / TIME.DAY;
}
