﻿import { browserAPI } from './browser-polyfill.js';
/**
 * 统一的调试配置
 * 
 * 生产环境：将 DEBUG_MODE 设为 false
 * 开发/调试：将 DEBUG_MODE 设为 true
 */

// ============== 全局调试开关 ==============
// ✅ 生产环境配置
export const DEBUG_MODE = false;

// ============== 模块级别调试开关 ==============
// 可以单独控制每个模块的日志输出
export const DEBUG_CONFIG = {
  // Service Worker
  serviceWorker: DEBUG_MODE,
  
  // URL Extractor (视频提取)
  urlExtractor: DEBUG_MODE,
  
  // Usage Monitor (使用量监控)
  usageMonitor: DEBUG_MODE,
  
  // API Client
  apiClient: DEBUG_MODE,
  
  // Auth
  auth: DEBUG_MODE,
  
  // Popup
  popup: DEBUG_MODE,
  
  // Content Script
  content: DEBUG_MODE,
  
  // Background
  background: DEBUG_MODE
};

// ============== 日志工具函数 ==============

/**
 * 创建日志记录器
 * @param {string} prefix - 日志前缀（如 '[ServiceWorker]'）
 * @param {boolean} enabled - 是否启用日志
 * @returns {Object} - 日志记录器对象
 */
export function createLogger(prefix, enabled = DEBUG_MODE) {
  const noop = () => {};
  
  return {
    debug: enabled ? (...args) => console.log(`${prefix}[DEBUG]`, ...args) : noop,
    info: enabled ? (...args) => console.log(`${prefix}[INFO]`, ...args) : noop,
    warn: enabled ? (...args) => console.warn(`${prefix}[WARN]`, ...args) : noop,
    error: (...args) => console.error(`${prefix}[ERROR]`, ...args), // 错误始终显示
    success: enabled ? (...args) => console.log(`${prefix}[SUCCESS]`, ...args) : noop,
    
    // 带样式的日志（用于重要信息）
    highlight: enabled ? (...args) => console.log(`%c${prefix}[HIGHLIGHT]`, 'color: #4285f4; font-weight: bold', ...args) : noop,
    
    // 分组日志
    group: enabled ? (label) => console.group(`${prefix} ${label}`) : noop,
    groupEnd: enabled ? () => console.groupEnd() : noop,
    
    // 表格日志
    table: enabled ? (data) => console.table(data) : noop,
    
    // 计时器
    time: enabled ? (label) => console.time(`${prefix} ${label}`) : noop,
    timeEnd: enabled ? (label) => console.timeEnd(`${prefix} ${label}`) : noop
  };
}

/**
 * 获取环境信息
 */
export function getEnvironmentInfo() {
  const manifest = browserAPI.runtime.getManifest();
  return {
    name: manifest.name,
    version: manifest.version,
    debugMode: DEBUG_MODE,
    timestamp: new Date().toISOString()
  };
}

/**
 * 打印环境信息（启动时调用）
 */
export function logEnvironmentInfo(logger) {
  const info = getEnvironmentInfo();
  logger.highlight('Environment Info:', info);
  
  if (DEBUG_MODE) {
    logger.warn('⚠️ DEBUG MODE IS ENABLED - Remember to disable before production!');
  } else {
    logger.info('✅ Production mode - Debug logs are disabled');
  }
}
