﻿/**
 * Ads Insight Pro - Export Module
 * Handles data export to various formats
 */

import { browserAPI } from './browser-polyfill.js';
import { fetchWithTimeout, TIMEOUTS } from './fetch-with-timeout.js';
import { createLogger, DEBUG_CONFIG } from './debug-config.js';

const log = createLogger('[Export]', DEBUG_CONFIG.background);

/**
 * Export ads to CSV format
 * @param {Array} ads - Array of ad objects
 * @param {Object} options - Export options
 * @param {boolean} options.withBOM - Add UTF-8 BOM for Excel compatibility (default: true)
 * @returns {string} CSV content
 */
export function toCSV(ads, options = {}) {
  const { withBOM = true } = options;
  
  if (!ads || ads.length === 0) {
    return '';
  }
  
  // Get headers from first ad
  const headers = Object.keys(ads[0]);
  
  // Build CSV with optional UTF-8 BOM for Excel compatibility
  // BOM: \uFEFF tells Excel to interpret the file as UTF-8
  let csv = withBOM ? '\uFEFF' : '';
  csv += headers.join(',') + '\n';
  
  for (const ad of ads) {
    const row = headers.map(header => {
      const value = ad[header];
      
      if (value === null || value === undefined) {
        return '';
      }
      
      let strValue = String(value);
      
      // v1.2.1: Clean up problematic characters that can break CSV parsing
      // Remove or replace control characters (except normal whitespace)
      strValue = strValue
        .replace(/\r\n/g, ' ')  // Windows newlines -> space
        .replace(/\r/g, ' ')    // CR -> space
        .replace(/\n/g, ' ')    // LF -> space
        .replace(/\t/g, ' ')    // TAB -> space
        .replace(/[\x00-\x08\x0B\x0C\x0E-\x1F]/g, ''); // Other control chars -> remove
      
      // Escape if contains comma, quote (for safety, always quote fields with special chars)
      if (strValue.includes(',') || strValue.includes('"') || strValue.includes(' ')) {
        return `"${strValue.replace(/"/g, '""')}"`;
      }
      
      return strValue;
    });
    
    csv += row.join(',') + '\n';
  }
  
  return csv;
}

/**
 * Export ads to JSON format
 * @param {Array} ads - Array of ad objects
 * @param {Object} options - Export options
 * @returns {string} JSON content
 */
export function toJSON(ads, options = {}) {
  const { pretty = true } = options;
  
  const data = {
    exportDate: new Date().toISOString(),
    totalAds: ads.length,
    ads: ads
  };
  
  return pretty ? JSON.stringify(data, null, 2) : JSON.stringify(data);
}

/**
 * Generate filename for export
 * @param {Object} options - Filename options
 * @param {string} options.domain - Domain name
 * @param {string} options.advertiserId - Advertiser ID
 * @param {string} options.extension - File extension
 * @returns {string} Generated filename
 */
export function generateFilename(options = {}) {
  const { domain, advertiserId, extension = 'csv' } = options;
  
  // Use domain or advertiser ID as base
  let base = domain || advertiserId || 'ads_export';
  
  // Clean up domain (replace dots with underscores)
  base = base.replace(/\./g, '_').replace(/[^a-zA-Z0-9_-]/g, '');
  
  // Add timestamp
  const now = new Date();
  const timestamp = now.toISOString()
    .replace(/[-:]/g, '')
    .replace('T', '_')
    .split('.')[0];
  
  return `${base}_${timestamp}.${extension}`;
}

/**
 * Download file in browser
 * @param {string} content - File content
 * @param {string} filename - Filename
 * @param {string} mimeType - MIME type
 */
export function downloadFile(content, filename, mimeType = 'text/csv') {
  const blob = new Blob([content], { type: mimeType });
  const url = URL.createObjectURL(blob);
  
  const a = document.createElement('a');
  a.href = url;
  a.download = filename;
  document.body.appendChild(a);
  a.click();
  document.body.removeChild(a);
  
  URL.revokeObjectURL(url);
}

/**
 * Download using Chrome downloads API (for extensions)
 * @param {string} content - File content
 * @param {string} filename - Filename
 * @param {string} mimeType - MIME type
 * @returns {Promise<number>} Download ID
 */
export async function chromeDownload(content, filename, mimeType = 'text/csv') {
  const blob = new Blob([content], { type: mimeType });
  const url = URL.createObjectURL(blob);
  
  return new Promise((resolve, reject) => {
    browserAPI.downloads.download({
      url: url,
      filename: filename,
      saveAs: true
    }, (downloadId) => {
      if (browserAPI.runtime.lastError) {
        reject(new Error(browserAPI.runtime.lastError.message));
      } else {
        resolve(downloadId);
      }
    });
  });
}

/**
 * Export to Excel via backend API (Pro/Enterprise only)
 * @param {Array} ads - Array of ad objects
 * @param {string} filename - Filename
 * @returns {Promise<number>} Download ID
 */
export async function exportToExcel(ads, filename) {
  try {
    // Import API module
    const { apiRequest } = await import('./api.js');
    
    // Send data to backend for Excel generation
    const apiUrl = await apiRequest('/export/excel');
    
    const response = await fetchWithTimeout(apiUrl, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        ads: ads,
        filename: filename
      })
    }, TIMEOUTS.EXPORT);
    
    if (!response.ok) {
      const errorData = await response.json();
      throw new Error(errorData.error?.message || 'Failed to generate Excel file');
    }
    
    // Get blob from response
    const blob = await response.blob();
    const url = URL.createObjectURL(blob);
    
    // Download using Chrome API
    return new Promise((resolve, reject) => {
      browserAPI.downloads.download({
        url: url,
        filename: filename,
        saveAs: true
      }, (downloadId) => {
        URL.revokeObjectURL(url);
        
        if (browserAPI.runtime.lastError) {
          reject(new Error(browserAPI.runtime.lastError.message));
        } else {
          resolve(downloadId);
        }
      });
    });
  } catch (error) {
    log.error('Excel export error:', error);
    throw error;
  }
}
