/**
 * Ads Insight Pro - Fetch with Timeout Utility
 * v1.2.0: Timeouts now configurable via remote config
 * Prevents requests from hanging indefinitely
 */

import { RemoteConfig } from './remote-config.js';

/**
 * Fetch with timeout support
 * @param {string} url - Request URL
 * @param {Object} options - Fetch options
 * @param {number} timeout - Timeout in milliseconds (default: 30000)
 * @returns {Promise<Response>} Fetch response
 * @throws {Error} If request times out or fails
 */
export async function fetchWithTimeout(url, options = {}, timeout = 30000) {
  const controller = new AbortController();
  const timeoutId = setTimeout(() => controller.abort(), timeout);
  
  try {
    const response = await fetch(url, {
      ...options,
      signal: controller.signal
    });
    clearTimeout(timeoutId);
    return response;
  } catch (error) {
    clearTimeout(timeoutId);
    
    if (error.name === 'AbortError') {
      throw new Error(`Request timeout after ${timeout}ms`);
    }
    
    throw error;
  }
}

/**
 * Fetch with retry support
 * @param {string} url - Request URL
 * @param {Object} options - Fetch options
 * @param {number} maxRetries - Maximum number of retries (default: 3)
 * @param {number} timeout - Timeout per request in milliseconds (default: 30000)
 * @returns {Promise<Response>} Fetch response
 */
export async function fetchWithRetry(url, options = {}, maxRetries = 3, timeout = 30000) {
  let lastError;
  
  for (let attempt = 0; attempt <= maxRetries; attempt++) {
    try {
      return await fetchWithTimeout(url, options, timeout);
    } catch (error) {
      lastError = error;
      
      // Don't retry on client errors (4xx)
      if (error.response && error.response.status >= 400 && error.response.status < 500) {
        throw error;
      }
      
      // Wait before retrying (exponential backoff)
      if (attempt < maxRetries) {
        const delay = Math.min(1000 * Math.pow(2, attempt), 10000);
        await new Promise(resolve => setTimeout(resolve, delay));
      }
    }
  }
  
  throw lastError;
}

/**
 * Default timeout values for different operations
 * v1.2.0: These can be overridden by remote configuration
 */
const DEFAULT_TIMEOUTS = {
  SHORT: 10000,    // 10 seconds - for quick operations
  NORMAL: 30000,   // 30 seconds - default
  LONG: 60000,     // 60 seconds - for large data transfers
  VERY_LONG: 120000 // 2 minutes - for exports
};

/**
 * Get timeout values from remote config with local fallback
 * v1.2.0: Dynamically reads from remote configuration
 */
function getTimeouts() {
  const remoteTimeouts = RemoteConfig.getTimeouts();
  return {
    SHORT: remoteTimeouts.short_ms || DEFAULT_TIMEOUTS.SHORT,
    NORMAL: remoteTimeouts.normal_ms || DEFAULT_TIMEOUTS.NORMAL,
    LONG: remoteTimeouts.long_ms || DEFAULT_TIMEOUTS.LONG,
    VERY_LONG: remoteTimeouts.very_long_ms || DEFAULT_TIMEOUTS.VERY_LONG
  };
}

/**
 * Exported TIMEOUTS getter for backward compatibility
 * v1.2.0: Now returns dynamic values from remote config
 */
export const TIMEOUTS = {
  get SHORT() { return getTimeouts().SHORT; },
  get NORMAL() { return getTimeouts().NORMAL; },
  get LONG() { return getTimeouts().LONG; },
  get VERY_LONG() { return getTimeouts().VERY_LONG; }
};
