/**
 * Ads Insight Pro - Logger Module
 * Centralized logging with levels and formatting
 */

const LOG_LEVELS = {
  DEBUG: 0,
  INFO: 1,
  WARN: 2,
  ERROR: 3
};

// Set current log level (change to DEBUG for verbose logging)
const CURRENT_LEVEL = LOG_LEVELS.WARN;

// Prefix for all logs
const PREFIX = '[AdsInsightPro]';

// Color codes for console
const COLORS = {
  DEBUG: '#888888',
  INFO: '#4285f4',
  WARN: '#fbbc04',
  ERROR: '#ea4335',
  SUCCESS: '#34a853'
};

class Logger {
  constructor(module) {
    this.module = module;
  }

  _log(level, levelName, color, ...args) {
    if (level < CURRENT_LEVEL) return;
    
    const timestamp = new Date().toISOString().split('T')[1].split('.')[0];
    const prefix = `%c${PREFIX}[${this.module}][${timestamp}][${levelName}]`;
    
    console.log(prefix, `color: ${color}; font-weight: bold`, ...args);
  }

  debug(...args) {
    this._log(LOG_LEVELS.DEBUG, 'DEBUG', COLORS.DEBUG, ...args);
  }

  info(...args) {
    this._log(LOG_LEVELS.INFO, 'INFO', COLORS.INFO, ...args);
  }

  warn(...args) {
    this._log(LOG_LEVELS.WARN, 'WARN', COLORS.WARN, ...args);
  }

  error(...args) {
    this._log(LOG_LEVELS.ERROR, 'ERROR', COLORS.ERROR, ...args);
  }

  success(...args) {
    this._log(LOG_LEVELS.INFO, 'SUCCESS', COLORS.SUCCESS, ...args);
  }

  // Log object with pretty formatting
  object(label, obj) {
    this.debug(label + ':');
    console.log(JSON.stringify(obj, null, 2));
  }

  // Log API request
  request(method, url, body) {
    this.info(`→ ${method} ${url}`);
    if (body) {
      this.debug('Request body:', body);
    }
  }

  // Log API response
  response(status, length, preview) {
    if (status >= 200 && status < 300) {
      this.success(`← ${status} OK (${length} bytes)`);
    } else {
      this.error(`← ${status} ERROR`);
    }
    if (preview) {
      this.debug('Response preview:', preview.substring(0, 200));
    }
  }

  // Log step in a process
  step(stepNum, total, description) {
    this.info(`[Step ${stepNum}/${total}] ${description}`);
  }

  // Group logs
  group(label) {
    console.group(`${PREFIX}[${this.module}] ${label}`);
  }

  groupEnd() {
    console.groupEnd();
  }

  // Table for structured data
  table(data) {
    console.table(data);
  }
}

// Factory function
function createLogger(module) {
  return new Logger(module);
}

// Export for different contexts
if (typeof module !== 'undefined' && module.exports) {
  module.exports = { createLogger, LOG_LEVELS };
} else if (typeof window !== 'undefined') {
  window.AdsInsightLogger = { createLogger, LOG_LEVELS };
}
