/**
 * Ads Insight Pro - OCR Service Module (Firefox Version)
 * v1.2.3: 直接在 background script 中调用 OCR.space API
 * 
 * Firefox MV3 允许 background script 直接调用 host_permissions 中的域名，
 * 不需要通过 Content Script 代理（Content Script 在标签页失去焦点时会被暂停）
 */

import { browserAPI } from './browser-polyfill.js';
import { createLogger, DEBUG_CONFIG } from './debug-config.js';
import { permissionsManager } from './permissions.js';
import { getCachedOCR, setCachedOCR } from './ocr-cache.js';
import { RemoteConfig } from './remote-config.js';

// OCR 调试时开启，生产环境设为 false
const OCR_DEBUG = true;
const log = createLogger('[AdsInsightPro][OCR]', OCR_DEBUG || DEBUG_CONFIG.serviceWorker);

// v1.2.0: OCR 配置从远程获取，本地默认值作为回退
const DEFAULT_OCR_TIMEOUT_MS = 15000;
const DEFAULT_MAX_RETRIES = 2;

// 域名正则
const DOMAIN_REGEX = /(?:https?:\/\/)?(?:www\.)?([a-zA-Z0-9][-a-zA-Z0-9]*(?:\.[a-zA-Z0-9][-a-zA-Z0-9]*)+)/gi;

/**
 * v1.2.0: 获取 OCR 超时配置
 */
function getOCRTimeout() {
  return RemoteConfig.get('ocr_config.timeout_ms', DEFAULT_OCR_TIMEOUT_MS);
}

/**
 * v1.2.0: 获取 OCR 最大重试次数
 */
function getOCRMaxRetries() {
  return RemoteConfig.get('ocr_config.max_retries', DEFAULT_MAX_RETRIES);
}

/**
 * v1.2.0: 检查 OCR 功能是否启用
 */
function isOCREnabled() {
  return RemoteConfig.isOCREnabled();
}

/**
 * v1.2.0: 获取 OCR 并发配置
 */
function getOCRConcurrencyConfig() {
  return {
    concurrent: RemoteConfig.get('ocr_config.concurrent', 2),
    max_retries: RemoteConfig.get('ocr_config.max_retries', 2),
    retry_delay_ms: RemoteConfig.get('ocr_config.retry_delay_ms', 1500),
    request_delay_ms: RemoteConfig.get('ocr_config.request_delay_ms', 500)
  };
}

/**
 * Sleep helper
 */
function sleep(ms) {
  return new Promise(resolve => setTimeout(resolve, ms));
}

/**
 * 解析 OCR 文本，提取域名、标题和描述
 */
function parseOCRText(text) {
  const result = { domain: '', title: '', description: '' };
  if (!text) return result;
  
  const AD_LABELS = [
    'sponsored', 'ad', 'advertisement', 'promoted', '广告', '推广',
    'learn more', 'shop now', 'buy now', 'sign up', 'get started',
    'click here', 'download', 'install', 'subscribe'
  ];
  
  const lines = text.split(/[\n\r]+/).map(l => l.trim()).filter(l => l.length > 2);
  
  // 提取域名
  for (const line of lines) {
    const domainMatch = line.match(DOMAIN_REGEX);
    if (domainMatch) {
      const domain = domainMatch[0]
        .replace(/^https?:\/\//i, '')
        .replace(/^www\./i, '')
        .split('/')[0]
        .toLowerCase();
      
      if (domain && !domain.includes('google') && !domain.includes('gstatic') && domain.includes('.')) {
        result.domain = domain;
        break;
      }
    }
  }
  
  // 过滤文本行
  const textLines = lines.filter(l => {
    const lower = l.toLowerCase();
    if (DOMAIN_REGEX.test(l)) return false;
    if (/^[0-9\s\-\.\,\!\?\@\#\$\%\^\&\*\(\)]+$/.test(l)) return false;
    if (l.length < 3) return false;
    if (AD_LABELS.some(label => lower === label)) return false;
    return true;
  });
  
  // 提取标题
  let titleIndex = 0;
  if (result.domain) {
    const brandName = result.domain.split('.')[0].toLowerCase();
    for (let i = 0; i < textLines.length; i++) {
      if (textLines[i].toLowerCase().includes(brandName) && textLines[i].length > 10) {
        titleIndex = i;
        break;
      }
    }
  }
  
  if (textLines.length > titleIndex) {
    result.title = textLines[titleIndex].substring(0, 200);
  }
  
  // 提取描述
  const descLines = textLines.filter((_, i) => i !== titleIndex);
  if (descLines.length > 0) {
    result.description = descLines.join(' ').substring(0, 500);
  }
  
  return result;
}

/**
 * 直接在 background script 中调用 OCR.space API
 * Firefox MV3 允许 background 直接访问 host_permissions 中的域名
 * @param {string} imageUrl - 图片 URL
 * @param {Object} ocrConfig - OCR API 配置
 * @returns {Promise<Object>} - OCR 结果
 */
async function callOCRSpaceAPI(imageUrl, ocrConfig) {
  log.info('Calling OCR.space API directly from background:', imageUrl.substring(0, 80));
  
  if (!ocrConfig || !ocrConfig.api_key) {
    throw new Error('OCR API key not configured');
  }
  
  // 获取图片
  const response = await fetch(imageUrl);
  if (!response.ok) {
    throw new Error(`Failed to fetch image: ${response.status}`);
  }
  
  const blob = await response.blob();
  
  // 确定文件扩展名
  let ext = 'png';
  if (blob.type === 'image/jpeg' || blob.type === 'image/jpg') ext = 'jpg';
  else if (blob.type === 'image/gif') ext = 'gif';
  else if (blob.type === 'image/webp') ext = 'webp';
  
  // 转换为 base64
  const arrayBuffer = await blob.arrayBuffer();
  const uint8Array = new Uint8Array(arrayBuffer);
  let binary = '';
  for (let i = 0; i < uint8Array.length; i++) {
    binary += String.fromCharCode(uint8Array[i]);
  }
  const base64Image = `data:${blob.type};base64,${btoa(binary)}`;
  
  // 构建请求
  const formData = new FormData();
  formData.append('base64Image', base64Image);
  formData.append('filetype', ext.toUpperCase());
  formData.append('language', ocrConfig.languages?.[0] || 'eng');
  formData.append('isOverlayRequired', 'false');
  formData.append('detectOrientation', 'true');
  formData.append('scale', 'true');
  formData.append('OCREngine', String(ocrConfig.engine || 1));
  
  const apiEndpoint = ocrConfig.api_endpoint || 'https://api.ocr.space/parse/image';
  
  const ocrResponse = await fetch(apiEndpoint, {
    method: 'POST',
    headers: { 'apikey': ocrConfig.api_key },
    body: formData
  });
  
  // 检查响应状态
  if (!ocrResponse.ok) {
    const errorText = await ocrResponse.text().catch(() => '');
    throw new Error(`OCR API error ${ocrResponse.status}: ${errorText || ocrResponse.statusText}`);
  }
  
  const ocrResult = await ocrResponse.json();
  
  if (ocrResult.ParsedResults?.length > 0) {
    const parsed = ocrResult.ParsedResults[0];
    return {
      success: true,
      text: parsed.ParsedText || '',
      confidence: parsed.ParsedText?.trim() ? 80 : 0
    };
  }
  
  // 检查错误
  if (ocrResult.ErrorMessage) {
    log.warn('OCR.space API error:', ocrResult.ErrorMessage);
    return {
      success: false,
      error: ocrResult.ErrorMessage
    };
  }
  
  return { success: true, text: '', confidence: 0 };
}

// 全局速率限制状态
let rateLimitState = {
  isLimited: false,
  limitUntil: 0,
  consecutiveErrors: 0
};

// 全局取消标志
let ocrAbortController = null;

/**
 * 设置取消控制器
 */
export function setOCRAbortController(controller) {
  ocrAbortController = controller;
}

/**
 * 检查是否已取消
 */
function isOCRAborted() {
  return ocrAbortController && ocrAbortController.signal.aborted;
}

/**
 * 带重试的 OCR API 调用
 * @param {string} imageUrl - 图片 URL
 * @param {Object} ocrConfig - OCR API 配置
 * @param {number} maxRetries - 最大重试次数
 * @returns {Promise<Object>} - OCR 结果
 */
async function callOCRSpaceAPIWithRetry(imageUrl, ocrConfig, maxRetries = 2) {
  let lastError;
  
  // 检查全局速率限制状态
  if (rateLimitState.isLimited && Date.now() < rateLimitState.limitUntil) {
    const waitTime = rateLimitState.limitUntil - Date.now();
    log.warn(`Global rate limit active, waiting ${Math.ceil(waitTime / 1000)}s`);
    await sleep(waitTime);
  }
  
  for (let attempt = 0; attempt <= maxRetries; attempt++) {
    try {
      const result = await callOCRSpaceAPI(imageUrl, ocrConfig);
      // 成功后重置速率限制状态
      rateLimitState.consecutiveErrors = 0;
      rateLimitState.isLimited = false;
      return result;
    } catch (e) {
      lastError = e;
      
      // 如果是 403 错误，设置全局速率限制
      if (e.message && (e.message.includes('403') || e.message.includes('rate') || e.message.includes('limit'))) {
        rateLimitState.consecutiveErrors++;
        // 递增等待时间：5秒 -> 15秒 -> 30秒
        const baseWait = 5000;
        const waitTime = baseWait * Math.pow(2, Math.min(rateLimitState.consecutiveErrors - 1, 3));
        rateLimitState.isLimited = true;
        rateLimitState.limitUntil = Date.now() + waitTime;
        
        log.warn(`Rate limited (403), consecutive errors: ${rateLimitState.consecutiveErrors}, waiting ${waitTime / 1000}s before retry ${attempt + 1}/${maxRetries}`);
        await sleep(waitTime);
      } else if (attempt < maxRetries) {
        // 其他错误，短暂等待后重试
        await sleep(1000);
      }
    }
  }
  
  throw lastError;
}

/**
 * 批量处理 OCR（直接在 background 中调用 API）
 * @param {Array} ads - 广告数组 [{creative_id, image_url}, ...]
 * @param {Object} ocrConfig - OCR API 配置
 * @param {number} concurrency - 并发数
 * @returns {Promise<Array>} - OCR 结果数组
 */
async function performOCRBatch(ads, ocrConfig, concurrency = 2) {
  log.info(`Processing ${ads.length} images with concurrency: ${concurrency}`);
  
  const results = [];
  let consecutiveErrors = 0;
  const MAX_CONSECUTIVE_ERRORS = 5;
  
  // 保持并发数为 2
  const effectiveConcurrency = Math.min(concurrency, 2);
  
  // 分批处理
  for (let i = 0; i < ads.length; i += effectiveConcurrency) {
    // 检查是否已取消
    if (isOCRAborted()) {
      log.warn('OCR batch processing aborted by user');
      // 将剩余广告标记为取消
      for (let j = i; j < ads.length; j++) {
        results.push({
          creative_id: ads[j].creative_id,
          success: false,
          error: 'Cancelled by user'
        });
      }
      break;
    }
    
    const batch = ads.slice(i, Math.min(i + effectiveConcurrency, ads.length));
    
    log.info(`Processing batch ${Math.floor(i / effectiveConcurrency) + 1}, items ${i + 1}-${i + batch.length}`);
    
    // 并行处理批次（带重试）
    const batchPromises = batch.map(async (ad) => {
      try {
        const result = await callOCRSpaceAPIWithRetry(ad.image_url, ocrConfig, 2);
        consecutiveErrors = 0; // 成功后重置连续错误计数
        return {
          creative_id: ad.creative_id,
          ...result
        };
      } catch (e) {
        consecutiveErrors++;
        log.error('OCR failed for', ad.creative_id, ':', e.message);
        return {
          creative_id: ad.creative_id,
          success: false,
          error: e.message
        };
      }
    });
    
    const batchResults = await Promise.all(batchPromises);
    results.push(...batchResults);
    
    log.info(`Batch complete: ${results.length}/${ads.length} processed`);
    
    // 检查是否连续失败太多，如果是则提前结束
    if (consecutiveErrors >= MAX_CONSECUTIVE_ERRORS) {
      log.warn(`Too many consecutive errors (${consecutiveErrors}), stopping batch processing`);
      // 将剩余的广告标记为失败
      for (let j = i + effectiveConcurrency; j < ads.length; j++) {
        results.push({
          creative_id: ads[j].creative_id,
          success: false,
          error: 'Batch processing stopped due to rate limiting'
        });
      }
      break;
    }
    
    // 批次间延迟 - 根据速率限制状态动态调整
    if (i + effectiveConcurrency < ads.length) {
      // 基础延迟 500ms，如果有连续错误则增加
      let delay = 500;
      if (rateLimitState.isLimited) {
        // 如果处于速率限制状态，等待更长时间
        delay = Math.max(3000, (rateLimitState.limitUntil - Date.now()));
      } else if (consecutiveErrors > 0) {
        delay = 2000 * consecutiveErrors;
      }
      await sleep(delay);
    }
  }
  
  log.success(`All OCR complete: ${results.length} results`);
  return results;
}

/**
 * 批量处理 OCR（直接在 background script 中调用 API）
 * @param {Array} ads - 广告数组 [{creative_id, image_url}, ...]
 * @param {Object} ocrConfig - OCR API 配置
 * @param {Function} onProgress - 进度回调
 * @returns {Promise<Array>} - OCR 结果数组
 */
async function processAdsBatch(ads, ocrConfig, onProgress) {
  log.info(`Processing ${ads.length} ads directly from background script`);
  
  const concurrencyConfig = getOCRConcurrencyConfig();
  
  try {
    // 直接在 background 中调用 OCR API
    const batchResults = await performOCRBatch(ads, ocrConfig, concurrencyConfig.concurrent);
    
    // 转换结果格式
    const results = batchResults.map(r => {
      if (r.success && r.text) {
        const parsed = parseOCRText(r.text);
        return {
          creative_id: r.creative_id,
          ad_domain: parsed.domain,
          ad_title: parsed.title,
          ad_description: parsed.description,
          confidence: (r.confidence || 0) / 100,
          raw_text: r.text
        };
      } else {
        return {
          creative_id: r.creative_id,
          ad_domain: '',
          ad_title: '',
          ad_description: '',
          confidence: 0,
          raw_text: '',
          error: r.error || 'No text found'
        };
      }
    });
    
    if (onProgress) {
      onProgress({
        processed: results.length,
        total: ads.length,
        percentage: 100
      });
    }
    
    return results;
    
  } catch (e) {
    log.error('Batch OCR failed:', e.message);
    // 返回空结果
    return ads.map(ad => ({
      creative_id: ad.creative_id,
      ad_domain: '',
      ad_title: '',
      ad_description: '',
      confidence: 0,
      raw_text: '',
      error: e.message
    }));
  }
}

/**
 * 对单张图片执行 OCR（公开接口，不消耗配额）
 * @param {string} imageUrl - 图片 URL
 * @returns {Promise<Object>} - OCR 结果
 */
export async function performOCR(imageUrl) {
  const result = {
    ad_domain: '',
    ad_title: '',
    ad_description: '',
    confidence: 0,
    raw_text: ''
  };
  
  try {
    if (!imageUrl || !imageUrl.startsWith('http')) {
      return result;
    }
    
    // 获取 OCR 配置
    const ocrConfig = await permissionsManager.getOCRConfig();
    if (!ocrConfig || !ocrConfig.api_key) {
      log.warn('OCR API key not configured');
      return result;
    }
    
    // 直接在 background 中调用 OCR API
    const ocrResult = await callOCRSpaceAPI(imageUrl, ocrConfig);
    
    if (ocrResult.success && ocrResult.text) {
      const parsed = parseOCRText(ocrResult.text);
      return {
        ad_domain: parsed.domain,
        ad_title: parsed.title,
        ad_description: parsed.description,
        confidence: (ocrResult.confidence || 0) / 100,
        raw_text: ocrResult.text || ''
      };
    }
    
  } catch (e) {
    log.error('OCR failed:', e);
  }
  
  return result;
}

/**
 * v1.1.0: 带配额管理的单张图片 OCR
 * @param {string} imageUrl - 图片 URL
 * @param {string} creativeId - 广告创意 ID（用于缓存）
 * @returns {Promise<Object>} - OCR 结果
 */
export async function performOCRWithQuota(imageUrl, creativeId = null) {
  const emptyResult = {
    ad_domain: '',
    ad_title: '',
    ad_description: '',
    confidence: 0,
    raw_text: '',
    ocr_status: 'skipped'
  };
  
  try {
    if (!imageUrl || !imageUrl.startsWith('http')) {
      return emptyResult;
    }
    
    // 1. 检查配额
    const quotaCheck = await permissionsManager.checkOCRQuota(1);
    if (!quotaCheck.allowed) {
      log.warn('OCR quota exceeded:', quotaCheck.message);
      return { ...emptyResult, ocr_status: 'quota_exceeded', quota_message: quotaCheck.message };
    }
    
    // 2. 先扣配额
    const consumeResult = await permissionsManager.consumeOCRQuota(1);
    if (!consumeResult.success) {
      log.warn('Failed to consume OCR quota:', consumeResult.message);
      return { ...emptyResult, ocr_status: 'consume_failed' };
    }
    
    const transactionId = consumeResult.transaction_id;
    log.info('OCR quota consumed, transaction:', transactionId);
    
    // 3. 检查本地缓存
    if (creativeId) {
      try {
        const cachedResult = await getCachedOCR(creativeId, imageUrl);
        if (cachedResult) {
          log.info('💾 OCR cache HIT - saved API call for:', creativeId);
          await permissionsManager.completeOCRTransaction(transactionId);
          return {
            ...cachedResult,
            ocr_status: 'success',
            from_cache: true
          };
        }
      } catch (cacheError) {
        log.warn('Cache check failed, proceeding with OCR:', cacheError.message);
      }
    }
    
    // 4. 执行实际 OCR（直接在 background 中调用 API）
    try {
      const ocrConfig = await permissionsManager.getOCRConfig();
      const apiResult = await callOCRSpaceAPI(imageUrl, ocrConfig);
      
      let ocrResult = { confidence: 0, raw_text: '' };
      if (apiResult.success && apiResult.text) {
        const parsed = parseOCRText(apiResult.text);
        ocrResult = {
          ad_domain: parsed.domain,
          ad_title: parsed.title,
          ad_description: parsed.description,
          confidence: (apiResult.confidence || 0) / 100,
          raw_text: apiResult.text || ''
        };
      }
      
      // 5. 标记交易完成
      await permissionsManager.completeOCRTransaction(transactionId);
      
      // 6. 保存到本地缓存
      if (creativeId && ocrResult.confidence > 0) {
        setCachedOCR(creativeId, imageUrl, ocrResult).catch(e => {
          log.warn('Failed to cache OCR result:', e.message);
        });
      }
      
      return {
        ...ocrResult,
        ocr_status: ocrResult.confidence > 0 ? 'success' : 'no_text'
      };
    } catch (ocrError) {
      // 7. OCR 失败，标记失败并尝试退款
      log.error('OCR failed, attempting refund:', ocrError.message);
      
      const failResult = await permissionsManager.failOCRTransaction(transactionId, ocrError.message);
      
      if (failResult.can_refund) {
        const refundResult = await permissionsManager.refundOCRQuota(transactionId);
        log.info('OCR quota refund:', refundResult.success ? 'success' : 'failed');
      }
      
      return { ...emptyResult, ocr_status: 'failed', error: ocrError.message };
    }
  } catch (e) {
    log.error('performOCRWithQuota error:', e);
    return { ...emptyResult, ocr_status: 'error', error: e.message };
  }
}

/**
 * v1.2.1: 带配额管理的批量 OCR 处理
 * @param {Array} ads - 广告数组
 * @param {Object} options - 选项
 * @param {Function} onProgress - 进度回调
 * @returns {Promise<Object>} - { ads: [], ocrStats: {} }
 */
export async function processAdsWithOCRAndQuota(ads, options = {}, onProgress) {
  const { maxOCR = Infinity, abortSignal = null } = options;
  
  // 筛选需要 OCR 的广告
  const imageAds = ads.filter(ad => {
    const imageUrl = ad.image_url || ad._temp_image_url;
    return imageUrl && 
           imageUrl.startsWith('http') &&
           !ad.ad_title;
  });
  
  const stats = {
    total: imageAds.length,
    processed: 0,
    successful: 0,
    skipped: 0,
    failed: 0,
    quota_exceeded: 0
  };
  
  if (imageAds.length === 0) {
    log.info('No image ads need OCR processing');
    return { ads, ocrStats: stats };
  }
  
  // 检查配额
  const quotaCheck = await permissionsManager.checkOCRQuota(Math.min(imageAds.length, maxOCR));
  const effectiveMax = Math.min(maxOCR, quotaCheck.effective_remaining);
  
  if (effectiveMax === 0) {
    log.warn('No OCR quota available');
    for (const ad of imageAds) {
      ad.ocr_status = 'quota_exceeded';
      stats.quota_exceeded++;
    }
    return { ads, ocrStats: stats };
  }
  
  // 准备要处理的广告
  const adsToProcess = imageAds.slice(0, effectiveMax).map(ad => ({
    creative_id: ad.creative_id,
    image_url: ad.image_url || ad._temp_image_url
  }));
  
  // 标记超出配额的广告
  for (let i = effectiveMax; i < imageAds.length; i++) {
    imageAds[i].ocr_status = 'skipped';
    stats.skipped++;
  }
  
  log.info(`Starting batch OCR for ${adsToProcess.length} of ${imageAds.length} image ads`);
  
  try {
    // 获取 OCR 配置
    const ocrConfig = await permissionsManager.getOCRConfig();
    
    if (!ocrConfig || !ocrConfig.api_key) {
      log.error('OCR API key not configured');
      for (const ad of imageAds.slice(0, effectiveMax)) {
        ad.ocr_status = 'failed';
        stats.failed++;
      }
      return { ads, ocrStats: stats };
    }
    
    // 直接调用远程 API 批量处理
    const ocrResults = await processAdsBatch(adsToProcess, ocrConfig, (progress) => {
      if (onProgress) {
        onProgress({
          ...progress,
          successful: stats.successful
        });
      }
    });
    
    if (ocrResults && ocrResults.length > 0) {
      log.info('Batch OCR completed:', ocrResults.length, 'results');
      
      // 处理结果
      const resultsMap = new Map(ocrResults.map(r => [r.creative_id, r]));
      
      for (const ad of imageAds.slice(0, effectiveMax)) {
        const ocrResult = resultsMap.get(ad.creative_id);
        
        if (ocrResult && (ocrResult.confidence > 0.1 || ocrResult.ad_title || ocrResult.raw_text)) {
          ad.ad_domain = ocrResult.ad_domain || ad.ad_domain || '';
          ad.ad_title = ocrResult.ad_title || ad.ad_title || '';
          ad.ad_description = ocrResult.ad_description || ad.ad_description || '';
          ad.confidence = ocrResult.confidence || 0;
          ad.ocr_status = 'success';
          stats.successful++;
        } else if (ocrResult && ocrResult.error) {
          ad.ocr_status = 'failed';
          stats.failed++;
        } else {
          ad.ocr_status = 'no_text';
          stats.successful++;
        }
        stats.processed++;
      }
      
      // 保存到缓存
      for (const result of ocrResults) {
        if (result.confidence > 0) {
          const adData = adsToProcess.find(a => a.creative_id === result.creative_id);
          if (adData) {
            setCachedOCR(result.creative_id, adData.image_url, result).catch(() => {});
          }
        }
      }
    } else {
      log.warn('Batch OCR failed or returned no results');
      for (const ad of imageAds.slice(0, effectiveMax)) {
        ad.ocr_status = 'failed';
        stats.failed++;
        stats.processed++;
      }
    }
    
  } catch (e) {
    log.error('Batch OCR error:', e);
    for (const ad of imageAds.slice(0, effectiveMax)) {
      if (!ad.ocr_status) {
        ad.ocr_status = 'failed';
        stats.failed++;
        stats.processed++;
      }
    }
  }
  
  // 最终进度回调
  if (onProgress) {
    onProgress({
      processed: stats.processed,
      total: adsToProcess.length,
      successful: stats.successful,
      percentage: 100
    });
  }
  
  // 消费 OCR 配额
  if (stats.successful > 0) {
    try {
      const consumeResult = await permissionsManager.consumeOCRQuota(stats.successful);
      if (consumeResult.success) {
        log.info(`OCR quota consumed: ${stats.successful}, transaction: ${consumeResult.transaction_id}`);
        await permissionsManager.completeOCRTransaction(consumeResult.transaction_id);
      } else {
        log.warn('Failed to consume OCR quota:', consumeResult.message);
      }
    } catch (e) {
      log.error('OCR quota consumption error:', e);
    }
  }
  
  log.success(`OCR complete: ${stats.successful}/${stats.processed} successful, ${stats.skipped} skipped, ${stats.failed} failed`);
  
  return { ads, ocrStats: stats };
}

/**
 * 批量处理广告的 OCR（原版本，不消耗配额，内部使用）
 * @param {Array} ads - 广告数组
 * @param {Function} onProgress - 进度回调
 * @returns {Promise<Array>} - 处理后的广告数组
 */
export async function processAdsWithOCR(ads, onProgress) {
  // 筛选需要 OCR 的广告
  const imageAds = ads.filter(ad => 
    ad.image_url && 
    ad.image_url.startsWith('http') &&
    !ad.ad_title
  );
  
  if (imageAds.length === 0) {
    log.info('No image ads need OCR processing');
    return ads;
  }
  
  log.info('Starting OCR for', imageAds.length, 'image ads');
  
  // 准备要发送的数据
  const adsToProcess = imageAds.map(ad => ({
    creative_id: ad.creative_id,
    image_url: ad.image_url
  }));
  
  try {
    // 获取 OCR 配置
    const ocrConfig = await permissionsManager.getOCRConfig();
    
    if (!ocrConfig || !ocrConfig.api_key) {
      log.error('OCR API key not configured');
      return ads;
    }
    
    // 直接调用远程 API
    const ocrResults = await processAdsBatch(adsToProcess, ocrConfig, onProgress);
    
    if (ocrResults && ocrResults.length > 0) {
      log.info('OCR results received:', ocrResults.length, 'results');
      const resultsMap = new Map(ocrResults.map(r => [r.creative_id, r]));
      
      let successful = 0;
      for (const ad of ads) {
        const ocrResult = resultsMap.get(ad.creative_id);
        
        if (ocrResult && (ocrResult.confidence > 0.1 || ocrResult.ad_title || ocrResult.raw_text)) {
          ad.ad_domain = ocrResult.ad_domain || ad.ad_domain || '';
          ad.ad_title = ocrResult.ad_title || ad.ad_title || '';
          ad.ad_description = ocrResult.ad_description || ad.ad_description || '';
          ad.confidence = ocrResult.confidence || 0;
          successful++;
        }
      }
      
      log.success(`OCR complete [ocr-space-api]: ${successful}/${imageAds.length} successful`);
      
      if (onProgress) {
        onProgress({
          processed: imageAds.length,
          total: imageAds.length,
          successful,
          percentage: 100,
          method: 'ocr-space-api'
        });
      }
    } else {
      log.error('OCR failed');
    }
  } catch (e) {
    log.error('OCR error:', e);
  }
  
  return ads;
}

/**
 * 从广告数据中提取文本内容（备用方案，从 HTML 提取）
 * @param {Object} ad - 广告对象
 * @param {Object} rawItem - API 返回的原始数据
 * @returns {Object} - 提取的文本内容
 */
export function extractTextFromAdData(ad, rawItem) {
  const result = {
    ad_domain: '',
    ad_title: '',
    ad_description: '',
    confidence: 0
  };
  
  try {
    // 尝试从 creative HTML 中提取
    const creative = rawItem?.["3"] || rawItem?.[3];
    if (creative) {
      const htmlContent = creative?.["3"]?.["2"] || creative?.[3]?.[2] || '';
      
      if (htmlContent && typeof htmlContent === 'string') {
        // 提取 href 中的域名
        const hrefMatch = htmlContent.match(/href=["']([^"']+)["']/i);
        if (hrefMatch) {
          const domainMatch = hrefMatch[1].match(DOMAIN_REGEX);
          if (domainMatch) {
            const domain = domainMatch[0]
              .replace(/^https?:\/\//i, '')
              .replace(/^www\./i, '')
              .split('/')[0];
            if (domain && !domain.includes('google')) {
              result.ad_domain = domain;
              result.confidence = 0.7;
            }
          }
        }
      }
    }
    
    // 使用 target_domain 作为备选
    if (!result.ad_domain && ad.target_domain) {
      result.ad_domain = ad.target_domain;
      result.confidence = Math.max(result.confidence, 0.9);
    }
    
  } catch (e) {
    // 静默失败
  }
  
  return result;
}

/**
 * 终止 OCR
 */
export async function terminateOCR() {
  log.info('OCR terminated');
  // 触发取消
  if (ocrAbortController) {
    ocrAbortController.abort();
    ocrAbortController = null;
  }
  // 重置速率限制状态
  rateLimitState = {
    isLimited: false,
    limitUntil: 0,
    consecutiveErrors: 0
  };
}

export default {
  performOCR,
  performOCRWithQuota,
  processAdsWithOCR,
  processAdsWithOCRAndQuota,
  extractTextFromAdData,
  terminateOCR,
  setOCRAbortController
};
