﻿/**
 * Ads Insight Pro - Storage Module
 * Handles local storage operations
 */

import { browserAPI } from './browser-polyfill.js';
import { createLogger, DEBUG_CONFIG } from './debug-config.js';

const log = createLogger('[Storage]', DEBUG_CONFIG.background);

// Storage keys
const KEYS = {
  USER_PLAN: 'userPlan',
  DAILY_USED: 'dailyUsed',
  LAST_USED_DATE: 'lastUsedDate',
  AUTH_TOKEN: 'authToken',
  AUTH_EXPIRES: 'authExpires',
  DEVICE_ID: 'deviceId',
  SETTINGS: 'settings',
  HISTORY: 'downloadHistory'
};

/**
 * Get value from storage
 * @param {string} key - Storage key
 * @param {*} defaultValue - Default value if not found
 * @returns {Promise<*>} Stored value
 */
export async function get(key, defaultValue = null) {
  try {
    const result = await browserAPI.storage.local.get(key);
    return result[key] !== undefined ? result[key] : defaultValue;
  } catch (e) {
    log.error('Storage get error:', e);
    return defaultValue;
  }
}

/**
 * Set value in storage
 * @param {string} key - Storage key
 * @param {*} value - Value to store
 * @returns {Promise<void>}
 */
export async function set(key, value) {
  try {
    await browserAPI.storage.local.set({ [key]: value });
  } catch (e) {
    log.error('Storage set error:', e);
    throw e;
  }
}

/**
 * Remove value from storage
 * @param {string} key - Storage key
 * @returns {Promise<void>}
 */
export async function remove(key) {
  try {
    await browserAPI.storage.local.remove(key);
  } catch (e) {
    log.error('Storage remove error:', e);
    throw e;
  }
}

/**
 * Get user plan info
 * @returns {Promise<Object>} Plan info
 */
export async function getUserPlan() {
  const plan = await get(KEYS.USER_PLAN, 'free');
  const dailyUsed = await get(KEYS.DAILY_USED, 0);
  const lastUsedDate = await get(KEYS.LAST_USED_DATE, '');
  
  // Reset daily count if new day
  const today = new Date().toDateString();
  if (lastUsedDate !== today) {
    await set(KEYS.DAILY_USED, 0);
    await set(KEYS.LAST_USED_DATE, today);
    return { plan, dailyUsed: 0 };
  }
  
  return { plan, dailyUsed };
}

/**
 * Increment daily usage
 * @returns {Promise<number>} New usage count
 */
export async function incrementUsage() {
  const { dailyUsed } = await getUserPlan();
  const newCount = dailyUsed + 1;
  await set(KEYS.DAILY_USED, newCount);
  return newCount;
}

/**
 * Get user settings
 * @returns {Promise<Object>} Settings object
 */
export async function getSettings() {
  return await get(KEYS.SETTINGS, {
    defaultPages: 5,
    defaultFormat: 'csv',
    defaultRegion: '',
    proxyEnabled: false,
    proxy: null
  });
}

/**
 * Save user settings
 * @param {Object} settings - Settings to save
 * @returns {Promise<void>}
 */
export async function saveSettings(settings) {
  const current = await getSettings();
  await set(KEYS.SETTINGS, { ...current, ...settings });
}

/**
 * Get download history
 * @param {number} limit - Max items to return
 * @returns {Promise<Array>} History items
 */
export async function getHistory(limit = 10) {
  const history = await get(KEYS.HISTORY, []);
  return history.slice(0, limit);
}

/**
 * Add item to download history
 * @param {Object} item - History item
 * @returns {Promise<void>}
 */
export async function addToHistory(item) {
  const history = await get(KEYS.HISTORY, []);
  
  // Add new item at beginning
  history.unshift({
    ...item,
    timestamp: Date.now()
  });
  
  // Keep only last 50 items
  const trimmed = history.slice(0, 50);
  await set(KEYS.HISTORY, trimmed);
}

/**
 * Clear download history
 * @returns {Promise<void>}
 */
export async function clearHistory() {
  await set(KEYS.HISTORY, []);
}

/**
 * Get or generate device ID
 * @returns {Promise<string>} Device ID
 */
export async function getDeviceId() {
  let deviceId = await get(KEYS.DEVICE_ID);
  
  if (!deviceId) {
    deviceId = generateDeviceId();
    await set(KEYS.DEVICE_ID, deviceId);
  }
  
  return deviceId;
}

/**
 * Generate unique device ID
 * @returns {string} Device ID
 */
function generateDeviceId() {
  const array = new Uint8Array(16);
  crypto.getRandomValues(array);
  return Array.from(array, b => b.toString(16).padStart(2, '0')).join('');
}

/**
 * Save auth token
 * @param {string} token - Auth token (JWT)
 * @param {number} expiresIn - Expiration time in seconds (optional, default 7 days)
 * @returns {Promise<void>}
 */
export async function saveAuthToken(token, expiresIn = 7 * 24 * 60 * 60) {
  await set(KEYS.AUTH_TOKEN, token);
  await set(KEYS.AUTH_EXPIRES, Date.now() + (expiresIn * 1000));
}

/**
 * Get auth token if valid
 * @returns {Promise<string|null>} Auth token or null
 */
export async function getAuthToken() {
  const token = await get(KEYS.AUTH_TOKEN);
  const expires = await get(KEYS.AUTH_EXPIRES, 0);
  
  if (!token || Date.now() > expires) {
    return null;
  }
  
  return token;
}

/**
 * Clear auth data
 * @returns {Promise<void>}
 */
export async function clearAuth() {
  await remove(KEYS.AUTH_TOKEN);
  await remove(KEYS.AUTH_EXPIRES);
}

// Export keys for reference
export { KEYS };
