﻿import { browserAPI } from './browser-polyfill.js';
/**
 * Usage Thresholds Configuration
 * 
 * 使用量阈值配�?- 可以根据实际情况轻松调整
 * 
 * 如何调整阈值：
 * 1. 直接修改下面的数�?
 * 2. 或者通过 updateThresholds() 函数动态更�?
 * 3. 或者从后端 API 获取最新配�?
 * 
 * 调整建议�?
 * - 如果用户反馈警告太频繁，提高阈�?
 * - 如果用户遇到 Google 限制，降低阈�?
 * - 可以根据用户计划（Free/Pro/Enterprise）设置不同阈�?
 */

// ============== DEBUG MODE ==============
// 设置�?true 开启调试模式，显示详细日志；设置为 false 关闭日志输出
const DEBUG_MODE = false;

// ============== 默认阈值配�?==============
// 注意：这些是初始估算值，可能需要根据实际使用情况调�?

const DEFAULT_THRESHOLDS = {
  // 每小时限�?
  hourly: {
    // 安全区间（绿色）- 完全安全，不显示任何警告
    safe: {
      requests: 100,      // 请求次数
      ads: 5000,          // 广告数量
      domains: 5          // 域名数量
    },
    
    // 警告区间（黄色）- 显示提示，建议减慢速度
    warning: {
      requests: 150,      // 100-150 之间显示警告
      ads: 8000,          // 5000-8000 之间显示警告
      domains: 8          // 5-8 之间显示警告
    },
    
    // 危险区间（红色）- 显示严重警告，建议停�?
    danger: {
      requests: 200,      // > 150 显示危险警告
      ads: 10000,         // > 8000 显示危险警告
      domains: 10         // > 8 显示危险警告
    }
  },
  
  // 每天限制
  // 基于实际测试: 单IP限制�?12,000 ads (100万条/60个IP)
  daily: {
    // 安全区间（绿色）
    safe: {
      downloads: 20,      // 下载次数
      ads: 10000,         // 广告数量 (降低到单IP限制以下)
      domains: 30         // 域名数量
    },
    
    // 警告区间（黄色）
    warning: {
      downloads: 40,      // 20-40 之间显示警告
      ads: 12000,         // 10000-12000 之间显示警告 (接近单IP限制)
      domains: 40         // 30-40 之间显示警告
    },
    
    // 危险区间（红色）
    danger: {
      downloads: 50,      // > 40 显示危险警告
      ads: 15000,         // > 12000 显示危险警告 (超过单IP限制)
      domains: 50         // > 40 显示危险警告
    }
  }
};

// ============== 不同计划的阈值倍数 ==============
// 可以根据用户计划调整阈�?

const PLAN_MULTIPLIERS = {
  free: 0.5,          // Free 用户：阈值减半（更保守）
  pro: 1.0,           // Pro 用户：使用默认阈�?
  enterprise: 2.0     // Enterprise 用户：阈值翻倍（更宽松）
};

// ============== 阈值管理类 ==============

class ThresholdManager {
  constructor() {
    this.thresholds = JSON.parse(JSON.stringify(DEFAULT_THRESHOLDS));
    this.currentPlan = 'pro';
  }
  
  /**
   * 获取当前阈�?
   */
  getThresholds() {
    return this.thresholds;
  }
  
  /**
   * 根据用户计划调整阈�?
   */
  applyPlanMultiplier(plan) {
    this.currentPlan = plan;
    const multiplier = PLAN_MULTIPLIERS[plan] || 1.0;
    
    // 重置为默认�?
    this.thresholds = JSON.parse(JSON.stringify(DEFAULT_THRESHOLDS));
    
    // 应用倍数
    this.thresholds.hourly.safe.ads = Math.round(DEFAULT_THRESHOLDS.hourly.safe.ads * multiplier);
    this.thresholds.hourly.warning.ads = Math.round(DEFAULT_THRESHOLDS.hourly.warning.ads * multiplier);
    this.thresholds.hourly.danger.ads = Math.round(DEFAULT_THRESHOLDS.hourly.danger.ads * multiplier);
    
    this.thresholds.daily.safe.ads = Math.round(DEFAULT_THRESHOLDS.daily.safe.ads * multiplier);
    this.thresholds.daily.warning.ads = Math.round(DEFAULT_THRESHOLDS.daily.warning.ads * multiplier);
    this.thresholds.daily.danger.ads = Math.round(DEFAULT_THRESHOLDS.daily.danger.ads * multiplier);
    
    if (DEBUG_MODE) log.info(`[ThresholdManager] Applied ${plan} multiplier (${multiplier}x):`, this.thresholds);
  }
  
  /**
   * 手动更新阈值（用于调试和测试）
   */
  updateThresholds(newThresholds) {
    this.thresholds = {
      ...this.thresholds,
      ...newThresholds
    };
    if (DEBUG_MODE) log.info('[ThresholdManager] Thresholds updated:', this.thresholds);
  }
  
  /**
   * 从后端获取最新阈值配�?
   */
  async fetchFromBackend(apiUrl) {
    try {
      const response = await fetch(`${apiUrl}/api/config/thresholds`);
      if (response.ok) {
        const data = await response.json();
        if (data.success && data.data.thresholds) {
          this.updateThresholds(data.data.thresholds);
          if (DEBUG_MODE) log.info('[ThresholdManager] Loaded thresholds from backend');
          return true;
        }
      }
    } catch (e) {
      log.warn('[ThresholdManager] Failed to fetch thresholds from backend:', e.message);
    }
    return false;
  }
  
  /**
   * 保存自定义阈值到 storage
   */
  async saveCustomThresholds() {
    await browserAPI.storage.local.set({
      customThresholds: this.thresholds,
      thresholdsUpdatedAt: Date.now()
    });
  }
  
  /**
   * 加载自定义阈�?
   */
  async loadCustomThresholds() {
    const data = await browserAPI.storage.local.get('customThresholds');
    if (data.customThresholds) {
      this.thresholds = data.customThresholds;
      if (DEBUG_MODE) log.info('[ThresholdManager] Loaded custom thresholds from storage');
      return true;
    }
    return false;
  }
  
  /**
   * 重置为默认阈�?
   */
  resetToDefaults() {
    this.thresholds = JSON.parse(JSON.stringify(DEFAULT_THRESHOLDS));
    if (DEBUG_MODE) log.info('[ThresholdManager] Reset to default thresholds');
  }
  
  /**
   * 检查某个值是否超过阈�?
   * @returns {string} 'safe' | 'warning' | 'danger'
   */
  checkLevel(type, metric, value) {
    const thresholds = this.thresholds[type]; // 'hourly' or 'daily'
    
    if (value >= thresholds.danger[metric]) {
      return 'danger';
    } else if (value >= thresholds.warning[metric]) {
      return 'warning';
    } else {
      return 'safe';
    }
  }
  
  /**
   * 获取进度百分比（相对于危险阈值）
   */
  getProgress(type, metric, value) {
    const dangerThreshold = this.thresholds[type].danger[metric];
    const percentage = Math.min((value / dangerThreshold) * 100, 100);
    return Math.round(percentage);
  }
  
  /**
   * 导出当前配置（用于调试）
   */
  exportConfig() {
    return {
      thresholds: this.thresholds,
      plan: this.currentPlan,
      timestamp: new Date().toISOString()
    };
  }
  
  /**
   * 导入配置
   */
  importConfig(config) {
    if (config.thresholds) {
      this.thresholds = config.thresholds;
      this.currentPlan = config.plan || 'pro';
      if (DEBUG_MODE) log.info('[ThresholdManager] Imported config:', config);
    }
  }
}

// ============== 导出 ==============

// 创建单例
export const thresholdManager = new ThresholdManager();

// 也导出默认值，方便参�?
export { DEFAULT_THRESHOLDS, PLAN_MULTIPLIERS };

// ============== 使用示例 ==============

/*
// 1. 基本使用
import { thresholdManager } from './usage-thresholds.js';

const thresholds = thresholdManager.getThresholds();
log.info(thresholds.hourly.warning.ads); // 8000

// 2. 根据用户计划调整
thresholdManager.applyPlanMultiplier('enterprise'); // 阈值翻�?
thresholdManager.applyPlanMultiplier('free');       // 阈值减�?

// 3. 手动调整（用于测试）
thresholdManager.updateThresholds({
  hourly: {
    warning: { ads: 6000 },  // 降低警告阈�?
    danger: { ads: 9000 }    // 降低危险阈�?
  }
});

// 4. 从后端加�?
await thresholdManager.fetchFromBackend('https://api.adsinsightpro.com');

// 5. 检查级�?
const level = thresholdManager.checkLevel('hourly', 'ads', 7500);
// 返回: 'warning'

// 6. 获取进度
const progress = thresholdManager.getProgress('hourly', 'ads', 5000);
// 返回: 50 (相对于危险阈�?10000)

// 7. 保存和加载自定义配置
await thresholdManager.saveCustomThresholds();
await thresholdManager.loadCustomThresholds();

// 8. 导出配置（用于调试）
const config = thresholdManager.exportConfig();
log.info(JSON.stringify(config, null, 2));

// 9. 重置
thresholdManager.resetToDefaults();
*/
