﻿import { browserAPI } from '../lib/browser-polyfill.js';

/**
 * Ads Insight Pro - Account Page
 */

import { getApiUrl } from '../lib/config.js';
import { fetchWithTimeout, TIMEOUTS } from '../lib/fetch-with-timeout.js';
import { PLAN_LIMITS, TIME, UI } from '../lib/constants.js';
import { bindDevice } from '../lib/auth.js';
import { createLogger, DEBUG_CONFIG } from '../lib/debug-config.js';

const log = createLogger('[Account]', DEBUG_CONFIG.popup);

const BACKEND_API = getApiUrl();

// State
let currentUser = null;

document.addEventListener('DOMContentLoaded', init);

async function init() {
  setupEventListeners();
  await checkAuthStatus();
}

function setupEventListeners() {
  // Back button
  document.getElementById('btn-back').addEventListener('click', () => {
    window.location.href = 'popup.html';
  });
  
  // Form toggles
  document.getElementById('link-signup').addEventListener('click', (e) => {
    e.preventDefault();
    showSection('register');
  });
  
  document.getElementById('link-login').addEventListener('click', (e) => {
    e.preventDefault();
    showSection('login');
  });
  
  document.getElementById('link-forgot').addEventListener('click', (e) => {
    e.preventDefault();
    showSection('forgot');
  });
  
  document.getElementById('link-back-login').addEventListener('click', (e) => {
    e.preventDefault();
    showSection('login');
  });
  
  // Forms
  document.getElementById('login-form').addEventListener('submit', handleLogin);
  document.getElementById('register-form').addEventListener('submit', handleRegister);
  document.getElementById('forgot-form').addEventListener('submit', handleForgotPassword);
  
  // Actions
  document.getElementById('btn-logout').addEventListener('click', handleLogout);
  document.getElementById('btn-upgrade').addEventListener('click', handleUpgrade);
  document.getElementById('link-manage-sub')?.addEventListener('click', handleManageSubscription);
}

function showSection(section) {
  document.getElementById('login-section').classList.toggle('hidden', section !== 'login');
  document.getElementById('register-section').classList.toggle('hidden', section !== 'register');
  document.getElementById('forgot-section').classList.toggle('hidden', section !== 'forgot');
  
  // Clear errors
  document.querySelectorAll('.form-error, .form-success').forEach(el => el.classList.add('hidden'));
}

async function checkAuthStatus() {
  try {
    const data = await browserAPI.storage.local.get(['authToken', 'authExpires', 'userPlan', 'userEmail']);
    
    if (data.authToken && data.authExpires > Date.now()) {
      // Verify token with server
      const response = await fetchWithTimeout(`${BACKEND_API}/api/auth/status`, {
        headers: { 'Authorization': `Bearer ${data.authToken}` }
      }, TIMEOUTS.NORMAL);
      
      if (response.ok) {
        const result = await response.json();
        if (result.success) {
          currentUser = result.data.user;
          await browserAPI.storage.local.set({
            userPlan: currentUser.plan,
            userEmail: currentUser.email
          });
          showLoggedInView();
          return;
        }
      }
      
      // Token invalid, clear it
      await browserAPI.storage.local.remove(['authToken', 'authExpires']);
    }
    
    showLoggedOutView();
  } catch (e) {
    log.error('Auth check failed:', e);
    showLoggedOutView();
  }
}

function showLoggedOutView() {
  // Hide loading screen
  const loadingScreen = document.getElementById('loading-screen');
  const mainContent = document.querySelector('.account-main');
  if (loadingScreen) loadingScreen.style.display = 'none';
  if (mainContent) mainContent.style.display = 'block';
  
  // Show logged out view
  document.getElementById('logged-out-view').style.display = 'block';
  document.getElementById('logged-in-view').style.display = 'none';
  showSection('login');
}

function showLoggedInView() {
  // Hide loading screen
  const loadingScreen = document.getElementById('loading-screen');
  const mainContent = document.querySelector('.account-main');
  if (loadingScreen) loadingScreen.style.display = 'none';
  if (mainContent) mainContent.style.display = 'block';
  
  // Show logged in view
  document.getElementById('logged-out-view').style.display = 'none';
  document.getElementById('logged-in-view').style.display = 'block';
  
  if (currentUser) {
    updateProfileUI();
    updatePlanUI();
    updateUsageUI();
    loadDevices();
  }
}

function updateProfileUI() {
  document.getElementById('user-email').textContent = currentUser.email;
  document.getElementById('user-avatar').textContent = currentUser.email[0].toUpperCase();
  
  const planNames = { free: 'Free Plan', pro: 'Pro Plan', enterprise: 'Enterprise Plan' };
  document.getElementById('user-plan').textContent = planNames[currentUser.plan] || 'Free Plan';
}

function updatePlanUI() {
  const plan = currentUser.plan || 'free';
  const limits = PLAN_LIMITS[plan];
  const planCard = document.getElementById('plan-card');
  
  planCard.className = `plan-card ${plan}`;
  document.getElementById('plan-name').textContent = plan.charAt(0).toUpperCase() + plan.slice(1);
  
  const details = [];
  // v1.1.0: Changed to ads-quota system, match official website descriptions
  const quotaDisplay = `${limits.dailyQuota.toLocaleString()} ads/day quota`;
  details.push(quotaDisplay);
  details.push(`${limits.maxPages} pages per export batch`);
  
  // Export formats and features by plan
  if (plan === 'free') {
    details.push('CSV export');
  } else if (plan === 'pro') {
    details.push('CSV, JSON, Excel export');
  } else {
    details.push('All export formats');
  }
  
  document.getElementById('plan-details').innerHTML = details.map(d => `<div>${escapeHtml(d)}</div>`).join('');
  
  // Show/hide upgrade button
  const upgradeBtn = document.getElementById('btn-upgrade');
  const subInfo = document.getElementById('subscription-info');
  
  if (plan === 'free') {
    upgradeBtn.classList.remove('hidden');
    subInfo.classList.add('hidden');
  } else {
    upgradeBtn.classList.add('hidden');
    subInfo.classList.remove('hidden');
    
    document.getElementById('sub-status').textContent = 'Active';
    if (currentUser.plan_expires_at) {
      const expiresDate = new Date(currentUser.plan_expires_at * TIME.SECOND);
      document.getElementById('sub-renews').textContent = expiresDate.toLocaleDateString();
    }
  }
}

async function updateUsageUI() {
  try {
    // 从服务器获取权威的使用量数据 (v1.1.0: 使用新的 quota/permissions API)
    const data = await browserAPI.storage.local.get('authToken');
    if (!data.authToken) {
      log.warn('No auth token, cannot fetch usage from server');
      return;
    }
    
    const response = await fetchWithTimeout(`${BACKEND_API}/api/quota/permissions`, {
      headers: { 'Authorization': `Bearer ${data.authToken}` }
    }, TIMEOUTS.NORMAL);
    
    if (response.ok) {
      const result = await response.json();
      if (result.success && result.data) {
        const { quota, ocr_quota } = result.data;
        
        // 更新广告配额显示
        const usedToday = quota.used_today || 0;
        const dailyLimit = quota.daily_limit || 100;
        
        document.getElementById('usage-used').textContent = usedToday.toLocaleString();
        document.getElementById('usage-limit').textContent = dailyLimit.toLocaleString();
        
        const percent = Math.min(UI.PROGRESS_PERCENTAGE_MAX, (usedToday / dailyLimit) * UI.PROGRESS_PERCENTAGE_MAX);
        document.getElementById('usage-fill').style.width = `${percent}%`;
        
        // 更新 OCR 配额显示 (v1.1.0) - 显示已用/总量
        if (ocr_quota) {
          const ocrUsageEl = document.getElementById('ocr-usage');
          const ocrUsedEl = document.getElementById('ocr-used');
          const ocrLimitEl = document.getElementById('ocr-limit');
          
          if (ocrUsageEl && ocrUsedEl && ocrLimitEl) {
            ocrUsedEl.textContent = (ocr_quota.daily_used || 0).toLocaleString();
            ocrLimitEl.textContent = (ocr_quota.daily_limit || 0).toLocaleString();
            ocrUsageEl.style.display = 'block';
          }
        }
        
        log.info(`Usage updated from server: ads=${usedToday}/${dailyLimit}, ocr=${ocr_quota?.daily_used || 0}/${ocr_quota?.daily_limit || 0}`);
        return;
      }
    }
    
    // 如果服务器请求失败,回退到本地数据(仅用于显示)
    log.warn('Failed to fetch usage from server, falling back to local data');
    const plan = currentUser?.plan || 'free';
    const limit = PLAN_LIMITS[plan].dailyQuota;
    
    document.getElementById('usage-used').textContent = '0';
    document.getElementById('usage-limit').textContent = limit.toLocaleString();
    document.getElementById('usage-fill').style.width = '0%';
  } catch (e) {
    log.error('Failed to update usage:', e);
  }
}

async function loadDevices() {
  try {
    const data = await browserAPI.storage.local.get('authToken');
    if (!data.authToken) return;
    
    const response = await fetchWithTimeout(`${BACKEND_API}/api/auth/devices`, {
      headers: { 'Authorization': `Bearer ${data.authToken}` }
    }, TIMEOUTS.NORMAL);
    
    if (response.ok) {
      const result = await response.json();
      if (result.success && result.data.devices) {
        renderDevices(result.data.devices);
      }
    }
  } catch (e) {
    log.error('Failed to load devices:', e);
  }
}

function renderDevices(devices) {
  const list = document.getElementById('devices-list');
  const plan = currentUser?.plan || 'free';
  const limit = PLAN_LIMITS[plan].maxDevices;
  
  list.innerHTML = devices.map((device, i) => `
    <div class="device-item ${i === 0 ? 'current' : ''}">
      <span class="device-icon">💻</span>
      <div class="device-info">
        <div class="device-name">${escapeHtml(device.name || `Device ${i + 1}`)}</div>
        <div class="device-meta">${i === 0 ? 'This device' : escapeHtml(formatLastActive(device.last_active_at))}</div>
      </div>
    </div>
  `).join('');
  
  document.getElementById('devices-used').textContent = devices.length;
  document.getElementById('devices-limit').textContent = limit;
}

// XSS 防护 - 转义 HTML
function escapeHtml(str) {
  if (!str) return '';
  const div = document.createElement('div');
  div.textContent = str;
  return div.innerHTML;
}

function formatLastActive(timestamp) {
  if (!timestamp) return 'Unknown';
  const date = new Date(timestamp * TIME.SECOND);
  const now = new Date();
  const diff = now - date;
  
  if (diff < 60000) return 'Active now';
  if (diff < 3600000) return `${Math.floor(diff / 60000)}m ago`;
  if (diff < 86400000) return `${Math.floor(diff / 3600000)}h ago`;
  return date.toLocaleDateString();
}

// Form Handlers
async function handleLogin(e) {
  e.preventDefault();
  
  const email = document.getElementById('login-email').value;
  const password = document.getElementById('login-password').value;
  const errorEl = document.getElementById('login-error');
  const btn = document.getElementById('btn-login');
  
  btn.disabled = true;
  btn.innerHTML = '<span>Signing in...</span>';
  errorEl.classList.add('hidden');
  
  try {
    const response = await fetchWithTimeout(`${BACKEND_API}/api/auth/login`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ email, password })
    }, TIMEOUTS.NORMAL);
    
    const result = await response.json();
    
    if (result.success) {
      await browserAPI.storage.local.set({
        authToken: result.data.token,
        authExpires: Date.now() + TIME.AUTH_TOKEN_EXPIRY,
        userPlan: result.data.user.plan,
        userEmail: result.data.user.email
      });
      
      currentUser = result.data.user;
      
      // Bind device
      try {
        await bindDevice();
        log.info('Device bound successfully');
      } catch (e) {
        log.warn('Failed to bind device:', e);
      }
      
      // Link install to user via background script
      try {
        browserAPI.runtime.sendMessage({
          type: 'LINK_INSTALL_TO_USER',
          userId: currentUser.id
        });
      } catch (e) {
        log.warn('Failed to link install:', e);
      }
      
      // 清除缓存的权限信息，强制下次从服务器重新获取
      try {
        // 1. 清除 storage 中的缓存
        await browserAPI.storage.local.remove([
          'cachedPermissions',
          'permissionsCacheTime',
          'currentReservation',
          'currentOCRTransaction'
        ]);
        // 2. 通知 background script 清除内存缓存
        await browserAPI.runtime.sendMessage({ type: 'CLEAR_PERMISSIONS_CACHE' });
        log.info('Cleared permissions cache on login (storage + memory)');
      } catch (e) {
        log.warn('Failed to clear permissions cache:', e);
      }
      
      // 从服务器恢复使用量数据
      try {
        browserAPI.runtime.sendMessage({
          type: 'RESTORE_USAGE_FROM_SERVER'
        });
      } catch (e) {
        log.warn('Failed to restore usage from server:', e);
      }
      
      showLoggedInView();
    } else {
      errorEl.textContent = getErrorMessage(result.error?.code) || 'Login failed. Please check your credentials.';
      errorEl.classList.remove('hidden');
    }
  } catch (e) {
    errorEl.textContent = 'Network error. Please try again.';
    errorEl.classList.remove('hidden');
  } finally {
    btn.disabled = false;
    btn.innerHTML = '<span>Sign In</span>';
  }
}

async function handleRegister(e) {
  e.preventDefault();
  
  const email = document.getElementById('register-email').value;
  const password = document.getElementById('register-password').value;
  const confirm = document.getElementById('register-confirm').value;
  const errorEl = document.getElementById('register-error');
  const successEl = document.getElementById('register-success') || createSuccessElement('register');
  const btn = document.getElementById('btn-register');
  
  if (password !== confirm) {
    errorEl.textContent = 'Passwords do not match';
    errorEl.classList.remove('hidden');
    return;
  }
  
  btn.disabled = true;
  btn.innerHTML = '<span>Creating account...</span>';
  errorEl.classList.add('hidden');
  successEl.classList.add('hidden');
  
  try {
    const response = await fetchWithTimeout(`${BACKEND_API}/api/auth/register`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ email, password })
    }, TIMEOUTS.NORMAL);
    
    const result = await response.json();
    
    if (result.success) {
      // 注册成功，提示用户验证邮箱
      successEl.textContent = 'Account created! Please check your email to verify your account, then sign in.';
      successEl.classList.remove('hidden');
      
      // 清空表单
      document.getElementById('register-email').value = '';
      document.getElementById('register-password').value = '';
      document.getElementById('register-confirm').value = '';
      
      // 3秒后切换到登录界面
      setTimeout(() => {
        showSection('login');
        // 预填邮箱
        document.getElementById('login-email').value = email;
      }, 3000);
    } else {
      errorEl.textContent = getErrorMessage(result.error?.code) || result.error?.message || 'Registration failed';
      errorEl.classList.remove('hidden');
    }
  } catch (e) {
    errorEl.textContent = 'Network error. Please try again.';
    errorEl.classList.remove('hidden');
  } finally {
    btn.disabled = false;
    btn.innerHTML = '<span>Create Account</span>';
  }
}

/**
 * 创建成功消息元素
 */
function createSuccessElement(section) {
  const form = document.getElementById(`${section}-form`);
  let successEl = document.getElementById(`${section}-success`);
  if (!successEl) {
    successEl = document.createElement('div');
    successEl.id = `${section}-success`;
    successEl.className = 'form-success hidden';
    form.insertBefore(successEl, form.firstChild);
  }
  return successEl;
}

/**
 * 获取用户友好的错误消息
 */
function getErrorMessage(code) {
  const messages = {
    'EMAIL_EXISTS': 'This email is already registered. Try signing in instead.',
    'INVALID_REQUEST': 'Please check your input and try again.',
    'INVALID_CREDENTIALS': 'Invalid email or password.',
    'RATE_LIMITED': 'Too many attempts. Please wait a few minutes.',
    'SERVER_ERROR': 'Something went wrong. Please try again later.',
    'EMAIL_NOT_VERIFIED': 'Please verify your email before signing in.'
  };
  return messages[code] || null;
}

async function handleForgotPassword(e) {
  e.preventDefault();
  
  const email = document.getElementById('forgot-email').value;
  const errorEl = document.getElementById('forgot-error');
  const successEl = document.getElementById('forgot-success');
  const btn = document.getElementById('btn-forgot');
  
  btn.disabled = true;
  btn.innerHTML = '<span>Sending...</span>';
  errorEl.classList.add('hidden');
  successEl.classList.add('hidden');
  
  try {
    const response = await fetchWithTimeout(`${BACKEND_API}/api/auth/forgot-password`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ email })
    }, TIMEOUTS.NORMAL);
    
    const result = await response.json();
    
    if (result.success) {
      successEl.textContent = 'Reset link sent! Check your email.';
      successEl.classList.remove('hidden');
    } else {
      errorEl.textContent = result.error?.message || 'Failed to send reset link';
      errorEl.classList.remove('hidden');
    }
  } catch (e) {
    errorEl.textContent = 'Network error. Please try again.';
    errorEl.classList.remove('hidden');
  } finally {
    btn.disabled = false;
    btn.innerHTML = '<span>Send Reset Link</span>';
  }
}

async function handleLogout() {
  await browserAPI.storage.local.remove(['authToken', 'authExpires', 'userEmail']);
  await browserAPI.storage.local.set({ userPlan: 'free' });
  
  // 清除权限缓存
  try {
    await browserAPI.storage.local.remove([
      'cachedPermissions',
      'permissionsCacheTime',
      'currentReservation',
      'currentOCRTransaction'
    ]);
    await browserAPI.runtime.sendMessage({ type: 'CLEAR_PERMISSIONS_CACHE' });
    log.info('Cleared permissions cache on logout');
  } catch (e) {
    log.warn('Failed to clear permissions cache:', e);
  }
  
  currentUser = null;
  showLoggedOutView();
}

function handleUpgrade() {
  browserAPI.tabs.create({ url: 'https://adsinsightpro.com/pricing' });
}

function handleManageSubscription(e) {
  e.preventDefault();
  browserAPI.tabs.create({ url: 'https://adsinsightpro.com/account/subscription' });
}
