﻿import { browserAPI } from '../lib/browser-polyfill.js';

/**
 * Ads Insight Pro - Settings Page
 */

import { getApiUrl } from '../lib/config.js';
import { fetchWithTimeout, TIMEOUTS } from '../lib/fetch-with-timeout.js';
import { TIME } from '../lib/constants.js';
import { createLogger, DEBUG_CONFIG } from '../lib/debug-config.js';

const log = createLogger('[Settings]', DEBUG_CONFIG.popup);

const BACKEND_API = getApiUrl();

// DOM Elements
const elements = {
  defaultPages: document.getElementById('default-pages'),
  defaultFormat: document.getElementById('default-format'),
  defaultRegion: document.getElementById('default-region'),
  historyList: document.getElementById('history-list'),
  version: document.getElementById('version')
};

document.addEventListener('DOMContentLoaded', init);

async function init() {
  await loadSettings();
  await loadHistory();
  await loadUsageStats();
  loadVersion();
  setupEventListeners();
}

function setupEventListeners() {
  document.getElementById('btn-back').addEventListener('click', () => {
    window.location.href = 'popup.html';
  });
  
  document.getElementById('btn-clear-history').addEventListener('click', clearHistory);
  document.getElementById('btn-save').addEventListener('click', saveSettings);
}

async function loadSettings() {
  try {
    const data = await browserAPI.storage.local.get('settings');
    const settings = data.settings || {};
    
    elements.defaultPages.value = settings.defaultPages || 5;
    elements.defaultFormat.value = settings.defaultFormat || 'csv';
    elements.defaultRegion.value = settings.defaultRegion || '';
  } catch (e) {
    log.error('Failed to load settings:', e);
  }
}

async function saveSettings() {
  try {
    const settings = {
      defaultPages: parseInt(elements.defaultPages.value) || 5,
      defaultFormat: elements.defaultFormat.value,
      defaultRegion: elements.defaultRegion.value
    };
    
    await browserAPI.storage.local.set({ settings });
    
    const btn = document.getElementById('btn-save');
    const originalText = btn.textContent;
    btn.textContent = 'Saved!';
    btn.style.background = '#34a853';
    
    setTimeout(() => {
      btn.textContent = originalText;
      btn.style.background = '';
    }, 1500);
  } catch (e) {
    log.error('Failed to save settings:', e);
    alert('Failed to save settings');
  }
}

async function loadHistory() {
  try {
    const data = await browserAPI.storage.local.get('downloadHistory');
    const history = data.downloadHistory || [];
    
    if (history.length === 0) {
      elements.historyList.innerHTML = '<p class="empty-state">No downloads yet</p>';
      return;
    }
    
    elements.historyList.innerHTML = history.slice(0, 10).map(item => `
      <div class="history-item">
        <span class="history-domain">${escapeHtml(item.domain || 'Unknown')}</span>
        <span class="history-meta">${escapeHtml(String(item.adsCount || 0))} ads · ${escapeHtml(formatDate(item.timestamp))}</span>
      </div>
    `).join('');
  } catch (e) {
    log.error('Failed to load history:', e);
  }
}

function escapeHtml(str) {
  if (!str) return '';
  const div = document.createElement('div');
  div.textContent = str;
  return div.innerHTML;
}

async function clearHistory() {
  if (!confirm('Clear all download history?')) return;
  
  try {
    await browserAPI.storage.local.set({ downloadHistory: [] });
    elements.historyList.innerHTML = '<p class="empty-state">No downloads yet</p>';
  } catch (e) {
    log.error('Failed to clear history:', e);
  }
}

function loadVersion() {
  const manifest = browserAPI.runtime.getManifest();
  elements.version.textContent = manifest.version;
}

function formatDate(timestamp) {
  if (!timestamp) return '';
  const date = new Date(timestamp);
  return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
}

async function loadUsageStats() {
  try {
    const response = await browserAPI.runtime.sendMessage({ type: 'GET_USAGE_STATS' });
    
    if (!response || !response.success) {
      log.error('Failed to load usage stats:', response?.error);
      showStatsError();
      return;
    }
    
    const { stats } = response;
    updateStatCard('hourly', stats.hourly);
    updateStatCard('daily', stats.daily);
    log.info('Usage stats loaded:', stats);
    
    await loadOCRUsage();
  } catch (e) {
    log.error('Failed to load usage stats:', e);
    showStatsError();
  }
}

function updateStatCard(type, data) {
  const adsElement = document.getElementById(`${type}-ads`);
  const progressElement = document.getElementById(`${type}-progress`);
  const statusElement = document.getElementById(`${type}-status`);
  
  if (!adsElement || !progressElement || !statusElement) {
    return;
  }
  
  adsElement.textContent = data.ads.toLocaleString();
  progressElement.style.width = `${data.progress}%`;
  progressElement.className = `progress-bar ${data.level}`;
  
  const statusText = {
    safe: 'Safe',
    warning: 'Warning',
    danger: 'Danger'
  };
  
  statusElement.textContent = statusText[data.level] || 'Unknown';
  statusElement.className = `stat-status ${data.level}`;
  
  if (type === 'hourly' && data.resetIn) {
    const minutes = Math.ceil(data.resetIn / 60000);
    statusElement.textContent += ` Resets in ${minutes}m`;
  }
}

function showStatsError() {
  const types = ['hourly', 'daily'];
  types.forEach(type => {
    const adsElement = document.getElementById(`${type}-ads`);
    const statusElement = document.getElementById(`${type}-status`);
    
    if (adsElement) adsElement.textContent = '-';
    if (statusElement) {
      statusElement.textContent = 'Error loading';
      statusElement.className = 'stat-status';
    }
  });
}

async function loadOCRUsage() {
  try {
    const data = await browserAPI.storage.local.get(['userPlan', 'authToken']);
    const plan = data.userPlan || 'free';
    
    if (plan !== 'pro' && plan !== 'enterprise') {
      return;
    }
    
    if (!data.authToken) {
      return;
    }
    
    const response = await fetchWithTimeout(`${getApiUrl()}/api/quota/permissions`, {
      headers: { 'Authorization': `Bearer ${data.authToken}` }
    }, TIMEOUTS.NORMAL);
    
    if (!response.ok) {
      log.warn('Failed to load OCR usage');
      return;
    }
    
    const result = await response.json();
    if (result.success && result.data.ocr_quota) {
      const { daily_used, daily_limit } = result.data.ocr_quota;
      
      const ocrCard = document.getElementById('ocr-usage-card');
      const ocrUsed = document.getElementById('ocr-used');
      const ocrLimit = document.getElementById('ocr-limit');
      const ocrProgress = document.getElementById('ocr-progress');
      
      if (ocrCard && ocrUsed && ocrLimit && ocrProgress) {
        ocrUsed.textContent = (daily_used || 0).toLocaleString();
        ocrLimit.textContent = daily_limit ? daily_limit.toLocaleString() : 'Unlimited';
        
        const percent = daily_limit > 0 ? Math.min(100, (daily_used / daily_limit) * 100) : 0;
        ocrProgress.style.width = `${percent}%`;
        
        ocrCard.style.display = 'block';
        
        log.info('OCR usage loaded:', { daily_used, daily_limit });
      }
    }
  } catch (e) {
    log.error('Failed to load OCR usage:', e);
  }
}