﻿import { browserAPI } from '../lib/browser-polyfill.js';
/**
 * Usage Warning UI
 * 
 * 在 Popup 中显示使用量警告横幅
 */

import { createLogger, DEBUG_CONFIG } from '../lib/debug-config.js';

const log = createLogger('[UsageWarningUI]', DEBUG_CONFIG.popup);

/**
 * 显示警告横幅
 * @param {Object} warning - 警告对象
 */
export function showWarningBanner(warning) {
  // 移除已存在的相同类型警告
  const existing = document.querySelector(`.warning-banner[data-type="${warning.type}"]`);
  if (existing) {
    existing.remove();
  }
  
  // 创建警告横幅
  const banner = document.createElement('div');
  banner.className = `warning-banner ${warning.level}`;
  banner.setAttribute('data-type', warning.type);
  
  // 警告图标
  const icon = warning.level === 'danger' ? '🔴' : '⚠️';
  
  // 构建 HTML
  banner.innerHTML = `
    <div class="warning-header">
      <span class="warning-icon">${icon}</span>
      <span class="warning-title">${escapeHtml(warning.title)}</span>
      <button class="warning-close" aria-label="Close">×</button>
    </div>
    <p class="warning-message">${escapeHtml(warning.message)}</p>
    ${warning.suggestions && warning.suggestions.length > 0 ? `
      <ul class="warning-suggestions">
        ${warning.suggestions.map(s => `<li>${escapeHtml(s)}</li>`).join('')}
      </ul>
    ` : ''}
  `;
  
  // 插入到页面顶部（header 之后）
  const container = document.querySelector('.container');
  const header = container.querySelector('.header');
  if (header && header.nextSibling) {
    container.insertBefore(banner, header.nextSibling);
  } else {
    container.insertBefore(banner, container.firstChild);
  }
  
  // 关闭按钮事件
  const closeBtn = banner.querySelector('.warning-close');
  closeBtn.addEventListener('click', () => {
    banner.remove();
  });
  
  // 自动关闭（警告级别10秒，危险级别不自动关闭）
  if (warning.level === 'warning') {
    setTimeout(() => {
      if (banner.parentNode) {
        banner.remove();
      }
    }, 10000);
  }
  
  log.info('Warning banner shown:', warning.title);
}

/**
 * 移除所有警告横幅
 */
export function clearAllWarnings() {
  const banners = document.querySelectorAll('.warning-banner');
  banners.forEach(banner => banner.remove());
}

/**
 * 移除特定类型的警告
 * @param {string} type - 'hourly' or 'daily'
 */
export function clearWarning(type) {
  const banner = document.querySelector(`.warning-banner[data-type="${type}"]`);
  if (banner) {
    banner.remove();
  }
}

/**
 * 转义 HTML 防止 XSS
 */
function escapeHtml(str) {
  if (!str) return '';
  const div = document.createElement('div');
  div.textContent = str;
  return div.innerHTML;
}

/**
 * 监听来自 Service Worker 的警告消息
 */
export function initWarningListener() {
  browserAPI.runtime.onMessage.addListener((message) => {
    if (message.type === 'USAGE_WARNING') {
      showWarningBanner(message.warning);
    }
  });
  
  log.info('Warning listener initialized');
}
